define("jira/components/libs/marionette-2.1.0", ['require'], function (require) {
    "use strict";
    var marionetteFactory = require('atlassian/libs/factories/marionette-2.1.0');
    var Backbone = require('jira/components/libs/backbone');
    var _ = require('jira/components/libs/underscore');
    var Marionette = marionetteFactory(_, Backbone);

    /**
     * Triggers a preventable event.
     *
     * The event will include an EventObject, and the handler can call eventObject.preventDefault() to prevent the event.
     *
     * @param {string} eventName Name of the event being triggered.
     * @param {EventObject} [eventObject] EventObject used as template to construct the actual EventObject used in the event.
     *
     * @returns {EventObject} EventObject passed to the event.
     */
    function triggerPreventable(eventName, eventObject) {
        /**
         * EventObject passed to preventable events
         * @typedef {Object} EventObject
         * @property {Object} emitter Original emitter of the event.
         * @property {boolean} isPrevented Whether the event has been prevented by the event handler.
         * @property {Function} preventDefault Syntax sugar for set the `isPrevented` value.
         */
        var event = _.defaults({}, eventObject || {}, {
            isPrevented: false,
            emitter: this,
            preventDefault: function () {
                this.isPrevented = true;
            }
        });

        this.trigger(eventName, event);
        return event;
    }

    function retriggerPreventable(eventName, eventObject) {
        var groupEvent = this.triggerPreventable(eventName, eventObject);
        if (groupEvent.isPrevented) {
            eventObject.preventDefault();
        }
    }

    _.extend(Marionette.View.prototype, {
        /**
         * This method unwraps the Backbone.View.
         *
         * By default, Backbone will create a <div> and render the template inside. By calling this
         * method, you can get rid of that <div>, so the main element in your template will be the
         * root element in your template.
         *
         * @param {string} [className] If specified, this method only unwraps the template if the `class` attribute of the
         *                             template's root element matches `className`.
         */
        unwrapTemplate: function unwrapTemplate(className) {
            if (className && !this.$el.hasClass(className)) return;

            if (this.$el.parent().length) {
                // If the template is already rendered in the page
                var children = this.$el.children();
                this.$el.replaceWith(children);
                this.setElement(children);
            } else {
                // If the template is in memory
                this.setElement(this.$el.children());
            }
        },
        triggerPreventable: triggerPreventable,
        retriggerPreventable: retriggerPreventable
    });

    _.extend(Marionette.Controller.prototype, {
        triggerPreventable: triggerPreventable,
        retriggerPreventable: retriggerPreventable
    });

    Marionette.ViewManager = Marionette.Object.extend({
        constructor: function() {
            //Make sure we initialize the 'views' map first, so descentants
            //of this class can build views in their constructor
            this.views = {};
            Marionette.Controller.apply(this, arguments);
        },

        hideView: function(viewName) {
            var view = this.views[viewName];
            if (view) {
                this.stopListening(view);
                if (!view.isDestroyed) {
                    view.destroy();
                }
                delete this.views[viewName];
            }
        },

        showView: function(viewName, factory) {
            var view = this.buildView(viewName, factory);
            view.render();
        },

        buildView: function(viewName, factory) {
            var view = this.views[viewName];
            if (!view) {
                view = factory.call(this);
                this.listenTo(view, "destroy", function() {
                    this.hideView(viewName);
                });
                this.views[viewName] = view;
            }
            return view;
        },

        getView: function(viewName) {
            return this.views[viewName];
        },

        onDestroy: function() {
            _.each(this.views, function(view, viewName) {
                this.hideView(viewName);
            }, this);
        }
    });


    return Marionette;
});

/**
 * Marionette 2.1.0
 *
 * @deprecated Should use {@link jira/components/libs/marionette-2.1.0}.
 * @exports jira/components/libs/marionette-2.1.0
 */
define("jira/components/lib/marionette-2.1.0", ['require'], function (require) {
    "use strict";
    return require('jira/components/libs/marionette-2.1.0');
});
