define("jira/components/issueviewer/views/issue", ["require"], function(require){
    "use strict";

    var MarionetteLayout = require('jira/components/libs/marionette-1.4.1/layout');
    var Templates = require("jira/components/issueviewer/templates");
    var ReplaceRegion = require('jira/components/libs/marionette-1.4.1/replaceregion');
    var Actions = require("jira/components/issueviewer/actions");
    var jQuery = require("jquery");

    /**
     * @class JIRA.Components.IssueViewer.Views.Issue
     *
     * Main view for rendering an issue
     *
     * @extends JIRA.Marionette.Layout
     */
    return MarionetteLayout.extend({
        template: Templates.layout,

        className: "issue-container",

        regions: {
            header: {
                selector: ".issue-header-container",
                regionType: ReplaceRegion
            },
            body: {
                selector: ".issue-body-container",
                regionType: ReplaceRegion
            }
        },

        modelEvents: {
            "updated": "update"
        },

        /**
         * Show the loading indication
         */
        showLoading: function() {
            this.$el.addClass("loading");
        },

        /**
         * Hide the loading indication
         */
        hideLoading: function() {
            this.$el.removeClass("loading");
        },

        /**
         * Focus a comment
         *
         * @param {string} commentId Comment to focus
         */
        focusComment: function(commentId) {
            this.$("#activitymodule .focused").removeClass("focused");
            this.$("#comment-" + commentId).addClass("focused");
            this.$(".js-stalker").trigger("refresh");
        },

        /**
         * Update this view with the data form the model
         *
         * @param props
         */
        update: function(props) {
            this._closeDropdowns();

            // Only render this view if this is the initial view.
            if (props.initialize) {
                this.render();
            }

            // If a comment has been edited, focus it
            if (props.reason && props.reason.action === Actions.EDIT_COMMENT) {
                this.focusComment(props.reason.meta.commentId);
            }

            this._bringViewIssueElementIntoView();
        },

        /**
         * Remove the view
         *
         * Override Backbone's method, as we don't want to remove the container from the view.
         *
         * @returns {*}
         */
        remove: function() {
            this.stopListening();
            this.$el.empty();
            return this;
        },

        /**
         * Handler for render events, things to do after the template has been rendered
         */
        onRender: function() {
            // $el could have been modified outside this view, we need to restore the className
            this.$el.addClass(this.className);
            this.$el.attr("tabindex", "-1");
            this._bringViewIssueElementIntoView();
        },

        onApplyToDom: function() {
            this._bringViewIssueElementIntoView();
            this.trigger("render");
        },

        /**
         * Bring some parts of the issue into view (eg: scrolls to focused comment)
         *
         * @private
         */
        _bringViewIssueElementIntoView: function() {
            var viewIssueQuery = this.model.get("entity").viewIssueQuery;
            if (viewIssueQuery) {
                var elementSelector;
                if (viewIssueQuery.focusedCommentId) {
                    elementSelector = "#comment-" + viewIssueQuery.focusedCommentId;
                } else if (viewIssueQuery.attachmentSortBy || viewIssueQuery.attachmentOrder || viewIssueQuery.attachmentViewMode) {
                    elementSelector = "#attachmentmodule";
                }

                if (elementSelector) {
                    jQuery(elementSelector).scrollIntoView({
                        marginBottom: 200,
                        marginTop: 200
                    });
                }
            }
        },

        _closeDropdowns: function () {
            this.$el.find(".aui-dropdown2-trigger.active").trigger("aui-button-invoke");
        }
    });

});
