define("jira/components/issueviewer/views/issueheader", ["require"], function(require){
    "use strict";

    var MarionetteLayout = require('jira/components/libs/marionette-1.4.1/layout');
    var Templates = require("jira/components/issueviewer/templates");
    var DarkFeatures = require("jira/components/issueviewer/services/darkfeatures");
    var _ = require("jira/components/libs/underscore");
    var IssueOpsbar = require("jira/components/issueviewer/views/issueopsbar");
    var Meta = require("jira/util/data/meta");
    var jQuery = require('jquery');

    /**
     * @class JIRA.Components.IssueViewer.Views.IssueHeader
     *
     * View for rendering the header of an issue. It renders key, summary, breadcrumbs... plus the opsbar and pager as
     * regions
     *
     * @extends JIRA.Marionette.Layout
     */
    return MarionetteLayout.extend({
        className: "issue-header js-stalker",

        id: "stalker",

        tagName: "header",

        template: Templates.Header.issueHeader,

        modelEvents: {
            "updated": "update"
        },

        regions: {
            opsbar: ".command-bar",
            pager: "#issue-header-pager"
        },

        /**
         * Extract the data from the model in the format needed by the template
         *
         * @returns {Object} Data to be rendered by the template
         */
        serializeData: function () {
            return {
                issue: this.model.getEntity(),
                hasProjectShortcut: DarkFeatures.PROJECT_SHORTCUTS.enabled()
            };
        },

        /**
         * Update this view with new data
         *
         * @param options
         */
        update: function (options) {
            if (options.initialize) {
                this._updateWindowTitle();
            } else {
                var editingSummary = _.include(options.fieldsInProgress, "summary");
                if (editingSummary) {
                    this.renderOpsBar();
                } else {
                    this.render();
                }
            }
            this.trigger("updated");
        },

        /**
         * Handler for applyToDom event, things to do after $el has been loaded from the DOM
         */
        onApplyToDom: function () {
            var view = new IssueOpsbar({model: this.model});
            // Since ops bar already is in the dom, we should use the current dom data as the view's element
            view.setElement(this.opsbar.el);
            this.opsbar.attachView(view);
            this.opsbar.currentView.applyToDom();
            this._updateWindowTitle();
        },

        onBeforeRender: function () {
            this.previousElement = this.$el;
        },

        /**
         * Handler for render event, things to do after the template has been rendered
         */
        onRender: function () {
            this.renderOpsBar();
            this._updateWindowTitle();
            this.trigger("panelRendered", "header", this.$el, this.previousElement || jQuery(""));
        },

        /**
         * Render the operations bar
         *
         * //TODO This composition should be done by the IssueController
         */
        renderOpsBar: function () {
            this.opsbar.show(new IssueOpsbar({model: this.model}));
            this.trigger("individualPanelRendered", this.$el);
        },

        /**
         * Updates the window title to contain information about the issue
         *
         * @private
         */
        _updateWindowTitle: function () {
            var entity = this.model.getEntity();
            var key = entity.key;
            var summary = entity.summary;
            var appTitle = Meta.get("app-title");

            if (appTitle && summary && key) {
                document.title = "[" + key + "] " + summary + " - " + appTitle;
            }
        },

        focus: function () {
            this.$el.find("a").eq(0).focus();
        }
    });
});
