AJS.test.require(["com.atlassian.jira.jira-issue-nav-components:issueviewer", "com.atlassian.jira.jira-issue-nav-components:issueviewer-test"], function() {
    "use strict";

    require([
        "jira/components/issueviewer/controllers/issue",
        "jira/components/issueviewer/entities/issue",
        "jquery"
    ], function(
        IssueController,
        IssueModel,
        jQuery
    ) {
        module('jira/components/issueviewer/controllers/issue', {
            setup: function() {
                this.sandbox = sinon.sandbox.create();

                this.model = new IssueModel();

                this.controller = new IssueController({
                    model: this.model
                });
                this.model.set({
                    id: 123,
                    key: 'TEST-1',
                    projectId: 456,
                    projectKey: 'TEST',
                    projectType: 'software'
                });
            },

            teardown: function() {
                this.sandbox.restore();
            },

            assertRenderEvent: function(spy, issue, loadedFromDom) {
                ok(spy.calledOnce, "It fires a 'render' event");

                ok(typeof spy.firstCall.args[0] === "object", "The 'render' event includes a list of regions as the first argument");
                ok(spy.firstCall.args[0].pager, "The 'render' event includes a reference to the 'pager' element");

                ok(typeof spy.firstCall.args[1] === "object", "The 'render' event includes custom options as second argument");
                ok(spy.firstCall.args[1].issueId === issue.id, "The options object includes issueId");
                ok(spy.firstCall.args[1].issueKey === issue.key, "The options object includes issueKey");
                ok(spy.firstCall.args[1].projectId === issue.projectId, "The options object includes projectId");
                ok(spy.firstCall.args[1].projectKey === issue.projectKey, "The options object includes projectKey");
                ok(spy.firstCall.args[1].projectType === issue.projectType, "The options object includes projectType");
                ok(spy.firstCall.args[1].loadedFromDom === loadedFromDom, "The options object includes loadedFromDom");
            }
        });

        test("When showing an issue for the first time, it throws a render event", function() {
            var spy = this.sandbox.spy();
            this.controller.on("render", spy);

            this.controller.show();

            this.assertRenderEvent(spy, {
                id: 123,
                key: 'TEST-1',
                projectId: 456,
                projectKey: 'TEST',
                projectType: 'software'
            }, false);
        });

        test("When showing an issue for the second time, it also throws a render event", function() {
            this.controller.show();
            var spy = this.sandbox.spy();
            this.controller.on("render", spy);

            this.controller.show();

            this.assertRenderEvent(spy, {
                id: 123,
                key: 'TEST-1',
                projectId: 456,
                projectKey: 'TEST',
                projectType: 'software'
            }, false);
        });

        test("When loading an issue from the DOM, it throws a render event", function() {
            var spy = this.sandbox.spy();
            this.controller.on("render", spy);

            this.controller.applyToDom({
                id: 123,
                key: 'TEST-1',
                project: {
                    id: 456,
                    key: "TEST",
                    projectType: 'software'
                }
            });

            this.assertRenderEvent(spy, {
                id: 123,
                key: 'TEST-1',
                projectId: 456,
                projectKey: 'TEST',
                projectType: 'software'
            }, true);
        });

        test("When the header is updated, it throws a render event", function() {
            this.controller.createView();

            var renderSpy = this.sandbox.spy();
            this.controller.on("render", renderSpy);

            this.model.trigger("updated", {});

            sinon.assert.calledOnce(renderSpy);
        });

        test("When an internal PanelsView triggers the event 'itemview:individualPanelRendered', the controller triggers the 'individualPanelRendered' event", function() {
            this.controller.createView();
            var payload = jQuery("<div/>");
            var view = {};
            var spy = this.spy();
            this.controller.on("individualPanelRendered", spy);

            this.controller.leftPanelsView.trigger("itemview:individualPanelRendered", view, payload);

            sinon.assert.calledOnce(spy);
            sinon.assert.calledWith(spy, payload);
        });

        test("When an internal PanelsView triggers the event 'individualPanelRendered', the controller triggers the 'individualPanelRendered' event", function() {
            this.controller.createView();
            var payload = jQuery("<div/>");
            var spy = this.spy();
            this.controller.on("individualPanelRendered", spy);

            this.controller.headerView.trigger("individualPanelRendered", payload);

            sinon.assert.calledOnce(spy);
            sinon.assert.calledWith(spy, payload);
        });

        test("When an internal PanelsView triggers the event 'itemview:panelRendered', the controller triggers the 'panelRendered' event", function() {
            this.controller.createView();
            var $new = jQuery("<div/>");
            var $existing = jQuery("<div/>");
            var panelId = "comments";
            var view = {};
            var spy = this.spy();
            this.controller.on("panelRendered", spy);

            this.controller.leftPanelsView.trigger("itemview:panelRendered", view, panelId, $new, $existing);

            sinon.assert.calledOnce(spy);
            sinon.assert.calledWith(spy, panelId, $new, $existing);
        });

        test("When the main view triggers the event 'render', the controller triggers the event 'renderMainView'", function() {
            this.controller.createView();
            var $el = this.controller.view.$el;
            var spy = this.spy();
            this.controller.on("renderMainView", spy);

            this.controller.view.trigger("render");

            sinon.assert.calledOnce(spy);
            sinon.assert.calledWith(spy, $el);
        });

        test("When focusing the issue, it delegates in the header view", function() {
            this.controller.createView();
            this.spy(this.controller.headerView, "focus");

            this.controller.focus();

            sinon.assert.calledOnce(this.controller.headerView.focus);
        });
    });
});
