define("jira/components/issueviewer", ["require"], function(require){
    "use strict";

    var _ = require("jira/components/libs/underscore");
    var Actions = require("jira/components/issueviewer/actions");
    var DarkFeatures = require("jira/components/issueviewer/services/darkfeatures");
    var ErrorController = require("jira/components/issueviewer/controllers/error");
    var Events = require('jira/util/events');
    var IssueController = require("jira/components/issueviewer/controllers/issue");
    var IssueEventBus = require("jira/components/issueviewer/legacy/issueeventbus");
    var IssueLoader = require("jira/components/issueviewer/services/issueloader");
    var IssueModel = require("jira/components/issueviewer/entities/issue");
    var jQuery = require("jquery");
    var LinksCapturer = require("jira/components/issueviewer/linkscapturer");
    var MarionetteController = require('jira/components/libs/marionette-1.4.1/controller');
    var Meta = require("jira/util/data/meta");
    var MetadataService = require("jira/components/issueviewer/services/metadataservice");
    var Reasons = require('jira/util/events/reasons');
    var Trace = window.JIRA;
    var Types = require('jira/components/issueviewer/eventtypes');
    var Utils = require("jira/components/issueviewer/utils");
    var ViewIssueData = require("jira/components/issueviewer/legacy/viewissuedata");
    var WRMData = require("wrm/data");

    var contextPath = window.AJS.contextPath();

    var trace = function() {
        Trace.trace.apply(Trace, arguments);
    };

    /**
     * @class JIRA.Components.IssueViewer
     *
     * This module provides the IssueViewer. It will load an issue, update it and render the UI to view the issue
     *
     * @extends JIRA.Marionette.Controller
     */
    return MarionetteController.extend({
        namedEvents: [
        /**
         * @event loadComplete
         * Triggered when an issue has loaded successfully.
         *
         * @param {JIRA.Components.IssueViewer.Models.Issue} model Model with the issue we have loaded
         * @param {Object} options
         * @param {boolean} options.isNewIssue Wheter the loaded issue is a new issue
         */
            "loadComplete",

        /**
         * @event loadError
         * Triggered when there is an error when loading an issue.
         *
         * @param {Object} options
         */
            "loadError",

        /**
         * @event close
         * We should close to issue view in response to some action.
         */
            "close",

        /**
         * @event replacedFocusedPanel
         * Triggered when the view has rendered a panel that has the focus
         * //TODO This seems to be too specific, why others needs to know about this?
         */
            "replacedFocusedPanel",

        /**
         * @event linkToIssue
         * When a user clicks on a link to an issue
         */
            "linkToIssue"
        ],

        /**
         * @constructor
         * Initialize this module and all the services/controllers
         *
         * @param {Object} options
         * @param {boolean|function} [options.showReturnToSearchOnError=false] Whether the error views should display a 'Return to Search' link
         * @param {boolean} [options.useJIRAEvents=true] Whether the component will trigger JIRA Events.
         */
        initialize: function(options) {
            options = options || {};

            options.useJIRAEvents = options.useJIRAEvents !== false;

            this.model = new IssueModel();
            this.eventBus = new IssueEventBus();
            this.viewIssueData = new ViewIssueData();

            // Services
            this._buildIssueLoader();
            this._buildLinksCapturer();

            // Controllers
            this._buildErrorController({
                showReturnToSearchOnError: options.showReturnToSearchOnError
            });
            this._buildIssueController();

            Events.bind(Types.REFRESH_ISSUE_PAGE, _.bind(function(e, issueId, options) {
                if (this.model.isCurrentIssue(issueId)) {
                    this.refreshIssue(options);
                }
            }, this));
        },

        /**
         * Builds the issueLoader service and listens for its events
         *
         * @private
         */
        _buildIssueLoader: function() {
            this.issueLoader = new IssueLoader();

            this.listenTo(this.issueLoader, "error", function(reason, props) {
                this.trigger("loadError", props);
                this.errorController.render(reason, props.issueKey);
                this.removeIssueMetadata();

                // Traces
                trace("jira.issue.refreshed", {id: props.issueId});

                if (this.options.useJIRAEvents) {
                    Events.trigger(Types.ISSUE_REFRESHED, [props.issueId]);
                }
            });

            this.listenTo(this.issueLoader, "issueLoaded", this._onIssueLoaded);
        },

        _buildLinksCapturer: function() {
            this.linksCapturer = new LinksCapturer();
            this.listenTo(this.linksCapturer, {
                "linkToIssue": function(options){
                    this.trigger("linkToIssue", options);
                },
                "refineViewer": function(event) {
                    this.trigger("refineViewer", event);
                },
                "reorderSubtasks": function() {
                    this.issueController.showLoading();
                    this.refreshIssue({
                        reason: Actions.UPDATE
                    }).always(_.bind(function(){
                        this.issueController.hideLoading();
                    }, this));
                }
            });
        },

        /**
         * Builds the ErrorController
         *
         * @param {Object} options
         * @param {boolean|function} [options.showReturnToSearchOnError=false] Whether the error views should display a 'Return to Search' link
         * @private
         */
        _buildErrorController: function(options) {
            options = options || {};

            this.errorController = new ErrorController({
                contextPath: contextPath,
                showReturnToSearchOnError: options.showReturnToSearchOnError
            });

            this.listenTo(this.errorController, "before:render", function() {
                this.issueController.close();
            });

            this.listenTo(this.errorController, "returnToSearch", function() {
                this.trigger("close");
            });

            this.listenAndRethrow(this.errorController, "render");
        },

        /**
         * Builds the Issue controller, the main controller for viewing issues
         * @private
         */
        _buildIssueController: function() {
            this.issueController = new IssueController({
                model: this.model
            });
            this.listenAndRethrow(this.issueController, "replacedFocusedPanel");
            this.listenTo(this.issueController, {
                "render": function(regions, options) {
                    Utils.hideDropdown();
                    this.errorController.close();
                    this.trigger("render", regions, options);

                    trace("jira.psycho.issue.refreshed", {id: this.model.getId()});
                },
                "renderMainView": function($el) {
                    if (this.options.useJIRAEvents) {
                        Events.trigger(Types.NEW_CONTENT_ADDED, [$el, Reasons.pageLoad]);
                        Events.trigger(Types.REFRESH_TOGGLE_BLOCKS, [this.model.getId()]);
                    }
                    this.trigger("renderMainView", $el, this.model.getId());
                },
                "panelRendered": function(panelId, $ctx, $existing) {
                    this.eventBus.triggerPanelRendered(panelId, $ctx);
                    if (this.options.useJIRAEvents) {
                        Events.trigger(Types.PANEL_REFRESHED, [panelId, $ctx, $existing]);
                    }
                    this.trigger("panelRendered", panelId, $ctx, $existing);
                },
                "close": function() {
                    Utils.hideDropdown();
                },
                "individualPanelRendered": function(renderedPanel) {
                    if (this.options.useJIRAEvents) {
                        Events.trigger(Types.NEW_CONTENT_ADDED, [renderedPanel, Reasons.panelRefreshed]);
                    }
                    this.trigger("individualPanelRendered", renderedPanel);
                }
            });
        },

        /**
         * Update our model with new data
         *
         * @param {Object} data
         * @param {Object} options
         */
        _updateModel: function(data, options) {
            this.model.update(data, options);
        },

        /**
         * Handler for issueLoaded, when an issue has been loaded by IssueLoader service
         *
         * @param {Object} data
         * @param {Object} meta
         * @param {Object} options
         * @private
         */
        _onIssueLoaded: function(data, meta, options) {
            //TODO Why issueEntity is not loaded from data?
            var isPrefetchEnabled = !DarkFeatures.NO_PREFETCH.enabled();
            var issueEntity = options.issueEntity;
            // TODO options.initialize, meta.mergeIntoCurrent and meta.isUpdate seems to represent the same thing
            //      Investigate if all of them are in use and are actually necessary
            var initialize = !meta.mergeIntoCurrent && options.initialize !== false;
            var isNewIssue = !this.model.isCurrentIssue(issueEntity.id);
            var detailView = !!issueEntity.detailView;

            // Clear previous model and errors if this is not an update or is the initial render
            if (!meta.isUpdate || initialize) {
                this.model.resetToDefault();
                this.errorController.close();
            }

            // Update the model with the new data
            this._updateModel(data, {
                initialize: initialize,
                changed: meta.changed,
                mergeIntoCurrent: meta.mergeIntoCurrent
            });

            // Clear previous render if this is not an update or is the initial render
            if (!meta.isUpdate || initialize) {
                this.issueController.close();
            }
            // Display the controller
            this.issueController.show();

            // Refresh the issue if it is loaded from the cache
            if (isPrefetchEnabled && meta.fromCache) {
                this.refreshIssue(issueEntity, {
                    fromCache: true,
                    mergeIntoCurrent: !meta.error, // If we previously showed error then load everything instead of merging.
                    detailView: detailView  // JRA-36659: keep track of whether we are in detail view
                });
            }

            // Save issue metadata
            MetadataService.addIssueMetadata(this.model);

            //TODO This should be moved to issueController. Also, issueEntity has no business with bringToFocus
            if (issueEntity.bringToFocus) {
                issueEntity.bringToFocus();
            }

            this._triggerLoadComplete(isNewIssue, meta, issueEntity);

            // Traces
            var traceData = {id: issueEntity.id};
            if (meta.fromCache) {
                trace('jira.issue.loadFromCache', traceData);
            } else {
                trace('jira.issue.loadFromServer', traceData);
            }
            trace("jira.issue.refreshed", traceData);

            if (this.options.useJIRAEvents) {
                Events.trigger(Types.ISSUE_REFRESHED, [issueEntity.id]);
            }
        },

        _triggerLoadComplete: function (isNewIssue, meta, issueEntity) {
            this.trigger("loadComplete", this.model, {
                isNewIssue: isNewIssue,
                issueId: issueEntity.id,
                issueKey: issueEntity.key,
                duration: meta.loadDuration,
                loadReason: meta.fromCache ? 'issues-cache-refresh' : undefined,
                fromCache: meta.fromCache,
                //inform upper layers that are handling this event that additional ISSUE_REFRESHED event will be generated after execution of loadComplete handlers
                issueRefreshedEvent: !!this.options.useJIRAEvents
            });
        },

        /**
         * Cancels any pending load so that their handlers aren't called
         */
        abortPending: function() {
            this.issueLoader.cancel();
        },

        /**
         * Shows a dirty form warning if the comment field has been modified.
         *
         * @returns {boolean}
         */
        canDismissComment: function() {
            return this.issueController.canDismissComment();
        },

        /**
         * Clean up before hiding an issue (hide UI widgets, remove metadata, etc.).
         */
        beforeHide: function() {
            Utils.hideLightbox();
            Utils.hideDropdown();
            this.abortPending();
            this.removeIssueMetadata();
        },

        /**
         * Prepare for an issue to be shown.
         */
        beforeShow: function() {
            MetadataService.addIssueMetadata(this.model);
        },

        /**
         * @return {null|number} the current issue's ID or null if no valid issue is selected.
         */
        getIssueId: function() {
            return this.model.getEntity().id || null;
        },

        /**
         * @return {null|string} the current issue's key or null if no valid issue is selected.
         */
        getIssueKey: function() {
            return this.model.getEntity().key || null;
        },

        /**
         * @return {null|string} the key of the project this issue belongs to, or null if that info is not available.
         */
        getProjectKey: function() {
            return this.model.get('projectKey') || null;
        },

        /**
         * @return {null|string} the type of the project this issue belongs to, or null if that info is not available.
         */
        getProjectType: function() {
            return this.model.get('projectType') || null;
        },

        /**
         * @return {null|number} the id of the project this issue belongs to, or null if that info is not available.
         */
        getProjectId: function() {
            return this.model.get('projectId') || null;
        },

        /**
         * Loads an issue already rendered by the server.
         *
         * @param {Object} issueEntity
         */
        _loadIssueFromDom: function(issueEntity) {
            // Many places in KickAss use the presence of an issue ID / key to determine if an issue is selected. We
            // can't extract either from an error message, so pass a dud ID to make it look like an issue is selected.
            if (!issueEntity.id || issueEntity.id === -1) {
                this.errorController.applyToDom("notfound", issueEntity.key);
                this.trigger("loadError");
            } else {
                this.issueController.applyToDom(_.extend({id: -1}, issueEntity));
            }

            // After initial load, the server rendered view issue page will be the same as
            // regular ajax view issue page. Thus removing the meta so it can resume
            // to work regularly.
            Meta.set("serverRenderedViewIssue", null);

            var traceData = {id: this.getIssueId()};
            //TODO These traces should be inside IssueController, as it has more knowledge about when the issue is loaded
            trace("jira.issue.refreshed", traceData);
        },

        /**
         * Load an issue and show it in the container.
         *
         * @param {Object} issueEntity
         * @param {number} issueEntity.id The issue's ID.
         * @param {string} issueEntity.key The issue's key.
         * @param {string} [issueEntity.viewIssueQuery] The query string that was provided
         *
         * @returns {jQuery.Promise}
         */
        loadIssue: function(issueEntity) {
            var isServerRendered = Meta.get("serverRenderedViewIssue");
            if (isServerRendered) {
                issueEntity.project = {
                    id: WRMData.claim('projectId'),
                    key: WRMData.claim('projectKey'),
                    projectType: WRMData.claim('projectType')
                };
                return this.loadIssueRenderedByTheServer(issueEntity);
            } else {
                if (!this.canDismissComment() || !issueEntity.key) {
                    return new jQuery.Deferred().reject();
                }

                this.issueController.showLoading();
                return this.issueLoader.load({
                    issueEntity: issueEntity,
                    viewIssueData: this.viewIssueData
                });
            }
        },

        /**
         * This method is called when the issue is render in the server.
         *
         * @param {Object} issueEntity, Object with the information about the issue to be loaded
         * @returns {promise} Because it is call from loadIssue, and loadIssue must return a promise,
         */
        loadIssueRenderedByTheServer: function(issueEntity) {
            var issueKey = jQuery("#key-val");
            var issueSummary = jQuery("#summary-val");
            issueEntity.id = issueKey.attr("rel") || -1;
            issueEntity.summary = issueSummary.text();
            this._loadIssueFromDom(issueEntity);
            this.trigger("loadComplete", this.model, {
                isNewIssue: false, //is false, because no new issues can be loaded from the server
                issueId: issueEntity.id,
                issueKey: issueEntity.key,
                duration: 0, //Because it was loaded in the server and not by the issueLoadService
                loadReason: undefined, //it must be undefined if this is not loaded from cache
                fromCache: false //from server
            });
            return new jQuery.Deferred().resolve().promise();
        },

        /**
         * Refresh the content of the issue, by merging changes from the server.
         *
         * The returned promise is:
         * - resolved when the selected issue is refreshed, or if there is no selected issue
         * - rejected *only* when refreshing the selected issue fails
         *
         * @param {boolean} [options.mergeIntoCurrent] Whether the refresh should merge the retrieved data into the current model
         * @param {function} [options.complete] a function to call after the update has finished
         * @returns {jQuery.Promise}
         */
        refreshIssue: function(options) {
            var promise;
            options = _.defaults({}, options, {
                mergeIntoCurrent: true
            });

            if (this.model.hasIssue()) {
                promise = this.issueLoader.update({
                    viewIssueData: this.viewIssueData,
                    issueEntity: this.model.getEntity(),
                    mergeIntoCurrent: options.mergeIntoCurrent,
                    detailView: options.detailView // JRA-36659: keep track of whether we are in detail view
                });

                if (options.complete) {
                    promise = promise.done(options.complete).fail(options.complete);
                }

                this.issueController.showLoading();
            } else {
                promise = new jQuery.Deferred().resolve().promise();
            }

            return promise;
        },

        /**
         * Remove the issue metadata
         */
        removeIssueMetadata: function() {
            MetadataService.removeIssueMetadata(this.model);
        },

        /**
         * Set the container that the issue should be rendered into.
         *
         * @param {jQuery} container The container the issue should be rendered into.
         */
        setContainer: function(container) {
            this.linksCapturer.capture(container);
            this.errorController.setElement(container);
            this.issueController.setElement(container);
        },

        /**
         * Returns a deferred that is resolved once issue has loaded.
         * Or straight away if you there are no issue loading in progress.
         *
         * @return {boolean}
         */
        isCurrentlyLoading: function() {
            return this.issueLoader.isLoading();
        },

        /**
         * Updates the current issue with a new ViewIssueQuery
         *
         * @param query {Object} New query to use for the request
         */
        updateIssueWithQuery: function(query) {
            this.issueController.showLoading();
            this.model.updateIssueQuery(query);
            this.issueLoader.update({
                viewIssueData: this.viewIssueData,
                issueEntity: this.model.getEntity(),
                mergeIntoCurrent: true
            });
        },

        /**
         * Closes the IssueViewer, cleaning the model and closing all the views
         */
        dismiss: function() {
            this.model.resetToDefault();
            this.errorController.close();
            this.issueController.close();
            this.linksCapturer.destroy();
        },

        close: function() {
            if (this.canDismissComment()) {
                this.dismiss();
                this.trigger("close");
            }
        },

        applyResponsiveDesign: function() {
            this.issueController.applyResponsiveDesign();
        },

        isShowingError: function() {
            return this.errorController.isRendered();
        },

        focus: function() {
            this.issueController.focus();
        }
    });

});
