AJS.test.require(["com.atlassian.jira.jira-issue-nav-components:issueviewer", "com.atlassian.jira.jira-issue-nav-components:issueviewer-test"], function() {
    "use strict";

    require([
        'jquery',
        "jira/message",
        "jira/components/issueviewer",
        "jira/components/issueviewer/services/metadataservice",
        "jira/util/data/meta",
        "jira/util/events",
        "jira/components/issueviewer/eventtypes",
        "wrm/data"
    ], function(
        jQuery,
        Messages,
        IssueViewer,
        MetadataService,
        Meta,
        Events,
        Types,
        WRMData
    ) {
        var CommentForm = JIRA.Issue.CommentForm;

        module('jira/components/issueviewer', {
            setup: function() {
                this.sandbox = sinon.sandbox.create();
                this.sandbox.useFakeServer();
                this.sandbox.stub(Messages, "showErrorMsg");
                this.sandbox.stub(WRMData, "claim");
                this.module = new IssueViewer();

                this.newIssue = true;
                this.meta = {
                    duration: 123,
                    fromCache: true
                };
                this.issueEntity = {
                    id: 321,
                    key: 'HSP-2'
                };
            },

            teardown: function() {
                this.sandbox.restore();
            },

            generateLoadCompleteOptions: function (useJiraEvents) {
                return {
                    duration: undefined,
                    fromCache: undefined,
                    isNewIssue: false,
                    issueId: undefined,
                    issueKey: undefined,
                    issueRefreshedEvent: useJiraEvents,
                    loadReason: undefined
                };
            }
        });

        test("When setting the container, it updates the container of the internal controllers", function() {
            this.module.setContainer(jQuery("#qunit-fixture"));

            ok(this.module.errorController._$el.is(jQuery("#qunit-fixture")), "The ErrorController's element was updated");
            ok(this.module.issueController.$el.is(jQuery("#qunit-fixture")), "The IssueController's element was updated");
        });

        test("When loading a valid issue, it should appropriately set the issue meta data", function() {
            this.sandbox.stub(MetadataService, "addIssueMetadata");

            this.module.issueLoader.trigger("issueLoaded",
                {issue: {id: 12345, key: "JRA-123"}},
                {},
                {issueEntity: {}}
            );

            ok(MetadataService.addIssueMetadata, "Metadata service is called");
        });

        test("When having an error loading an issue, it should appropriately remove the issue meta data", function() {
            this.sandbox.stub(MetadataService, "removeIssueMetadata");

            this.module.issueLoader.trigger("error", "generic", {});

            ok(MetadataService.removeIssueMetadata, "Metadata service is called");
        });

        test("When loading an issue when comment is dirty, it should show a confirmation", function() {
            this.sandbox.stub(window, "confirm");

            var $form = jQuery("<form id='issue-comment-add' />").appendTo("#qunit-fixture");
            var $comment = jQuery("<textarea id='comment' name='comment'></textarea>").appendTo($form);
            $comment.val("I am a dirty dirty field");

            var response = this.module.loadIssue({id: 1, key: "KEY-1"});

            ok(window.confirm.calledOnce, "Expected dirty form warning to appear");
            ok(response.isRejected(), "Expected deferred to be rejected");
        });

        test("When loading an issue when comment is dirty and the user does not allow the dismissal of comment, the comment is focused", function() {
            this.sandbox.stub(window, "confirm").returns(false);
            this.sandbox.spy(CommentForm, "focus");

            var $form = jQuery("<form id='issue-comment-add' />").appendTo("#qunit-fixture");
            var $comment = jQuery("<textarea id='comment' name='comment' />").appendTo($form);
            $comment.val("I am a dirty dirty field");

            var response = this.module.canDismissComment();

            ok(!response, "Expected not to be able to dismiss comment");
            ok(CommentForm.focus.calledOnce, "Expected comment to be focused");
        });

        test("When loading an issue when comment is dirty, the confirm dialog only appears once", function() {
            this.sandbox.stub(window, "confirm").returns(true);

            var $form = jQuery("<form id='issue-comment-add' />").appendTo("#qunit-fixture");
            var $comment = jQuery("<textarea id='comment' name='comment'></textarea>").appendTo($form);
            $comment.val("I am a dirty dirty field");

            this.module.canDismissComment();
            this.module.canDismissComment();
            this.module.canDismissComment();

            ok(window.confirm.calledOnce, "Expected confirm dialog to be called once");
        });

        test("When loading an issue when comment is dirty, the confirm dialog appears if cancelled and then invoked again", function() {
            this.sandbox.stub(window, "confirm").returns(false);

            var $form = jQuery("<form id='issue-comment-add' />").appendTo("#qunit-fixture");
            var $comment = jQuery("<textarea id='comment' name='comment'></textarea>").appendTo($form);
            $comment.val("I am a dirty dirty field");

            this.module.canDismissComment();
            this.module.canDismissComment();

            ok(window.confirm.calledTwice, "Expected confirm dialog to be called twice");
        });

        test("When an issue is updated, the viewIssueQuery is saved", function() {
            sinon.stub(this.module.viewIssueData, "get").returns(new jQuery.Deferred());

            var query = {
                "attachmentOrder": "DESC"
            };
            this.module.model.setId("1");
            this.module.updateIssueWithQuery(query);

            ok(this.module.viewIssueData.get.calledOnce, "ViewIssueData.get is called once");
            deepEqual(this.module.viewIssueData.get.firstCall.args[2].issueEntity.viewIssueQuery, query, "New ViewIssueQuery is passsed to ViewIssueData.get");
        });

        test("When an issue is refreshed, update is called with mergeIntoCurrent = true by default.", function() {
            var issueLoaderSpy = sinon.spy(this.module.issueLoader, "update");
            this.module.model.set("id", "1");
            this.module.refreshIssue();
            ok(issueLoaderSpy.calledOnce, "update() should have been called once.");
            equal(issueLoaderSpy.firstCall.args[0].mergeIntoCurrent, true, "mergeIntoCurrent should equal true.");
        });

        test("When an issue is refreshed, update is called with mergeIntoCurrent = false when passed as options", function() {
            var issueLoaderSpy = sinon.spy(this.module.issueLoader, "update");
            this.module.model.set("id", "1");
            this.module.refreshIssue({
                mergeIntoCurrent: false
            });
            ok(issueLoaderSpy.calledOnce, "update() should have been called once.");
            equal(issueLoaderSpy.firstCall.args[0].mergeIntoCurrent, false, "mergeIntoCurrent should equal false.");
        });

        test("When a new issue is loaded, the old issue is closed before showing the new one", function() {
            var issueViewer = new IssueViewer();
            var issueLoader = issueViewer.issueLoader;
            var issueController = issueViewer.issueController;

            this.sandbox.stub(issueController, "close");
            issueLoader.trigger("issueLoaded",
                {issue: {id: 12345, key: "JRA-123"}},
                {},
                {issueEntity: {}}
            );
            ok(issueController.close.calledOnce, "The issue is closed");
        });

        test("When a new issue is loaded, the old issue is not closed if the new one is an update", function() {
            var issueViewer = new IssueViewer();
            var issueLoader = issueViewer.issueLoader;
            var issueController = issueViewer.issueController;

            this.sandbox.stub(issueController, "close");
            issueLoader.trigger("issueLoaded",
                {issue: {id: 12345, key: "JRA-123"}},
                {isUpdate: true, mergeIntoCurrent: true},
                {issueEntity: {}}
            );
            ok(!issueController.close.called, "The issue is not closed");
        });

        test("When a new issue is loaded, the old issue is not closed if the new one is a new set of data", function() {
            var issueViewer = new IssueViewer();
            var issueLoader = issueViewer.issueLoader;
            var issueController = issueViewer.issueController;

            this.sandbox.stub(issueController, "close");
            issueLoader.trigger("issueLoaded",
                {issue: {id: 12345, key: "JRA-123"}},
                {isUpdate: true, mergeIntoCurrent: true},
                {issueEntity: {}}
            );
            ok(!issueController.close.called, "The issue is not closed");
        });

        test("Triggers a 'loadError' event on loading an error from the DOM", function() {
            var issueViewer = new IssueViewer();
            var spy = sinon.spy();

            issueViewer.on("loadError", spy);
            this.sandbox.stub(Meta, "get").withArgs("serverRenderedViewIssue").returns(true);
            this.sandbox.stub(Meta, "set");

            issueViewer.loadIssue({});
            equal(spy.callCount, 1, "A loadError event was triggered");
        });

        test("Triggers a 'render' event on loading an issue from the DOM", function() {
            var issueViewer = new IssueViewer();
            var spy = sinon.spy();

            issueViewer.on("render", spy);
            this.sandbox.stub(Meta, "get").withArgs("serverRenderedViewIssue").returns(true);
            this.sandbox.stub(Meta, "set");

            issueViewer.loadIssue({});
            equal(spy.callCount, 1, "A render event was triggered");
        });

        test("Load project information from WRM when loaded from the DOM", function() {
            var issueViewer = new IssueViewer();
            this.sandbox.stub(Meta, "get").withArgs("serverRenderedViewIssue").returns(true);

            issueViewer.loadIssue({});

            sinon.assert.calledOnce(WRMData.claim.withArgs("projectId"));
            sinon.assert.calledOnce(WRMData.claim.withArgs("projectKey"));
            sinon.assert.calledOnce(WRMData.claim.withArgs("projectType"));
        });

        test("When having an error loading an issue, it triggers a JIRA.Event", function() {
            var issueViewer = new IssueViewer();
            var spy = sinon.spy();
            Events.bind(Types.ISSUE_REFRESHED, spy);

            issueViewer.issueLoader.trigger("error", "generic", {issueId: 1234});

            sinon.assert.calledOnce(spy);
            sinon.assert.calledWith(spy, sinon.match.any, 1234);
        });

        test("When having an error loading an issue with events disabled, it does not trigger a JIRA.Event", function() {
            var issueViewer = new IssueViewer({
                useJIRAEvents: false
            });
            var spy = sinon.spy();
            Events.bind(Types.ISSUE_REFRESHED, spy);

            issueViewer.issueLoader.trigger("error", "generic", {});

            sinon.assert.notCalled(spy);
        });

        test("When loading a valid issue, it triggers a JIRA.Event", function() {
            var issueViewer = new IssueViewer();
            var spy = sinon.spy();
            Events.bind(Types.ISSUE_REFRESHED, spy);

            issueViewer.issueLoader.trigger("issueLoaded",
                {issue: {id: 12345, key: "JRA-123"}},
                {},
                {issueEntity: {id: 1234}}
            );

            sinon.assert.calledOnce(spy);
            sinon.assert.calledWith(spy, sinon.match.any, 1234);
        });

        test("When the IssueController triggers the 'individualPanelRendered' event, it triggers a JIRA.Event", function() {
            var issueViewer = new IssueViewer();
            var spy = sinon.spy();
            var payload = jQuery("<div/>");
            Events.bind(Types.NEW_CONTENT_ADDED, spy);

            issueViewer.issueController.trigger("individualPanelRendered", payload);

            sinon.assert.calledOnce(spy);
            sinon.assert.calledWith(spy, sinon.match.any, payload);
        });

        test("When the IssueController triggers the 'individualPanelRendered' event with events disabled, it does not trigger a JIRA.Event", function() {
            var issueViewer = new IssueViewer({
                useJIRAEvents: false
            });
            var onJiraEvent = sinon.spy();
            var onEvent = sinon.spy();
            var payload = jQuery("<div/>");
            Events.bind(Types.NEW_CONTENT_ADDED, onJiraEvent);
            issueViewer.on('individualPanelRendered', onEvent);

            issueViewer.issueController.trigger("individualPanelRendered", payload);

            sinon.assert.notCalled(onJiraEvent);
            sinon.assert.calledOnce(onEvent);
            sinon.assert.calledWith(onEvent, payload);
        });

        test("When the IssueController triggers the 'panelRendered' event, it triggers a JIRA.Event", function() {
            var issueViewer = new IssueViewer();
            var spy = sinon.spy();
            var panelId = "test";
            var $newPanel = jQuery("<div/>");
            var $existingPanel = jQuery("<div/>");
            Events.bind(Types.PANEL_REFRESHED, spy);

            issueViewer.issueController.trigger("panelRendered", panelId, $newPanel, $existingPanel);

            sinon.assert.calledOnce(spy);
            sinon.assert.calledWith(spy, sinon.match.any, panelId, $newPanel, $existingPanel);
        });

        test("When the IssueController triggers the 'panelRendered' event with events disabled, it does not trigger a JIRA.Event", function() {
            var issueViewer = new IssueViewer({
                useJIRAEvents: false
            });
            var onJiraEvent = sinon.spy();
            var onEvent = sinon.spy();
            var panelId = "test";
            var $newPanel = jQuery("<div/>");
            var $existingPanel = jQuery("<div/>");
            Events.bind(Types.NEW_CONTENT_ADDED, onJiraEvent);
            issueViewer.on('panelRendered', onEvent);

            issueViewer.issueController.trigger("panelRendered", panelId, $newPanel, $existingPanel);

            sinon.assert.notCalled(onJiraEvent);
            sinon.assert.calledOnce(onEvent);
            sinon.assert.calledWith(onEvent, panelId, $newPanel, $existingPanel);
        });

        test("When the IssueController triggers the 'renderMainView' event, it triggers two JIRA.Events", function() {
            var issueViewer = new IssueViewer();
            issueViewer.model.setId("1");
            var $el = jQuery("<div/>");
            var onNewContentAdded = sinon.spy();
            var onRefreshToggleBlocks = sinon.spy();
            Events.bind(Types.NEW_CONTENT_ADDED, onNewContentAdded);
            Events.bind(Types.REFRESH_TOGGLE_BLOCKS, onRefreshToggleBlocks);

            issueViewer.issueController.trigger("renderMainView", $el);

            sinon.assert.calledOnce(onNewContentAdded);
            sinon.assert.calledWith(onNewContentAdded, sinon.match.any, $el);
            sinon.assert.calledOnce(onRefreshToggleBlocks);
            sinon.assert.calledWith(onRefreshToggleBlocks, sinon.match.any, "1");
        });

        test("When the IssueController triggers the 'renderMainView' event with events disabled, it does not trigger any JIRA.Event", function() {
            var issueViewer = new IssueViewer({
                useJIRAEvents: false
            });
            issueViewer.model.setId("1");
            var $el = jQuery("<div/>");
            var onNewContentAdded = sinon.spy();
            var onRefreshToggleBlocks = sinon.spy();
            var onEvent = sinon.spy();
            Events.bind(Types.NEW_CONTENT_ADDED, onNewContentAdded);
            Events.bind(Types.REFRESH_TOGGLE_BLOCKS, onRefreshToggleBlocks);
            issueViewer.on('renderMainView', onEvent);

            issueViewer.issueController.trigger("renderMainView", $el);

            sinon.assert.notCalled(onNewContentAdded);
            sinon.assert.notCalled(onRefreshToggleBlocks);
            sinon.assert.calledOnce(onEvent);
            sinon.assert.calledWith(onEvent, $el);
        });

        test("When focusing the component, it delegates into an internal IssueController", function() {
            var issueViewer = new IssueViewer({
                useJIRAEvents: false
            });
            this.spy(issueViewer.issueController, "focus");

            issueViewer.focus();

            sinon.assert.calledOnce(issueViewer.issueController.focus);
        });

        test("It gets the project key of the issue", function() {
            this.module.issueLoader.trigger("issueLoaded",
                {issue: {id: 12345, key: "JRA-123", project: {id: 1000, key: "JRA", projectType: "software", avatarUrls: {}}}},
                {},
                {issueEntity: {}}
            );

            equal(this.module.getProjectKey(), "JRA");
        });

        test("It gets the project type of the issue", function() {
            this.module.issueLoader.trigger("issueLoaded",
                {issue: {id: 12345, key: "JRA-123", project: {id: 1000, key: "JRA", projectType: "software", avatarUrls: {}}}},
                {},
                {issueEntity: {}}
            );

            equal(this.module.getProjectType(), "software");
        });

        test("It gets the project type id of the issue", function() {
            this.module.issueLoader.trigger("issueLoaded",
                {issue: {id: 12345, key: "JRA-123", project: {id: 1000, key: "JRA", projectType: "software", avatarUrls: {}}}},
                {},
                {issueEntity: {}}
            );

            equal(this.module.getProjectId(), 1000);
        });

        test("When options.useJIRAEvents===true then triggering loadComplete should inform that ISSUE_REFRESHED event will be generated", function () {
            this.sandbox.spy(this.module, "trigger");
            this.module.options = {useJIRAEvents: true};
            this.module.issueLoader.trigger("issueLoaded",
                {issue: {id: 12345, key: "JRA-123"}},
                {},
                {issueEntity: {}}
            );

            sinon.assert.calledWith(this.module.trigger, "loadComplete", this.module.model, this.generateLoadCompleteOptions(true));
        });

        test("When options.useJIRAEvents===false then triggering loadComplete should inform that ISSUE_REFRESHED event will not be generated", function () {
            this.sandbox.spy(this.module, "trigger");

            this.module.options = {useJIRAEvents: false};
            this.module.issueLoader.trigger("issueLoaded",
                {issue: {id: 12345, key: "JRA-123"}},
                {},
                {issueEntity: {}}
            );

            sinon.assert.calledWith(this.module.trigger, "loadComplete", this.module.model, this.generateLoadCompleteOptions(false));
        });

        test("When options.useJIRAEvents is undefined then triggering loadComplete should inform that ISSUE_REFRESHED event will not be generated", function () {
            this.sandbox.spy(this.module, "trigger");

            this.module.options = {};
            this.module.issueLoader.trigger("issueLoaded",
                {issue: {id: 12345, key: "JRA-123"}},
                {},
                {issueEntity: {}}
            );

            sinon.assert.calledWith(this.module.trigger, "loadComplete", this.module.model, this.generateLoadCompleteOptions(false));
        });
    });

});
