AJS.test.require(["com.atlassian.jira.jira-issue-nav-components:issueeditor", "com.atlassian.jira.jira-issue-nav-components:issueeditor-test"], function() {
    "use strict";

    require([
        "jira/components/libs/underscore",
        "jquery",
        "jira/components/issueeditor/entities/fields",
        "jira/components/issueeditor/views/field",
        "jira/components/issueviewer/legacy/issueeventbus",
        'jira/util/events',
        "jira/components/issueeditor/eventtypes",
        "jira/components/issueeditor/inlineeditutils",
        'jira/util/events/reasons',
        'jira/ajs/select/multi-select',
        'jira/ajs/select/single-select',
        'jira/components/issueeditor/analytics/viewissue',
        'jira/components/issueviewer/services/darkfeatures'
    ], function(
        _,
        jQuery,
        FieldsCollection,
        FieldView,
        IssueEventBus,
        Events,
        EventTypes,
        InlineEditUtils,
        Reasons,
        AJSMultiSelect,
        AJSSingleSelect,
        Analytics,
        DarkFeatures
    ) {
        var AJSParams = AJS.params;

        module('jira/components/issueeditor/views/field', {
            setup: function() {
                this.sandbox = sinon.sandbox.create();

                //Stop inline layer from appending to window.top so qunit iframe runner works
                AJSParams.ignoreFrame = true;
                this.$el = jQuery("<div class='value editable-field inactive'><a href='#'><span class='link-child'>initial</span><strong class='twixi'></strong></a>" +
                    "<span class='overlay-icon aui-iconfont-edit'></span></div>").appendTo("#qunit-fixture");
                this.collection = new FieldsCollection();
                this.collection.add({
                    id: "summary",
                    editHtml: "<div class='field-group'><input type=\"text\" name=\"myinput\" value=\"initial\"></div>"
                });
                this.model = this.collection.at(0);
                this.view = new FieldView({
                    issueEventBus: new IssueEventBus(),
                    model: this.model,
                    el: this.$el
                });

                this.assertInEditMode = _.bind(function() {
                    equal(this.model.getEditing(), true, "Model says we are editing");
                    ok(this.$el.find("[name=myinput]"), "Input element exists");
                    equal(this.$el.hasClass("inactive"), false, "Inactive class is not present on the element");
                }, this);

                this.assertInReadMode = _.bind(function() {
                    equal(!!this.model.getEditing(), false, "Model says we are not editing");
                    equal(this.$el.find("span").html(), "initial", "Span exists");
                    equal(this.$el.hasClass("inactive"), true, "Inactive class is present on the element");
                }, this);
            },
            teardown: function() {
                this.$el.remove();
                this.sandbox.restore();
            },

            setEditableFieldAnalyticsSpies: function() {
                this.spy(Analytics, 'triggerEditableFieldSingleClick');
                this.spy(Analytics, 'triggerEditableFieldDoubleClick');
                this.spy(Analytics, 'triggerEditableFieldClickPencilIcon');
            }
        });

        test("Readonly mode is default state", function() {
            this.assertInReadMode();
        });

        asyncTest("when readonly view is clicked, we switch to edit", function() {
            this.$el.click();
            var tester = this;
            setTimeout(function() {
                tester.assertInEditMode();
                start();
            }, 400);
        });

        test("when model is set to editing, so is the view", function() {
            this.model.edit();
            this.assertInEditMode();
        });

        test("when model is set to editing, inline edit started event is fired", function() {
            var spy = sinon.spy();
            Events.bind(EventTypes.INLINE_EDIT_STARTED, spy);
            this.model.edit();
            equal(spy.callCount, 1);
            equal(spy.args[0][1], "summary");
            var $inputEls = spy.args[0][3].children();
            equal($inputEls.length, 1);
            ok($inputEls.is(".field-group"));
        });

        test("when model is set to editing, inline edit receives multiple elements", function() {
            this.model.setEditHtml("<input type=\"text\" name=\"myinput1\" ><input type=\"text\" name=\"myinput2\" ><input type=\"text\" name=\"myinput3\" >");
            var spy = sinon.spy();
            Events.bind(EventTypes.INLINE_EDIT_STARTED, spy);
            this.model.edit();
            equal(spy.callCount, 1);
            var $inputEls = spy.args[0][3].children();
            equal($inputEls.length, 3);
        });

        test("When cancel event is triggered on collection we switch to readonly mode", function() {
            this.model.edit(); // switch to edit
            this.collection.cancelEdit();
            this.assertInReadMode();
        });

        test("Update causes params to be updated in model", function() {
            this.model.edit(); // switch to edit
            this.$el.find(":input").val("My New Val");
            this.model.update(this.$el);
            deepEqual(this.model.getParams(), {myinput: "My New Val"});
        });

        test("Cancel is triggered either by ESC or via onCancel call", function() {

            this.model.edit(); // switch to edit
            var spy = sinon.spy();

            this.model.cancelEdit = spy;
            var e = new jQuery.Event("keyup");
            e.keyCode = 50;
            this.$el.trigger(e);
            equal(spy.callCount, 0, "Random keyup shouldn't cancel the edit.");
            e = new jQuery.Event("keyup");
            e.keyCode = 27;
            this.$el.trigger(e);
            equal(spy.callCount, 1, "ESC should cancel the edit.");

            this.model.edit({}); // switch to edit

            this.$el.find(".cancel").click();
            equal(spy.callCount, 2, "Clicking on cancel should also cancel the edit.");
        });

        test("Validation error in model switches view to edit mode", function() {
            this.model.setValidationError("<input/>", "An error occur");
            equal(this.view.$el.find(":text").length, 1, "Expected field to be shown");
            equal(this.model.getEditing(), true, "Expected model to be in edit mode");
        });

        test("Validation error should be appended", function() {
            this.model.setValidationError("<input/><div class=\"error\">An error occur</div>", "An error occur");
            equal(this.view.$el.find(".error").length, 1, "Expected only on error message");
            equal(this.view.$el.find(".error").text(), "An error occur");
        });

        asyncTest("Clicking on link doesn't enter edit", function() {
            var spy = sinon.spy();
            var $el = this.$el;
            this.model.edit = spy;
            $el.find("span.link-child").click();
            setTimeout(function() {
                equal(spy.callCount, 0, "Expected edit not to be called because we clicked a child of a link");
                $el.find("summary").click();
                setTimeout(function() {
                    equal(spy.callCount, 0, "Expected edit not to be called because we clicked a link");
                    $el.click();
                    setTimeout(function() {
                        equal(spy.callCount, 1, "Expected edit to to be called because we DIDN'T click a link");
                        start();
                    }, 400);
                }, 400);
            }, 400);
        });

        asyncTest("No duplicate INLINE_EDIT_FOCUSED and INLINE_EDIT_BLURRED events are fired", function() {

            var focusedSpy = sinon.spy();
            var blurredSpy = sinon.spy();
            var blurDelay = InlineEditUtils.BLUR_FOCUS_TIMEOUT + 1;

            Events.bind(EventTypes.INLINE_EDIT_FOCUSED, focusedSpy);
            Events.bind(EventTypes.INLINE_EDIT_BLURRED, blurredSpy);

            this.model.edit();

            var $input = this.$el.find("input");
            $input.focus();
            $input.focus();

            setTimeout(function() {
                $input.blur();
                setTimeout(function() {
                    $input.blur();
                    setTimeout(function() {
                        equal(focusedSpy.callCount, 1, "Only one INLINE_EDIT_FOCUSED event fired");
                        equal(blurredSpy.callCount, 1, "Only one INLINE_EDIT_BLURRED event fired");
                        start();
                    }, blurDelay);
                }, blurDelay);
            }, 0);
        });

        test("Clicking on element that prevents default doesn't enter edit", function() {
            var spy = sinon.spy();
            var handlerRan = false;

            this.model.edit = spy;

            this.$el.click(function(e) {
                handlerRan = true;
                e.preventDefault();
            });
            this.$el.find(".twixi").click();
            ok(handlerRan);
            equal(spy.callCount, 0, "Expected edit not to be called because we clicked a child of a link");
        });

        test("Cancel on blur when clean", function() {
            var cancelSpy = sinon.spy();
            var saveSpy = sinon.spy();

            this.model.cancelEdit = cancelSpy;
            this.model.onSave(saveSpy);
            this.model.edit();

            Events.trigger(EventTypes.INLINE_EDIT_FOCUSED, ["summary"]);
            Events.trigger(EventTypes.INLINE_EDIT_BLURRED, ["summary"]);

            equal(cancelSpy.callCount, 1, "Expected cancel to occur since field is clean");
            equal(saveSpy.callCount, 0, "Expected save to NOT occur since field is clean");
        });

        test("Save on blur when dirty", function() {
            var cancelSpy = sinon.spy();
            var saveSpy = sinon.spy();

            this.model.cancelEdit = cancelSpy;
            this.model.onSave(saveSpy);
            this.model.edit();
            this.$el.find('input').val('dirty');

            Events.trigger(EventTypes.INLINE_EDIT_FOCUSED, ["summary"]);
            Events.trigger(EventTypes.INLINE_EDIT_BLURRED, ["summary"]);

            equal(cancelSpy.callCount, 0, "Expected cancel to NOT occur since field is dirty");
            equal(saveSpy.callCount, 1, "Expected save to occur since field is dirty");
        });

        test("Blurring twice does not trigger two saves", function() {
            var cancelSpy = sinon.spy();
            var saveSpy = sinon.spy();

            this.model.cancelEdit = cancelSpy;
            this.model.onSave(function() {
                // mock out what the EditIssueController would do
                this.handleSaveStarted();
            });
            this.model.onSave(saveSpy);
            this.model.edit();
            this.$el.find('input').val('dirty');

            Events.trigger(EventTypes.INLINE_EDIT_FOCUSED, ["summary"]);
            Events.trigger(EventTypes.INLINE_EDIT_BLURRED, ["summary"]);
            Events.trigger(EventTypes.INLINE_EDIT_BLURRED, ["summary"]);

            equal(cancelSpy.callCount, 0, "Expected cancel to NOT occur since field is dirty");
            equal(saveSpy.callCount, 1, "Expected save to occur only once");
        });

        test("Save or cancel is not triggered when save options gain focus", function() {
            var clock = sinon.useFakeTimers();
            var cancelSpy = sinon.spy();
            var saveSpy = sinon.spy();

            this.model.cancelEdit = cancelSpy;
            this.model.onSave(saveSpy);
            this.model.edit();
            this.$el.find('input').val('dirty');

            this.$el.find("input").focus();
            this.$el.find("input").blur();
            this.$el.find('.save-options button:first').focus();

            clock.tick(InlineEditUtils.BLUR_FOCUS_TIMEOUT + 1);

            equal(cancelSpy.callCount, 0, "Expected cancel to NOT occur since save options has focus");
            equal(saveSpy.callCount, 0, "Expected save to NOT occur since save options has focus");

            clock.restore();
        });

        test("Save or cancel is not triggered when an input regains focus from save options", function() {
            var clock = sinon.useFakeTimers();
            var cancelSpy = sinon.spy();
            var saveSpy = sinon.spy();

            this.model.cancelEdit = cancelSpy;
            this.model.onSave(saveSpy);
            this.model.edit();
            this.$el.find('input').val('dirty');
            this.$el.find('.save-options button:first').focus();
            this.$el.find('.save-options button:first').blur();
            this.$el.find('input').focus();

            clock.tick(InlineEditUtils.BLUR_FOCUS_TIMEOUT + 1);

            equal(cancelSpy.callCount, 0, "Expected cancel to NOT occur since save options has focus");
            equal(saveSpy.callCount, 0, "Expected save to NOT occur since save options has focus");

            clock.restore();
        });

        test("reveal is triggered on both edit and focus", function() {
            var revealSpy = sinon.spy();
            this.$el.bind("reveal", revealSpy);
            this.view.switchToEdit();
            equal(revealSpy.callCount, 1);
            this.view.focus();
            equal(revealSpy.callCount, 2);

        });

        test("JRA-28241 can prevent submit through event", function() {
            var cancelSpy = sinon.spy();
            var saveSpy = sinon.spy();

            this.model.cancelEdit = cancelSpy;
            this.model.onSave(saveSpy);
            this.model.edit();
            this.$el.find('input').val('dirty');

            this.$el.find("form").bind("before-submit", function(e) {
                e.preventDefault();
            });

            this.$el.find("form").submit();

            equal(cancelSpy.callCount, 0, "Expected cancel to NOT occur since save options has focus");
            equal(saveSpy.callCount, 0, "Expected save to NOT occur since save options has focus");
        });

        asyncTest("JRADEV-12018 - Inline edit is not enabled when event is transmitted from within a parent with class uneditable", function() {
            var spy = sinon.spy();
            var $el = this.$el;
            this.model.edit = spy;
            var uneditableElement = jQuery("<div class='uneditable'>This is an uneditable div</div>");
            $el.append(uneditableElement);
            uneditableElement.click();
            setTimeout(function() {
                equal(spy.callCount, 0, "Expected edit not to be called because we clicked an uneditable element");
                start();
            }, 400);
        });

        test("_handleEditingStarted() should trigger a EventTypes.NEW_CONTENT_ADDED", function() {
            var triggerSpy = sinon.spy(Events, "trigger");

            this.view._handleEditingStarted();

            equal(triggerSpy.callCount, 2, "Two events are triggered");
            equal(triggerSpy.firstCall.args[0], EventTypes.NEW_CONTENT_ADDED, "Event is EventTypes.NEW_CONTENT_ADDED");
            equal(typeof triggerSpy.firstCall.args[1][1], "string", "Ensure event has a reason argument");
            equal(triggerSpy.firstCall.args[1][1], Reasons.inlineEditStarted, "Event reason is JIRA.CONTENT_ADDED_REASON.inlineEditStarted");
            equal(triggerSpy.secondCall.args[0], EventTypes.INLINE_EDIT_STARTED, "Second event is EventTypes.INLINE_EDIT_STARTED");

            triggerSpy.restore();
        });

        test("_handleEditingStarted() should trigger showSuggestions for single selects only", function() {

            var showSingleSuggestionsSpy = sinon.spy();
            var showMultiSuggestionsSpy = sinon.spy();
            var $singleSelect = jQuery("<select><option>blah</option></select>").bind("showSuggestions", showSingleSuggestionsSpy);
            var $multiSelect = jQuery("<select multiple='multiple'><option>blah</option><option>blah2</option></select>").bind("showSuggestions", showMultiSuggestionsSpy);

            this.view.$el.append($singleSelect);
            this.view.$el.append($multiSelect);

            new AJSMultiSelect({
                element: $multiSelect
            });

            new AJSSingleSelect({
                element: $singleSelect
            });

            this.view._handleEditingStarted();

            equal(showSingleSuggestionsSpy.callCount, 1, "Expected event to fire for single-selects");
            equal(showMultiSuggestionsSpy.callCount, 0, "Expected event not to fire for multi-selects");
        });

        test("editField is triggered on switchToEdit", function() {
            var triggerCallback = sinon.spy();
            this.view.on("editField", triggerCallback);
            this.view.switchToEdit();

            sinon.assert.calledOnce(triggerCallback);
            sinon.assert.calledWith(triggerCallback, sinon.match({ fieldId: "summary" }));
        });

        test("editFieldCancel is triggered on switchToRead", function() {
            var triggerCallback = sinon.spy();
            this.view.on("editFieldCancel", triggerCallback);
            this.view.switchToRead();

            sinon.assert.calledOnce(triggerCallback);
            sinon.assert.calledWith(triggerCallback, sinon.match({ fieldId: "summary" }));
        });

        test("analytics event is triggered on single click of editable field", function() {
            this.setEditableFieldAnalyticsSpies();
            var clock = sinon.useFakeTimers();

            this.$el.click();
            clock.tick(400);
            sinon.assert.calledOnce(Analytics.triggerEditableFieldSingleClick);
            sinon.assert.notCalled(Analytics.triggerEditableFieldDoubleClick);
            sinon.assert.notCalled(Analytics.triggerEditableFieldClickPencilIcon);
            clock.restore();
        });

        test("analytics event is triggered on double click of editable field", function() {
            this.setEditableFieldAnalyticsSpies();

            this.$el.click();
            this.$el.click();
            sinon.assert.calledOnce(Analytics.triggerEditableFieldDoubleClick);
            sinon.assert.notCalled(Analytics.triggerEditableFieldSingleClick);
            sinon.assert.notCalled(Analytics.triggerEditableFieldClickPencilIcon);
        });

        test("analytics event is triggered on pencil icon click of editable field", function() {
            this.setEditableFieldAnalyticsSpies();

            this.$el.find(".overlay-icon.aui-iconfont-edit").click();
            sinon.assert.calledOnce(Analytics.triggerEditableFieldClickPencilIcon);
            sinon.assert.notCalled(Analytics.triggerEditableFieldSingleClick);
            sinon.assert.notCalled(Analytics.triggerEditableFieldDoubleClick);
        });

        test("Should show button bar if flag enabled", function() {
            this.sandbox.stub(DarkFeatures.RTE_ENABLED, "enabled").returns(true);

            this.model.setValidationError('<div class="jira-wikifield"><textarea name="myinput"> </textarea><div class="wiki-preview" id="preview-link"> </div></div>', "An error occur");

            ok(this.$el.find(".save-options").hasClass("show-button-bar"));
        });

        test("Should not show button bar if flag not enabled", function() {
            this.sandbox.stub(DarkFeatures.RTE_ENABLED, "enabled").returns(false);

            this.model.setValidationError('<div class="jira-wikifield"><textarea name="myinput"> </textarea><div class="wiki-preview" id="preview-link"> </div></div>', "An error occur");

            ok(!this.$el.find(".save-options").hasClass("show-button-bar"));
        });

        test("Should not show button bar if flag enabled but wikifield not present", function() {
            this.sandbox.stub(DarkFeatures.RTE_ENABLED, "enabled").returns(true);

            this.model.setValidationError('<div class="jira-field"><textarea name="myinput"> </textarea><div class="wiki-preview" id="preview-link"> </div></div>', "An error occur");

            ok(!this.$el.find(".save-options").hasClass("show-button-bar"));
        });
    });
});
