define("jira/components/detailslayout", ["require"], function(require) {
    "use strict";

    var _ = require("jira/components/libs/underscore");
    var IssueComponentResolver = require("jira/components/issuecomponentresolver");
    var Marionette = require("jira/components/libs/marionette-2.1.0");
    var SimpleIssueList = require("jira/components/simpleissuelist");
    var ViewContainer = require("jira/components/detailslayout/views/container");
    var ViewEmpty = require("jira/components/detailslayout/views/empty");
    var ViewLayout = require("jira/components/detailslayout/views/layout");
    var ViewLoading = require("jira/components/detailslayout/views/loading");
    var Tools = require("jira/components/detailslayout/controllers/tools");

    /**
     * Utility method to create a debounced version of a function. The generated function will have
     * the properties:
     *
     *   - Initial call will happen after <initial> ms
     *   - After the initial call, it will postpone its execution until after <cooldown> milliseconds have elapsed
     *     since the last time it was invoked.
     *   - After the <cooldown> period, the function will be delayed <initial> ms again.
     *
     * Example timelines with values initial=100 and cooldown=500
     *   - Calling the function once:
     *          0ms - Function called
     *        100ms - Function executed
     *
     *   - Calling the function 4 times every 50ms, then again after 1s
     *          0ms - Function called
     *         50ms - Function called
     *        100ms - Function executed (1)
     *        100ms - Function called
     *        150ms - Function called
     *        650ms - Function executed (2)
     *       1000ms - Function called
     *       1100ms - Function executed (3)
     *
     * @param {Function} fn Function to execute
     * @param {number} initial Initial delay in ms.
     * @param {number} cooldown Subsequent delays, in ms.
     * @returns {Function}
     */
    function variableDebounce(fn, initial, cooldown) {
        var timer;
        var currentTimer = initial;

        return function() {
            var args = arguments;
            clearTimeout(timer);
            timer = setTimeout(function() {
                fn.apply(this, args);
                currentTimer = initial;
            }, currentTimer);
            currentTimer = cooldown;
        };
    }

    return Marionette.ViewManager.extend({
        _buildIssueComponent: function () {
            var IssueComponent = IssueComponentResolver.resolve();
            var component = this.issueComponent = new IssueComponent();
            var options = this.options;


            this._loadIssueInViewerComponent = variableDebounce(_.bind(function(issueData) {
                component.loadIssue({
                    id: issueData.id,
                    key: issueData.key,
                    detailView: !options.shouldUpdateCurrentProject
                }).always(_.bind(function() {
                    this.hideLoading();
                }, this));
            }, this), 100, 500);

            this.listenTo(component, {
                "loadError": function (issueData) {
                    this.simpleIssueList.disableIssue(issueData.issueId);
                    this.trigger("editorError");
                },
                "loadComplete": function (issue, options) {
                    this.adjustSize();

                    // The editor model does not provide info about the issue type, we can't
                    // update that value on the list.
                    this.simpleIssueList.updateIssue(issue.id, {
                        key: issue.get('entity').key,
                        summary: issue.get('entity').summary,
                        status: issue.get('entity').status.name
                    });

                    var eventPayload = {
                        issueKey: issue.get('entity').key,
                        issueId: issue.id,
                        issueEditorOptions: options,
                        projectId: issue.get('projectId'),
                        projectKey: issue.get('projectKey'),
                        projectType: issue.get('projectType')
                    };
                    if (options && options.loadReason === "issues-cache-refresh") {
                        this.trigger("editorLoadedFromCache", eventPayload);
                    } else {
                        this.trigger("editorLoaded", eventPayload);
                    }
                },
                "fieldsLoaded": function() {
                    this.trigger("editorFieldsLoaded");
                },
                "saveSuccess": function (event) {
                    this.simpleIssueList.refreshIssue(event.issueId);
                    this.trigger("editor:saveSuccess", {
                        event: event.issueId,
                        savedFieldIds: event.savedFieldIds,
                        savedFieldTypes: event.savedFieldTypes,
                        duration: event.duration
                    });
                },
                "editField": function (event) {
                    this.trigger("editor:editField", event);
                },
                "editFieldCancel": function (event) {
                    this.trigger("editor:editFieldCancel", event);
                },
                "linkToIssue": function(event) {
                    this.trigger("linkToIssue", event);
                },
                "refineViewer": function(event) {
                    event.preventDefault();
                    this.updateIssueComponent(event.query);
                },
                "linkInErrorMessage": function(event) {
                    event.preventDefault();
                    this.simpleIssueList.selectIssue(event.issueData.id);
                },
                "individualPanelRendered": function(panelRendered) {
                    this.trigger("individualPanelRendered", panelRendered);
                }
            });
        },

        _buildSimpleIssueList: function () {
            this.simpleIssueList = new SimpleIssueList({
                baseURL: this.baseURL,
                displayInlineIssueCreate: this.options.displayInlineIssueCreate
            });
            this.listenTo(this.simpleIssueList, {
                "select": function (issueData) {
                    this.trigger('select', issueData);
                },
                "update": function () {
                    this.trigger("list:update");
                },
                "refresh": function () {
                    this.trigger("list:refresh");
                },
                "sort": function (jql) {
                    this.trigger("list:sort", jql);
                },
                "list:select": function(event) {
                    this.trigger("list:select", {
                        id: event.id,
                        key: event.key,
                        absolutePosition: event.absolutePosition,
                        relativePosition: event.relativePosition
                    });
                },
                "list:pagination": function() {
                    this.trigger("list:pagination");
                },
                "before:loadpage": function() {
                    this.showLoading();
                },
                "error:loadpage": function(errorInfo) {
                    this.hideLoading();
                    this.trigger("error:loadpage", errorInfo);
                },
                "issueCreated": function(issueInfo) {
                    this.trigger("issueCreated", issueInfo);
                }
            });
        },

        _buildContainerView: function () {
            this.containerView = this.buildView("containerView", function() {
                var view = new ViewContainer();
                this.listenTo(view, {
                    "render": function() {
                        this.getView("layoutView").render();
                        this.getView("emptyView").render();
                        this.getView("loadingView").render();
                        view.$el.prepend(this.getView("layoutView").$el);
                    }
                });
                return view;
            });
        },

        _buildEmptyView: function() {
            this.emptyView = this.buildView("emptyView", function() {
                var view;
                if (typeof this.options.emptyViewFactory === "function") {
                    view = this.options.emptyViewFactory();
                } else {
                    view = new ViewEmpty();
                }

                this.listenTo(view, {
                    "render": function () {
                        this.containerView.$el.append(view.$el);
                        this.adjustSize();
                        this.trigger("empty");
                    }
                });
                return view;
            });
        },

        _showEmptyView: function () {
            this.layoutView.$el.detach();
            this.containerView.$el.empty();
            this.containerView.showView(this.emptyView);
            this.adjustSize();
            this.trigger("empty");
        },

        _buildTools: function() {
            this.tools = new Tools({
                showExpand: this.useExpand
            });
            this.listenTo(this.tools, {
                "expand": function() {
                    this.trigger('expand');
                },
                "pager:next": function() {
                    var data = this.selectNext();
                    this.trigger("pager:next", data);
                },
                "pager:previous": function() {
                    var data = this.selectPrevious();
                    this.trigger("pager:previous", data);
                }
            });
        },

        _buildLayoutView: function() {
            this.layoutView = this.buildView("layoutView", function () {
                var view = new ViewLayout();
                this.listenTo(view, {
                    "render": function () {
                        view.issueComponent._ensureElement();
                        this.issueComponent.setContainer(view.issueComponent.$el);

                        view.issuesList._ensureElement();
                        this.simpleIssueList.show(view.issuesList.$el);

                        view.tools._ensureElement();
                        this.tools.show(view.tools.$el);
                    },
                    "resize": function () {
                        this.adjustSize();
                    }
                });
                return view;
            });
        },

        _buildLoadingView: function() {
            this.loadingView = this.buildView("loadingView", function () {
                return new ViewLoading();
            });
        },

        _buildComponents: function() {
            this._buildIssueComponent();
            this._buildSimpleIssueList();
            this._buildTools();
        },

        _buildViews: function() {
            this._buildLayoutView();
            this._buildEmptyView();
            this._buildLoadingView();
            this._buildContainerView();

            this.containerView.render();
        },

        initialize: function (options) {
            this.baseURL = options.baseURL;
            this.useExpand = options.useExpand || false;

            this._buildComponents();
            this._buildViews();
        },

        _showLoadingView: function () {
            this.containerView.showLoading(this.loadingView);
        },

        _hideLoadingView: function () {
            this.containerView.hideLoading(this.loadingView);
        },

        _showLayoutView: function() {
            this.emptyView.$el.detach();
            this.containerView.showView(this.layoutView);
            this.layoutView.showDraggable();
            this.adjustSize();
        },

        show: function (el) {
            el.replaceWith(this.containerView.$el);
            return this.containerView.$el;
        },

        onDestroy: function() {
            this.tools.destroy();
            this.issueComponent.close();
            this.simpleIssueList.destroy();
            Marionette.ViewManager.prototype.onDestroy.call(this);
        },

        showLoading: function () {
            this._showLoadingView();
        },

        hideLoading: function() {
            this._hideLoadingView();
        },

        load: function (searchResults, issueIdOrKey) {
            if (this.searchResults) {
                this.stopListening(this.searchResults);
                delete this.searchResults;
            }
            this.searchResults = searchResults;
            this.listenTo(this.searchResults, {
                "issueDeleted": function () {
                    if (!this.searchResults.length) {
                        this._showEmptyView();
                    }
                },
                "select selectIssueNotInList": function (issueModel) {
                    this.showLoading();
                    this._loadIssueInViewerComponent({
                        id: issueModel.id,
                        key: issueModel.get('key')
                    });
                }
            });

            if (this.searchResults.isEmptySearch()) {
                this._showEmptyView();
            } else {
                this.simpleIssueList.load(this.searchResults, issueIdOrKey);
                this.tools.load(this.searchResults);
                this._showLayoutView();
            }

            this.trigger("list:render");
        },

        adjustSize: function () {
            _.defer(_.bind(function () {
                if (this.getView("layoutView")) {
                    this.simpleIssueList.adjustSize();
                    this.getView("layoutView").maximizeDetailPanelHeight();
                    this.issueComponent.applyResponsiveDesign();
                    this.getView("layoutView").updateDraggable();
                }

                if (this.getView("emptyView")) {
                    var emptyViewContainer = this.getView("emptyView").$el;
                    var issueContainerTop = emptyViewContainer.length && emptyViewContainer.offset().top;
                    emptyViewContainer.css("height", window.innerHeight - issueContainerTop);
                }
            }, this));
        },

        refreshIssue: function (issueId) {
            this.simpleIssueList.refreshIssue(issueId);
            return this.issueComponent.refreshIssue();
        },

        removeIssue: function (issueId) {
            this.showLoading();
            return this.simpleIssueList.removeIssue(issueId)
                .always(_.bind(function () {
                    this.hideLoading();
                }, this))
                .done(_.bind(function (listLength) {
                    if (listLength === 0) {
                        this._showEmptyView();
                    }
                }, this));
        },

        getActiveIssueId: function () {
            return this.issueComponent.getIssueId();
        },

        getActiveIssueKey: function () {
            return this.issueComponent.getIssueKey();
        },

        getActiveProjectId: function() {
            return this.issueComponent.getProjectId();
        },

        getActiveProjectKey: function() {
            return this.issueComponent.getProjectKey();
        },

        getActiveProjectType: function() {
            return this.issueComponent.getProjectType();
        },

        selectNext: function() {
            return this.simpleIssueList.selectNext();
        },

        selectPrevious: function() {
            return this.simpleIssueList.selectPrevious();
        },

        updateIssueComponent: function(params) {
            this.showLoading();
            this.listenToOnce(this.issueComponent, "loadComplete", function() {
                this.hideLoading();
            });
            this.issueComponent.updateIssueWithQuery(params);
        },

        isLoading: function() {
            return this.issueComponent.isCurrentlyLoading();
        },

        hasSavesInProgress: function() {
            if ("hasSavesInProgress" in this.issueComponent) {
                return this.issueComponent.hasSavesInProgress();
            } else {
                return false;
            }
        },

        canDismissComment: function() {
            return this.issueComponent.canDismissComment();
        },

        getEditorFields: function() {
            if ("getFields" in this.issueComponent) {
                return this.issueComponent.getFields();
            } else {
                return undefined;
            }
        },

        abortPending: function() {
            return this.issueComponent.abortPending();
        },

        beforeHide: function() {
            return this.issueComponent.beforeHide();
        },

        beforeShow: function() {
            return this.issueComponent.beforeShow();
        },

        removeIssueMetadata: function() {
            return this.issueComponent.removeIssueMetadata();
        },

        editField: function(field) {
            if ("editField" in this.issueComponent) {
                return this.issueComponent.editField(field);
            }
        },

        focusList: function() {
            this.simpleIssueList.focus();
        },

        focusEditor: function() {
            this.issueComponent.focus();
        }
    });
});
