AJS.test.require("com.atlassian.jira.jira-issue-nav-components:detailslayout-test", function () {
    "use strict";

    require([
        "jira/components/issuecomponentresolver",
        "jira/components/issueeditor",
        "jira/components/issueviewer",
        "jira/components/detailslayout/controllers/tools",
        "jira/components/detailslayout/views/empty",
        "jira/components/detailslayout/views/loading",
        "jira/components/detailslayout/views/layout",
        "jira/components/detailslayout/views/container",
        "jira/components/search/results",
        "jira/components/simpleissuelist",
        "jira/components/test-utils/marionettemocker",
        "jira/components/test-utils/mockutils",
        "jquery"
    ], function(
        IssueComponentResolver,
        IssueEditor,
        IssueViewer,
        ToolsController,
        EmptyView,
        LoadingView,
        LayoutView,
        ContainerView,
        SearchResults,
        SimpleIssueList,
        MarionetteMocker,
        MockUtils,
        jQuery
    ) {
        module("jira/components/detailslayout", {
            setup: function () {
                this.setupViews();

                this.sandbox = sinon.sandbox.create();

                this.tools = MarionetteMocker.createEventedMock(sinon, ToolsController);
                this.searchResults = MarionetteMocker.createEventedMock(sinon, SearchResults);
                this.simpleIssueList = MarionetteMocker.createEventedMock(sinon, SimpleIssueList);

                this.issueComponentResolver = this.sandbox.stub(IssueComponentResolver);
                this.issueViewerComponent = MarionetteMocker.createEventedMock(sinon, IssueViewer);
                this.issueEditorComponent = MarionetteMocker.createEventedMock(sinon, IssueEditor);

                this.issueComponentResolver.resolve.returns(this.issueEditorComponent.constructor);
                this.issueEditorComponent.loadIssue.returns(new jQuery.Deferred());

                this.DetailsLayoutConstructor = MockUtils.requireWithMocks("jira/components/detailslayout", {
                    "jira/components/detailslayout/controllers/tools": this.tools.constructor,
                    "jira/components/detailslayout/views/empty": this.emptyView.constructor,
                    "jira/components/detailslayout/views/loading": this.loadingView.constructor,
                    "jira/components/detailslayout/views/layout": this.layoutView.constructor,
                    "jira/components/detailslayout/views/container": this.containerView.constructor,
                    "jira/components/simpleissuelist": this.simpleIssueList.constructor,
                    "jira/components/issuecomponentresolver": this.issueComponentResolver
                });
                this.detailsLayout = new this.DetailsLayoutConstructor();
            },

            teardown: function() {
                this.sandbox.restore();
            },

            setupViews: function() {
                this.defaultElement = jQuery("<div></div>");

                this.emptyView = MarionetteMocker.createEventedMock(sinon, EmptyView, { $el: this.defaultElement });
                this.loadingView = MarionetteMocker.createEventedMock(sinon, LoadingView, { $el: this.defaultElement });
                this.layoutView = MarionetteMocker.createEventedMock(sinon, LayoutView, { $el: this.defaultElement });
                this.containerView = MarionetteMocker.createEventedMock(sinon, ContainerView, { $el: this.defaultElement });
            },

            generateIssueMock: function() {
                return new Backbone.Model({
                    id: 1,
                    entity: {
                        key: 'TEST-1',
                        summary: 'My issue',
                        status: {
                            name: "ToDo"
                        }
                    },
                    projectId: 1000,
                    projectKey: 'TEST',
                    projectType: 'software'
                });
            },

            resetWithInlineEditingDisabled: function() {
                this.issueComponentResolver.resolve.returns(this.issueViewerComponent.constructor);
                this.detailsLayout = new this.DetailsLayoutConstructor();
            }
        });

        test("When SimpleIssueList wants to be refreshed, the layout triggers the 'list:refresh' event", function () {
            var onRefresh = this.spy();
            this.detailsLayout.on("list:refresh", onRefresh);

            this.simpleIssueList.trigger("refresh");

            sinon.assert.calledOnce(onRefresh);
        });

        test("When SimpleIssueList wants to be sorted with a different JQL, the layout triggers the 'list:sort' event", function () {
            var onSort = this.spy();
            this.detailsLayout.on("list:sort", onSort);

            this.simpleIssueList.trigger("sort", "project IN (DEMO)");

            sinon.assert.calledOnce(onSort);
            sinon.assert.calledWith(onSort, "project IN (DEMO)");
        });

        test("When SimpleIssueList triggers the 'list:select' event, it gets re-triggered", function () {
            var onSelect = this.spy();
            this.detailsLayout.on("list:select", onSelect);
            var issueData = {id: 1, key: 'DEMO-1'};

            this.simpleIssueList.trigger("list:select", issueData);

            sinon.assert.calledOnce(onSelect, issueData);
        });

        test("When the editor fails to load an issue, it disables the issue in the list", function () {
            this.issueEditorComponent.trigger("loadError", {issueId: "DEMO-123"});

            sinon.assert.calledOnce(this.simpleIssueList.disableIssue);
            sinon.assert.calledWith(this.simpleIssueList.disableIssue, "DEMO-123");
        });

        test("Removing an issue removes it from the SimpleIssueList", function () {
            var deferred = new jQuery.Deferred();
            this.simpleIssueList.removeIssue.returns(deferred.promise());

            this.detailsLayout.removeIssue(1);

            sinon.assert.calledOnce(this.simpleIssueList.removeIssue);
            sinon.assert.calledWith(this.simpleIssueList.removeIssue, 1);
        });

        test("Removing an issue shows the loading screen", function () {
            var deferred = new jQuery.Deferred();
            this.simpleIssueList.removeIssue.returns(deferred.promise());

            this.detailsLayout.removeIssue(1);

            sinon.assert.calledOnce(this.containerView.showLoading);
            deferred.resolve(10);
            sinon.assert.calledOnce(this.containerView.hideLoading);
        });

        test("Removing an issue shows the empty view if it was the last issue", function () {
            var deferred = new jQuery.Deferred();
            this.simpleIssueList.removeIssue.returns(deferred.promise());

            this.detailsLayout.removeIssue(1);
            deferred.resolve(0); //No more issues left

            sinon.assert.calledWith(this.containerView.showView, this.emptyView);
        });

        test("When it receives a 'before:loadpage' event, it shows the loading view", function() {
            this.spy(this.detailsLayout, "showLoading");

            this.detailsLayout.simpleIssueList.trigger("before:loadpage");

            sinon.assert.called(this.detailsLayout.showLoading);
        });

        test("When it receives an 'error:loadpage' event, it hides the loading view", function() {
            this.spy(this.detailsLayout, "hideLoading");
            var payload = { some: 'data' };

            this.detailsLayout.simpleIssueList.trigger("error:loadpage", payload);

            sinon.assert.called(this.detailsLayout.hideLoading);
        });

        test("When it receives an 'error:loadpage' event, it retriggers it with the event payload", function() {
            this.spy(this.detailsLayout, "trigger");
            var payload = { some: 'data' };

            this.detailsLayout.simpleIssueList.trigger("error:loadpage", payload);

            sinon.assert.calledWith(this.detailsLayout.trigger, "error:loadpage", payload);
        });

        test("When it is loaded, it triggers a 'list:render' event", function() {
            this.spy(this.detailsLayout, "trigger");

            this.detailsLayout.load(this.searchResults);

            sinon.assert.calledWith(this.detailsLayout.trigger, "list:render");
        });

        test("When an issue is selected, it shows the loading view", function() {
            this.spy(this.detailsLayout, "showLoading");
            var mockIssueModel = {
                id: 1,
                get: function() {
                    return "KEY";
                }
            };

            this.detailsLayout.load(this.searchResults);
            this.searchResults.trigger("select", mockIssueModel);

            sinon.assert.calledWith(this.detailsLayout.showLoading);
        });

        test("It ask SimpleIssueList to use IIC", function() {
            new this.DetailsLayoutConstructor({
                displayInlineIssueCreate: true
            });

            equal(this.simpleIssueList.constructor.lastCall.args[0].displayInlineIssueCreate, true);
        });

        test("When SimpleIssueList triggers the event 'issueCreated' it is re-triggered", function() {
            var eventInfo = {};
            var onIssueCreated = this.spy();
            this.detailsLayout.on('issueCreated', onIssueCreated);

            this.simpleIssueList.trigger('issueCreated', eventInfo);

            sinon.assert.calledOnce(onIssueCreated);
            sinon.assert.calledWith(onIssueCreated, eventInfo);
        });

        test("It uses the default EmptyView if a custom one is not provided", function() {
            this.searchResults.isEmptySearch.returns(true);
            this.detailsLayout.load(this.searchResults);

            sinon.assert.calledOnce(this.containerView.showView);
            sinon.assert.calledWith(this.containerView.showView, this.emptyView);
        });

        test("It uses the provided EmptyView", function() {
            var customEmptyView = MarionetteMocker.createEventedMock(sinon, EmptyView, { $el: this.defaultElement });
            var customEmptyViewFactory = this.spy(function() {
                return customEmptyView;
            });
            var detailsLayout = new this.DetailsLayoutConstructor({
                emptyViewFactory: customEmptyViewFactory
            });
            this.searchResults.isEmptySearch.returns(true);

            detailsLayout.load(this.searchResults);

            sinon.assert.calledOnce(customEmptyViewFactory);
            sinon.assert.calledOnce(this.containerView.showView);
            sinon.assert.calledWith(this.containerView.showView, customEmptyView);
        });

        test("When the paginator is used, an event is triggered", function() {
            this.simpleIssueList.selectNext.returns({ current: 2, total: 2 });
            var nextSpy = this.spy();
            this.detailsLayout.on('pager:next', nextSpy);
            this.tools.trigger('pager:next');
            sinon.assert.calledOnce(nextSpy);
            sinon.assert.calledWith(nextSpy, { current: 2, total: 2 });

            this.simpleIssueList.selectPrevious.returns({ current: 1, total: 2 });
            var prevSpy = this.spy();
            this.detailsLayout.on('pager:previous', prevSpy);
            this.tools.trigger('pager:previous');
            sinon.assert.calledOnce(prevSpy);
            sinon.assert.calledWith(prevSpy, { current: 1, total: 2 });
        });

        test("When the expand button is used, an event is triggered", function() {
            var onExpand = this.spy();
            this.detailsLayout.on('expand', onExpand);

            this.tools.trigger('expand');

            sinon.assert.calledOnce(onExpand);
        });

        test("When the detailsLayout component is destroyed, the internal components are destroyed", function() {
            this.detailsLayout.destroy();

            sinon.assert.calledOnce(this.tools.destroy);
            sinon.assert.calledOnce(this.issueEditorComponent.close);
            sinon.assert.calledOnce(this.simpleIssueList.destroy);
        });

        test("When the editor emits editor:saveSuccess event it should be forwarded", function () {
            var spy = this.spy();

            this.detailsLayout.on('editor:saveSuccess', spy);

            this.issueEditorComponent.trigger("saveSuccess", {
                issueId: '123456',
                savedFieldIds: ['fieldId-1'],
                savedFieldTypes: ['fieldType-1'],
                duration: 123
            });

            sinon.assert.calledOnce(spy);
            sinon.assert.calledWith(spy, {
                event: '123456',
                savedFieldIds: ['fieldId-1'],
                savedFieldTypes: ['fieldType-1'],
                duration: 123
            });
        });

        test("When the details layout focuses the list, it delegates into SimpleIssueList", function() {
            this.detailsLayout.focusList();

            sinon.assert.calledOnce(this.simpleIssueList.focus);
        });

        test("When the details layout focuses the editor, it delegates into IssueEditor", function() {
            this.detailsLayout.focusEditor();

            sinon.assert.calledOnce(this.issueEditorComponent.focus);
        });

        test("When the editor emits the 'individualPanelRendered' event, it should be forwarded", function() {
            var spy = this.spy();
            var payload = {data: 123};
            this.detailsLayout.on('individualPanelRendered', spy);

            this.issueEditorComponent.trigger("individualPanelRendered", payload);

            sinon.assert.calledOnce(spy);
            sinon.assert.calledWith(spy, payload);
        });

        test("When the editor emits the 'loadComplete' event and the issue is not from the cache, DetailsLayout emits an 'editorLoaded' event", function() {
            var onEditorLoaded = this.spy();
            this.detailsLayout.on('editorLoaded', onEditorLoaded);

            this.issueEditorComponent.trigger("loadComplete", this.generateIssueMock(), {});

            sinon.assert.calledOnce(onEditorLoaded);
            sinon.assert.calledWith(onEditorLoaded, {
                issueKey: 'TEST-1',
                issueId: 1,
                issueEditorOptions: {},
                projectId: 1000,
                projectKey: 'TEST',
                projectType: 'software'
            });
        });

        test("When the editor emits the 'loadComplete' event and the issue is from the cache, DetailsLayout emits an 'editorLoadedFromCache' event", function() {
            var onEditorLoadedFromCache = this.spy();
            this.detailsLayout.on('editorLoadedFromCache', onEditorLoadedFromCache);

            var options = {
                loadReason: "issues-cache-refresh"
            };
            this.issueEditorComponent.trigger("loadComplete", this.generateIssueMock(), options);

            sinon.assert.calledOnce(onEditorLoadedFromCache);
            sinon.assert.calledWith(onEditorLoadedFromCache, {
                issueKey: 'TEST-1',
                issueId: 1,
                issueEditorOptions: options,
                projectId: 1000,
                projectKey: 'TEST',
                projectType: 'software'
            });
        });

        test("When getting the project Id, it delegates into the IssueEditor", function() {
            this.issueEditorComponent.getProjectId.returns(123);

            var projectId = this.detailsLayout.getActiveProjectId();

            sinon.assert.calledOnce(this.issueEditorComponent.getProjectId);
            equal(projectId, 123);
        });

        test("When getting the project key, it delegates into the IssueEditor", function() {
            this.issueEditorComponent.getProjectKey.returns("KEY");

            var projectKey = this.detailsLayout.getActiveProjectKey();

            sinon.assert.calledOnce(this.issueEditorComponent.getProjectKey);
            equal(projectKey, "KEY");
        });

        test("When getting the project type, it delegates into the IssueEditor", function() {
            this.issueEditorComponent.getProjectType.returns("software");

            var projectType = this.detailsLayout.getActiveProjectType();

            sinon.assert.calledOnce(this.issueEditorComponent.getProjectType);
            equal(projectType, "software");
        });

        test("When Inline Editing is enabled, 'hasSavesInProgress' should be delegated to the Issue Component", function() {
            this.issueEditorComponent.hasSavesInProgress.returns(true);

            var isSaveInProgress = this.detailsLayout.hasSavesInProgress();

            ok(isSaveInProgress);
            sinon.assert.calledOnce(this.issueEditorComponent.hasSavesInProgress);
        });

        test("When Inline Editing is disabled, 'hasSavesInProgress' is always false", function() {
            this.resetWithInlineEditingDisabled();
            this.issueEditorComponent.hasSavesInProgress.returns(true);

            var isSaveInProgress = this.detailsLayout.hasSavesInProgress();

            ok(!isSaveInProgress);
            equal(0, this.issueEditorComponent.hasSavesInProgress.callCount);
        });

        test("When Inline Editing is enabled, 'getEditorFields' should be delegated to the Issue Component", function() {
            this.issueEditorComponent.getFields.returns({data: 123});

            var editorFields = this.detailsLayout.getEditorFields();

            equal(123, editorFields.data);
            sinon.assert.calledOnce(this.issueEditorComponent.getFields);
        });

        test("When Inline Editing is disabled, 'getEditorFields' returns undefined", function() {
            this.resetWithInlineEditingDisabled();
            this.issueEditorComponent.getFields.returns({data: 123});

            var editorFields = this.detailsLayout.getEditorFields();

            ok(typeof editorFields === 'undefined');
            equal(0, this.issueEditorComponent.getFields.callCount);
        });

        test("When Inline Editing is disabled, 'editField' should be delegated to the Issue Component", function() {
            this.detailsLayout.editField('field');

            sinon.assert.calledOnce(this.issueEditorComponent.editField);
            sinon.assert.calledWith(this.issueEditorComponent.editField, 'field');
        });

        test("When Inline Editing is disabled, 'editField' does nothing", function() {
            this.resetWithInlineEditingDisabled();

            this.detailsLayout.editField('field');

            equal(0, this.issueEditorComponent.editField.callCount);
        });
    });
});
