define('jira/inline-issue-create/prefillable-create-issue-dialog/init-form', [
    'underscore',
    'jira/inline-issue-create/prefillable-create-issue-dialog/custom-form-container-factory'
], function (
    _,
    CustomFormContainerFactory
) {
    "use strict";

    /**
     * Initializes the JIRA.Forms.Container form and resolves passed deferred param with JIRA.Forms.Container.
     *
     * @param {jQuery.Deferred} deferred - Deferred that is resolved to the JIRA.Forms.Container form after the WRM resources have been downloaded and the form is ready to be displayed. Can be used to asynchronously show the form when all resources are downloaded.
     * @param {jQuery.Promise} fieldValuesRetrieved - Promise that is resolved with form parameters obtained from rest resource. The parameters are needed to ensure that the issue created will satisfy the actual JQL entered by user.
     */
    return function (deferred, fieldValuesRetrieved)  {
        if (!_.isObject(fieldValuesRetrieved)) {
            throw new Error("You must provide promise that will resolve with fieldValues to create this form. Use standard create issue form if you have none.");
        }

        var JiraFormsCreateIssueModel = JIRA.Forms.CreateIssueModel;
        var JiraFormsConfigurableCreateIssueForm = JIRA.Forms.ConfigurableCreateIssueForm;
        var JiraFormsUnconfigurableCreateIssueForm = JIRA.Forms.UnconfigurableCreateIssueForm;
        var JiraFormsCreateIssueError = JIRA.Forms.CreateIssueError;
        var applyErrorsToForm = JIRA.applyErrorsToForm;
        var applyErrorMessageToForm = JIRA.applyErrorMessageToForm;

        var data;
        fieldValuesRetrieved.done(function whenInitializedAndHasFields(responseData) {
            data = responseData;

            data.prefilledFields = data.prefilledFields || {};
            data.errors = data.errors || {};
            data.errorMessages = data.errorMessages || {};
        }).fail(function whenFailedInitialization() {
            data = {
                prefilledFields: {},
                errors: {},
                errorMessages: {}
            };
        });

        // Copied from Quick Edit plugin to override the title
        var CustomFormContainer = CustomFormContainerFactory.build();
        var form = new CustomFormContainer(function containerInitFunction() {
            var containerOptions = {
                title: AJS.I18n.getText('admin.issue.operations.create'),
                id: "prefillable-create-issue-dialog",
                globalEventNamespaces: ["PrefillableCreateIssueDialog"],
                issueType: undefined // we will set that later when context is resolved
            };

            // model that gets fields and sets user preferences
            var model = new JiraFormsCreateIssueModel({
                projectId: undefined, // we will set that later when context is resolved
                issueType: undefined // we will set that later when context is resolved
            });

            var configurableForm = new JiraFormsConfigurableCreateIssueForm({
                model: model,
                title: containerOptions.title,
                globalEventNamespaces: containerOptions.globalEventNamespaces
            });

            var unconfigurableForm = new JiraFormsUnconfigurableCreateIssueForm({
                model: model,
                title: containerOptions.title,
                globalEventNamespaces: containerOptions.globalEventNamespaces
            });

            return {
                globalEventNamespaces: containerOptions.globalEventNamespaces,
                model: model,
                errorHandler: new JiraFormsCreateIssueError(),
                configurableForm: configurableForm,
                unconfigurableForm: unconfigurableForm,
                // when this variable is used it will be already defined because
                // this function isn't called until AFTER fieldValuesRetrieved is resolved
                prefilledValues: data.prefilledFields
            };
        });

        var dialog = form.asDialog({
            windowTitle: AJS.I18n.getText('admin.issue.operations.create'),
            id: "prefillable-create-issue-dialog"
        });

        // we can modify dialog's options after creation, specifically the
        // delayShowUntil option, as it is used in the show method for the first time
        dialog.options.delayShowUntil = function delayShowUntilFieldValuesRetrieved() {
            return fieldValuesRetrieved;
        };

        form.asDialog = function asDialog() {
            return form.dialog;
        };

        dialog.onContentReady(function formInitialized() {
            if (_.keys(data.errors).length) {
                // add global error message indicating that there are field errors            }
                data.errorMessages.unshift(AJS.I18n.getText("iic.dialog.errors"));
                // add per-field error messages
                applyErrorsToForm(form.$element, data.errors);
            }
            if (data.errorMessages.length) {
                // add global error message, join multiple messsages with <br> if needed
                applyErrorMessageToForm(form.$element, data.errorMessages.join("<br/>"));
            }
        });

        deferred.resolve(form);
    };
});
