define('jira/inline-issue-create/views/widget', [
    'jquery',
    'underscore',
    'jira/inline-issue-create/lib/marionette',
    'jira/inline-issue-create/util'
], function (
    $,
    _,
    Marionette,
    util
) {
    "use strict";
    var ERROR_ANIMATION_LENGTH = 500;

    var WidgetView = Marionette.LayoutView.extend({
        template: JIRA.Templates.InlineIssueCreate.widget,

        regions: {
            issueTypeSelector: ".iic-widget__issue-type-selector-container",
            description: ".iic-widget__description-container"
        },
        ui: {
            summary: '.iic-widget__summary',
            more: '.iic-widget__more',
            cancel: '.iic-widget__cancel',
            form: '.iic-widget__form',
            withTitle: '[title]'
        },
        events: {
            "input @ui.summary": "onSummaryChange",
            "change @ui.summary": "onSummaryChange",
            "keyup @ui.summary": function(e) {
                var value = this.ui.summary.val();
                if (value.length > 255) {
                    this.ui.summary.val(value.substr(0, 255));
                }
                this.onSummaryChange(e);
            },
            "keydown @ui.summary": function onSummaryKeyboard(e) {
                var value = this.ui.summary.val();

                if (e.keyCode === util.keyCode.ENTER) {
                    e.preventDefault();
                    if (value !== "" && value.length <= 255) {
                        if (this.model.getLockState() === this.model.constants.STATE_LOCK_UNLOCKED) {
                            this.ui.summary.blur();
                            this.trigger("createIssue");
                        } else {
                            // TODO unhack this, should reference the appropriate view, probably raise an event to which the view would react
                            util.addClassAndRemoveOnAnimationEnd(this.$(".issue-type-selector"), "iic-error", ERROR_ANIMATION_LENGTH);
                        }
                    } else {
                        util.addClassAndRemoveOnAnimationEnd(this.ui.summary, "iic-error", ERROR_ANIMATION_LENGTH);
                    }
                } else if (e.keyCode === util.keyCode.ESC) {
                    this.trigger("deactivate", {
                        reason: "keypress"
                    });
                } else if (value.length > 255) {
                    util.addClassAndRemoveOnAnimationEnd(this.ui.summary, "iic-error", ERROR_ANIMATION_LENGTH);
                }
            },
            "click @ui.cancel": function onCancelClick(e) {
                e.preventDefault();
                if (this.model.getLockState() !== this.model.constants.STATE_LOCK_SENDING) {
                    this.trigger("deactivate", {
                        reason: "cancel"
                    });
                }

            },
            "click @ui.more": function onMoreClick(e) {
                e.preventDefault();
                if (this.model.getLockState() !== this.model.constants.STATE_LOCK_SENDING) {
                    this.trigger("openFullIssueDialog");
                }
            }
        },
        modelEvents: {
            "change:summary": function onSummaryChange() {
                this.ui.summary.val(this.model.getSummary());
                this.adjustSummarySize();
            },
            "change:state": function onStateChange(model, state) {
                if (state === this.model.constants.STATE_ACTIVE) {
                    this.activate();
                } else if (state === this.model.constants.STATE_CLOSED) {
                    this.deactivate();
                }
            },
            "change:lockState": function onStateLockChange(model, state) {
                if (this.ui.summary.prop) {
                    if (state === this.model.constants.STATE_LOCK_SENDING) {
                        this.ui.summary.prop("disabled", true);
                    } else {
                        this.ui.summary.prop("disabled", false);
                    }
                }
            }
        },
        initialize: function initialize(options) {
            this.cssClass = options.cssClass || "";
            this.listenTo(this.issueTypeSelector, "show", function () {
                this.listenTo(this.issueTypeSelector.currentView, "issueTypeSelected", function () {
                    this.focusSummary();
                });
            });

            _.bindAll(this, 'adjustSummarySize', "onBodyClick");
        },
        onClose: function onClose() {
            $(document).off("click.iic", this.onBodyClick);
        },
        onBodyClick: function onBodyClick(e) {
            // check if target is not the clicked element and hasn't it inside
            if (!this.$el.is(e.target) && this.$el.has(e.target).length === 0) {
                if (this.ui.summary.val() === "") {
                    this.trigger("deactivate", {
                        reason: "cancel"
                    });
                } else if (this.ui.summary.is(":focus")) {
                    // When clicking on issue in agile, even though cursor isn't in the textarea anymore,
                    // blur event isn't fired properly and technically textarea is still focused.
                    // This is why we have to blur it manually.
                    this.ui.summary.blur();
                }
            }
        },
        serializeData: function serializeData() {
            return {
                summary: this.model.getSummary(),
                cssClass: this.cssClass
            };
        },
        onRender: function onRender() {
            this.unwrapTemplate();
            this.ui.withTitle.tooltip({aria: true});

            var that = this;
            this.ui.summary.on('change cut paste drop keydown input',  function deferedAdjustSummarySize() {
                _.defer(that.adjustSummarySize);
            });

            // JAG renders sprint content detached from the DOM so we need to defer this until after it is attached
            _.defer(that.adjustSummarySize);

            this.ui.summary.placeholder();
            if (this.model.getState() === this.model.constants.STATE_ACTIVE) {
                this.activate();
            } else {
                this.deactivate();
            }
        },
        adjustSummarySize: function adjustSummarySize() {
            this.ui.summary.css('height', 'auto');
            var outerHeight = this.ui.summary.outerHeight();
            var scrollHeight = Math.max(outerHeight + this.ui.summary.prop('scrollHeight') - this.ui.summary.prop('clientHeight'), outerHeight);
            if (scrollHeight) {
                this.ui.summary.height(scrollHeight+'px');
            }
        },
        reset: function reset() {
            this.ui.summary.attr("style", "");
        },
        hide: function hide() {
            this.$el.addClass("iic-widget_hidden");
            $(document).off("click.iic", this.onBodyClick);
        },
        show: function show() {
            this.$el.removeClass("iic-widget_hidden");
            this.adjustSummarySize();

            // Add event handler to check if we can close the widget when users clicks outside of it.
            // We need to defer this because event bubbling would cause this to be immediately
            // executed when user clicks on the trigger to show the widget
            var instance = this;
            _.defer(function() {
                $(document).off("click.iic", instance.onBodyClick);
                $(document).on("click.iic", instance.onBodyClick);
            });
        },
        focusSummary: function focusSummary() {
            this.ui.summary.focus();
        },
        activate: function activate() {
            this.show();
            this.trigger("activated");
        },
        focus: function focus() {
            this.focusSummary();
            this.trigger("focused");
        },
        deactivate: function deactivate() {
            this.hide();
            this.trigger("deactivated");
        },
        onSummaryChange: function onSummaryChange(e) {
            this.trigger("summaryChanged", this.ui.summary.val());
        }
    });

    return WidgetView;
});