define('jira/inline-issue-create/widget', [
    'jquery',
    'underscore',
    'jira/util/logger',
    'jira/inline-issue-create/lib/marionette',
    'jira/inline-issue-create/util',
    'jira/inline-issue-create/entities/widget',
    'jira/inline-issue-create/entities/issue-types',
    'jira/inline-issue-create/entities/projects',
    'jira/inline-issue-create/entities/inline-issue',
    'jira/inline-issue-create/views/widget',
    'jira/inline-issue-create/views/issue-type-dropdown',
    'jira/inline-issue-create/views/description',
    'jira/inline-issue-create/prefillable-create-issue-dialog',
    'jira/inline-issue-create/analytics'
], function (
    $,
    _,
    logger,
    Marionette,
    Util,
    WidgetModel,
    IssueTypeCollection,
    ProjectsCollection,
    InlineIssue,
    WidgetLayout,
    IssueTypeDropdownView,
    DescriptionView,
    PrefillableCreateIssueDialog,
    Analytics
) {
    "use strict";

    /**
     * This function takes options hash and creates a options.fieldName field in options.context object.
     * If options.parameter is a function it is used as a constructor, if it is an object it is taken as-is.
     * If it is empty, the new object is created using default constructor.
     * @param options {parameter, context, fieldName, defaultOptions, defaultConstructor}
     */
    function createFieldFromParam(options) {
        if (_.isFunction(options.parameter)) {
            options.context[options.fieldName] = new options.parameter(options.defaultOptions);
        } else if (_.isObject(options.parameter)) {
            options.context[options.fieldName] = options.parameter;
        } else if (undefined === options.parameter) {
            options.context[options.fieldName] = new options.defaultConstructor(options.defaultOptions);
        }
    }

    var IicWidget = Marionette.Controller.extend({
        initialize: function initialize(options) {
            this._initializeComponents(options);

            if (options.jqls) {
                this.setCurrentJQLs(options.jqls);
            }
            if (options.container) {
                this.attach(options.container);
            }
            this._initializeEvents(options);
        },

        _initializeComponents: function _initializeComponents(options) {
            // createFieldFromParam wasn't used here as the constructor for collection takes 2 parameters
            if (_.isFunction(options.issueTypes)) {
                this.issueTypes = new options.issueTypes(undefined, {
                    jqls: options.jqls
                });
            } else if (_.isObject(options.issueTypes)) {
                this.issueTypes = options.issueTypes;
            } else {
                this.issueTypes = new IssueTypeCollection(undefined, {
                    jqls: options.jqls
                });
            }

            // createFieldFromParam wasn't used here as the constructor for collection takes 2 parameters
            if (_.isFunction(options.projects)) {
                this.projects = new options.projects(undefined, {
                    jqls: options.jqls
                });
            } else if (_.isObject(options.projects)) {
                this.projects = options.projects;
            } else {
                this.projects = new ProjectsCollection(undefined, {
                    jqls: options.jqls
                });
            }

            createFieldFromParam({
                context: this,
                fieldName: "model",
                defaultOptions: {
                    issueTypes: this.issueTypes
                },
                parameter: options.model,
                defaultConstructor: WidgetModel
            });

            createFieldFromParam({
                context: this,
                fieldName: "dropdownView",
                defaultOptions: {
                    model: this.model,
                    collection: this.issueTypes,
                    widgetId: this.model.cid
                },
                parameter: options.dropdownView,
                defaultConstructor: IssueTypeDropdownView
            });

            createFieldFromParam({
                context: this,
                fieldName: "descriptionView",
                defaultOptions: {
                    model: this.model,
                    widgetId: this.model.cid
                },
                parameter: options.descriptionView,
                defaultConstructor: DescriptionView
            });

            this.view = new WidgetLayout({
                model: this.model,
                cssClass: options.cssClass
            });
            this.analytics = new Analytics({widget: this});
            this._currentInlineIssue = undefined;
        },
        _initializeEvents: function _initializeEvents(options) {
            this.listenTo(this.view, "summaryChanged", function onSummaryChanged(newSummary) {
                this.model.setSummary(newSummary);
            });
            this.listenTo(this.view, "createIssue", this.createIssue);
            this.listenTo(this.view, "deactivate", this.deactivate);
            this.listenTo(this.view, "openFullIssueDialog", function onOpenFullIssueDialog() {
                this.trigger("openCreateIssueDialog", {reason: "manual"});
                this.openCreateIssueDialog();
            });
            this.listenTo(this.model, 'change:state', this.onStateChange);
            this.listenTo(this.model, 'change:lockState', this.onLockStateChange);
            this.listenTo(this.issueTypes, 'reset', this.checkWidgetLocked);
            this.listenTo(this.projects, 'reset', this.checkWidgetLocked);

            this.on("issueCreated", function resetSummaryOnIssueCreated() {
                this.model.setSummary("");
            });
            //kick off the world
            this.checkWidgetLocked();
        },
        render: function render() {
            //detaching regions as render would destroy all of them
            //we want to re-attach just after we'll rerender the view
            this.view.issueTypeSelector.attachView(undefined);
            this.view.description.attachView(undefined);
            this.view.render();

            this.view.issueTypeSelector.show(this.dropdownView);
            this.view.description.show(this.descriptionView);
            this.listenTo(this.dropdownView, "issueTypeSelected", function (selectedIssueTypeModel) {
                this.model.setSelectedIssueType(selectedIssueTypeModel);
            });
        },
        onStateChange: function (model, state) {
            if (state === this.model.constants.STATE_ACTIVE) {
                this.trigger('activated');
            } else if (state === this.model.constants.STATE_CLOSED) {
                this.trigger('deactivated');
            }
        },
        onLockStateChange: function onLockStateChange(model, lockState) {
            if (lockState === this.model.constants.STATE_LOCK_PERM_INACTIVE) {
                if (this.model.getState() === this.model.constants.STATE_ACTIVE) {
                    this.deactivate();
                }
            }
        },
        activate: function activate() {
            if (this.model.getLockState() !== this.model.constants.STATE_LOCK_PERM_INACTIVE) {
                this.model.changeState(this.model.constants.STATE_ACTIVE);
                return true;
            } else {
                return false;
            }
        },
        deactivate: function deactivate() {
            this.model.changeState(this.model.constants.STATE_CLOSED);
        },
        checkWidgetLocked: function checkWidgetLocked() {
            //are collections loading?
            if (this.issueTypes.isFetching() || !this.issueTypes.isInitialized()) {
                return this.model.setLockState(this.model.constants.STATE_LOCK_TEMP_INACTIVE);
            }
            if (this.projects.isFetching() || !this.projects.isInitialized()) {
                return this.model.setLockState(this.model.constants.STATE_LOCK_TEMP_INACTIVE);
            }
            //or maybe we are actually saving new issue
            if (this._currentInlineIssue && this._currentInlineIssue.isSaving()) {
                return this.model.setLockState(this.model.constants.STATE_LOCK_SENDING);
            }
            //are collections empty?
            if (!this.issueTypes.length) {
                return this.model.setLockState(this.model.constants.STATE_LOCK_PERM_INACTIVE);
            }
            if (this.projects.length !== 1) {
                return this.model.setLockState(this.model.constants.STATE_LOCK_PERM_INACTIVE);
            }
            //okay! we are good!
            this.model.setLockState(this.model.constants.STATE_LOCK_UNLOCKED);
        },
        focus: function focus() {
            this.view.focus();
        },
        reset: function reset() {
            this.model.setSummary("");
            this.view.reset();
        },
        attach: function attach(container) {
            var $widgetContainer = $(container);
            $widgetContainer.append(this.view.el);
        },
        setCurrentJQLs: function setCurrentJQLs(jqls) {
            this.jqls = jqls;
        },
        setSummary: function setSummary(newSummary) {
            this.model.setSummary(newSummary);
        },
        getSummary: function getSummary() {
            return this.model.getSummary();
        },
        setSelectedIssueTypeIfAvailable: function setSelectedIssueTypeIfAvailable(newIssueType) {
            this.model.setSelectedIssueTypeIfAvailable(newIssueType);
        },
        getSelectedIssueType: function getSelectedIssueType() {
            return this.model.getSelectedIssueType();
        },
        setOverrides: function setOverrides(overrides) {
            this.overrides = overrides;
        },
        createIssue: function createIssue() {
            var instance = this;
            //we need to _.clone objects allowing beforeIssueCreated to modify those data
            var data = {
                summary: this.model.getSummary(),
                contexts: this.jqls,
                issueTypeId: this.model.getSelectedIssueType().getId(),
                overrides: _.clone(this.overrides)
            };
            this.trigger("beforeIssueCreated", instance, data);

            //persist new issue
            this._currentInlineIssue = new InlineIssue(data);
            this._currentInlineIssue.save().done(function successInlineCreate(data) {
                instance.trigger("issueCreated", data.issue, data.prefilledFields, instance, {
                    source: "inline"
                });
                logger.trace("inline.create.issue.created");
            }).fail(function failureInlineCreate(response) {
                var dialogDataDeferred = new jQuery.Deferred();
                instance.openCreateIssueDialog(dialogDataDeferred);
                try {
                    var data = JSON.parse(response.responseText);
                    dialogDataDeferred.resolve({
                        errors: data.errors,
                        errorMessages: data.generalErrors,
                        prefilledFields: data.prefilledFields
                    });
                } catch (err) {
                    Util.error(instance, "Could not parse response: " + err, response);
                    var errorMessage = AJS.I18n.getText("iic.dialog.could.not.create.issue");
                    dialogDataDeferred.resolve({
                        errors: {},
                        errorMessages: [errorMessage],
                        prefilledFields: {}
                    });
                }


                instance.trigger("openCreateIssueDialog", {reason: "createFailed"});
            }).always(function checkStateAfterSending() {
                instance.checkWidgetLocked();
            });
            this.checkWidgetLocked();
        },
        openCreateIssueDialog: function openCreateIssueDialog(dialogDataDeferred) {
            var instance = this;
            var issueData = this.model.toIssueData();
            if (this.projects.size() === 1) {
                issueData.project = this.projects.at(0).toJSON();
            }
            var context = {
                summary: issueData.summary,
                contexts: this.jqls,
                issueTypeId: this.model.getSelectedIssueType() ? this.model.getSelectedIssueType() .getId() : undefined,
                overrides: _.clone(this.overrides)
            };
            this.trigger("beforeIssueCreated", instance, context);

            if (!dialogDataDeferred) {
                dialogDataDeferred = this._resolveInlineContext(context, instance);
            }

            var promiseForm = PrefillableCreateIssueDialog.createPrefillableCreateIssueForm(dialogDataDeferred.promise(), true);
            return promiseForm.pipe(function (form) {
                var dialog = form.asDialog();
                function triggerIssueCreated(e, issue) {
                    instance.trigger("issueCreated", issue, issue.fields, instance, {
                        source: "dialog"
                    });
                    logger.trace("inline.create.issue.created");
                }
                form.bind("initialized", function () {
                    this.unconfigurableForm.bind("issueCreated", triggerIssueCreated);
                    this.configurableForm.bind("issueCreated", triggerIssueCreated);
                });

                dialog.show();
                return dialog;
            });
        },
        _resolveInlineContext: function(context, instance) {
            var fieldValuesRetrieved = new jQuery.Deferred();
            var fieldValuesResolver = $.ajax(AJS.contextPath() + "/rest/inline-create/1.0/context/resolve", {
                data: JSON.stringify(context),
                contentType: 'application/json',
                type: 'POST'
            });
            fieldValuesResolver.done(function whenInitializedAndHasFields(data) {
                fieldValuesRetrieved.resolve({
                    prefilledFields: data.createIssueInput
                });
            }).fail(function whenFailedInitializingFields() {
                // very basic error recovery
                fieldValuesRetrieved.resolve({
                    prefilledFields: {
                        summary: context.summary ? context.summary : "",
                        issuetype: instance.model.getSelectedIssueType() ?
                            instance.model.getSelectedIssueType().getId() : undefined
                    }
                });
            });
            return fieldValuesRetrieved;
        }
    });

    return IicWidget;
});
