AJS.test.require(["com.atlassian.plugins.helptips.jira-help-tips:help-tip"], function(){

    var $ = require("jquery");
    var Manager = require("jira-help-tips/feature/help-tip-manager");
    var HelpTip = require("jira-help-tips/feature/help-tip");

    module('jira-help-tips/feature/help-tip');

    test('can create new tips', function() {
        var tip = new HelpTip();
        equal(typeof tip, "object", "should be an object");
    });

    test('get a unique client id (cid)', function() {
        var ids = [];
        for(var i=0; i<20; i++) {
            ids.push((new HelpTip()).cid);
        }
        equal($.unique(ids).length, ids.length, "should be as many unique ids as tips created");
    });

    test('can be programmatically dismissed', function() {
        var tip = new HelpTip({body:"whee"});
        ok(!tip.isDismissed(), "has not been dismissed yet");
        tip.dismiss();
        ok(tip.isDismissed(), "should be dismissed");
    });

    test('can bind dismissal of tip to other elements', function() {
        var tip = new HelpTip({body:"whee"});
        var anchor = $("<a>click me to dismiss</a>");
        anchor.click(function() { tip.dismiss() });
        anchor.trigger('click');
        ok(tip.isDismissed(), "should be dismissed after clicking the anchor link");
    });

    // TODO: This isn't a pure unit test. Split out the components.
    test('tips with the same id have a linked dismissed value', function() {
        this.sandbox.stub(Manager, "sync");

        var tip1 = new HelpTip({id:"testtip",body:"firsttip"});
        var tip2 = new HelpTip({id:"testtip",body:"secondtip"});
        tip1.dismiss();
        ok(tip2.isDismissed(), "tip2 is considered dismissed because tip1 was.");
    });

    test('tips with no id are considered different', function() {
        var tip1 = new HelpTip({body:"firsttip"});
        var tip2 = new HelpTip({body:"secondtip"});
        tip1.dismiss();
        ok(!tip2.isDismissed(), "tip2 is different to tip1 as neither have the same id.");
    });

    module('anchored tip', {
        setup: function() {
            $.fx.off = true;

            this.context = AJS.test.mockableModuleContext();
            this.sandbox = sinon.sandbox.create({
                useFakeTimers: true,
                useFakeServer: true
            });
            this.clock = this.sandbox.clock;

            this.fakeSync = this.sandbox.stub(Manager, "sync");
            this.fakeShow = this.sandbox.stub(Manager, "show", function(showTip) { showTip(); });

            this.container = $("<div>").attr("id", "content").appendTo("#qunit-fixture");
            this.box = $("<div></div>").appendTo("#qunit-fixture");
            this.anchor = $("<a>test</a>").appendTo(this.box);

            this.tip = new HelpTip({body:"test " + new Date(),anchor:this.anchor});
        }, teardown: function() {
            this.container.remove();
            this.box.remove();
            this.tip = null;
            this.sandbox.restore();
            $.fx.off = false;
        },
        setupSequences: function() {
            Manager.loaded = $.Deferred();
            Manager.showSequences();
            Manager.loaded.resolve();
        },
        teardownSequences: function() {
            Manager.dismissedTipIds = [];
            Manager.loaded = $.Deferred();
            Manager.clearSequences();
            Manager.loaded.resolve();
        }
    });

    test('can be anchored to an element', function() {
        this.tip.show();
        this.clock.tick(0);
        ok(this.tip.isVisible(), "tip should be visible");
    });

    test('cannot be dismissed by clicking outside the tip', function() {
        this.tip.show();
        this.clock.tick(0);
        $(document.body).trigger('click');
        ok(this.tip.isVisible(), "tip should still be visible");
        ok(!this.tip.isDismissed(), "not dismissed yet");
    });

    test('tip disappears when dismissed', function() {
        this.tip.show();
        this.clock.tick(0);
        this.tip.dismiss();
        this.clock.tick(0);
        ok(!this.tip.isVisible(), "should no longer be visible");
    });

    test('can click a close button to dismiss', function() {
        this.tip.show();
        this.clock.tick(0);
        var button = $(".helptip-close", this.tip.view.$el); // TODO: probably a better way to get this.
        ok(button.length, "there should be a button on the anchored tip to dismiss it");
        ok(!this.tip.isDismissed(), "is not yet dismissed");
        button.trigger('click');
        this.clock.tick(0);
        ok(this.tip.isDismissed(), "should be dismissed after clicking the close button");
    });

    test('once dismissed, cannot be re-opened', function() {
        this.tip.show();
        this.clock.tick(0);
        this.tip.dismiss();
        this.clock.tick(0);
        this.tip.show();
        ok(!this.tip.isVisible(), "should not be visible once closed");
    });

    test('tip can be hidden', function() {
        this.tip.show();
        this.clock.tick(0);
        this.tip.hide();
        this.clock.tick(0);
        ok(!this.tip.isVisible(), "should be hidden");
    });

    test('can show again after hide', function() {
        this.tip.show();
        this.clock.tick(0);
        this.tip.hide();
        this.clock.tick(0);
        this.tip.show();
        this.clock.tick(0);
        ok(this.tip.isVisible(), "should be visible after hiding");
    });

    test('refresh on a tip with hidden anchor hides the tip', function() {
        this.tip.show();
        this.clock.tick(0);
        ok(this.tip.isVisible(), "should be visible");
        this.anchor.hide();
        this.clock.tick(0);
        ok(this.tip.isVisible(), "should be visible after anchor is hidden and refresh not called");
        this.tip.refresh();
        this.clock.tick(0);
        ok(!this.tip.isVisible(), "should not be visible after anchor is hidden and refresh called");
    });

    test('refresh on a tip that isnt shown yet shows the tip', function() {
        ok(!this.tip.isVisible(), "should not be visible");
        this.tip.refresh();
        this.clock.tick(0);
        ok(this.tip.isVisible(), "should be visible after refresh");
    });

    test('test inline dialog options is passed from helptip constructors', function() {
        expect(5);
        var inlineDialogStub = this.sandbox.stub().returns({hide: $.noop});
        this.context.mock("aui/inline-dialog", inlineDialogStub);
        var HelpTip = this.context.require("jira-help-tips/feature/help-tip");

        var expectedInlineDialogOpts = {
            offsetX: 999,
            offsetY: -999,
            arrowOffsetX: 9001,
            container: "fixture",
            initCallback: this.sandbox.spy()
        };

        new HelpTip({
            body:"test " + new Date(),
            anchor:this.anchor,
            callbacks: {init: $.noop},
            inlineDialogOpts: expectedInlineDialogOpts});

        var actualInlineDialogOpts = inlineDialogStub.args[0][3];
        _.each(expectedInlineDialogOpts, function(expectedValue, expectedKey) {
            ok(actualInlineDialogOpts[expectedKey] === expectedValue,
                "options passed in from constructor should replace the default inline dialog options from helptip");
        });
    });

    test('test helptips to be shown in sequence for next button', function() {
        $("#qunit-fixture").append("<a class='anchorOne'></a><a class='anchorTwo'></a>");

        var tipOne =  new HelpTip({id:"seq1",body:"I am a test 1.", isSequence: true, anchor: ".anchorOne"});
        var tipTwo =  new HelpTip({id:"seq2",body:"I am a test 2.", isSequence: true, anchor: ".anchorTwo"});

        this.setupSequences();

        var next = $(".helptip-next", tipOne.view.$el);
        equal($(".helptip-sequence-paging", tipOne.view.$el).text(), "1/2",
            "Helptips include paging when shown in sequence");
        equal(next.length, 1, "Helptips include next button when shown in sequence");
        next.click();

        equal($(".helptip-sequence-paging", tipTwo.view.$el).text(), "2/2",
            "Helptips include paging when shown in sequence");
        equal($(".helptip-close", tipTwo.view.$el).length, 1,
            "Helptips include close button when it is the last one");
        equal($(".helptip-next", tipTwo.view.$el).length, 0,
            "Helptips does not include next button when it is the last one");

        this.teardownSequences();
    });

    test('test can move to the next help tip in sequence programatically', function() {
        $("#qunit-fixture").append("<a class='anchorOne'></a><a class='anchorTwo'></a>");

        var tipOne =  new HelpTip({id:"seq1",body:"I am a test 1.", isSequence: true, anchor: ".anchorOne"});
        var tipTwo =  new HelpTip({id:"seq2",body:"I am a test 2.", isSequence: true, anchor: ".anchorTwo"});

        equal($('.helptip-sequence-paging').length, 0);

        this.setupSequences();

        equal($('.helptip-sequence-paging').length, 1);
        equal($('.helptip-sequence-paging').last().text(), "1/2");

        tipOne.showNextHelpTipInSequence();
        equal($('.helptip-sequence-paging').length, 2);
        equal($('.helptip-sequence-paging').last().text(),"2/2");

        tipTwo.showNextHelpTipInSequence();
        equal($('.helptip-sequence-paging').length, 2);


        this.teardownSequences();
    });

    test('test never show paging info for non-sequence help tip', function() {
        $("#qunit-fixture").append("<a class='anchorOne'></a><a class='anchorTwo'></a><a class='anchorThree'></a>");

        var tipOne =  new HelpTip({id:"seq1",body:"I am a test 1.", isSequence: true, anchor: ".anchorOne"});
        var tipTwo =  new HelpTip({id:"seq2",body:"I am a test 2.", isSequence: true, anchor: ".anchorTwo"});
        var tipThree =  new HelpTip({id:"non-seq",body:"I am a test 3.", isSequence: false, anchor: ".anchorThree"});

        equal($('.helptip-sequence-paging').length, 0);

        this.setupSequences();

        tipThree.show();

        equal($('.helptip-sequence-paging').length, 1);
        equal($('.helptip-sequence-paging').last().text(), "1/2");

        tipOne.showNextHelpTipInSequence();
        equal($('.helptip-sequence-paging').length, 2);
        equal($('.helptip-sequence-paging').last().text(),"2/2");

        tipTwo.showNextHelpTipInSequence();
        equal($('.helptip-sequence-paging').length, 2);

        this.teardownSequences();
    });

    test('test close button will be enabled on last tip in sequence if last tip is filtered out as dismissed', function() {
        $("#qunit-fixture").append("<a class='anchorOne'></a><a class='anchorTwo'></a>");

        var tipOne =  new HelpTip({id:"seq1",body:"I am a test 1.", isSequence: true, showCloseButton:false, anchor: ".anchorOne"});
        var tipTwo =  new HelpTip({id:"seq2",body:"I am a test 2.", isSequence: true, showCloseButton:true, anchor: ".anchorTwo"});

        Manager.dismissedTipIds = ["seq2"];

        equal($('.helptip-sequence-paging').length, 0);

        this.setupSequences();

        equal($('.helptip-sequence-paging').length, 0);

        equal($(".helptip-close", tipOne.view.$el).length, 1,
                "Helptips include close button when it is the last one");

        this.teardownSequences();
    });

    test('test helptips to be shown in sequence for close button', function() {
        $("#qunit-fixture").append("<a class='anchorOne'></a><a class='anchorTwo'></a>");

        var tipOne =  new HelpTip({id:"seq1",body:"I am a test 1.", isSequence: true, anchor: ".anchorOne"});
        var tipTwo =  new HelpTip({id:"seq2",body:"I am a test 2.", isSequence: true, anchor: ".anchorTwo"});

        this.setupSequences();

        var close = $(".helptip-close", tipOne.view.$el);
        close.click();

        equal($(".helptip-body", tipTwo.view.$el).length, 0,
            "Closing the first tip shouldn't open the second tip");

        this.teardownSequences();
    });

    test('test helptips can choose to not have a close button', function() {
        $("#qunit-fixture").append("<a class='anchorOne'></a>");

        var helpTip =  new HelpTip({id:"helptip", body:"I am a test.", showCloseButton: false, anchor: ".anchorOne"});
        helpTip.show();

        equal($(".helptip-close", helpTip.view.$el).length, 0, "The helptip should not have a close button");
    });

    test('test HelptipManager for tips to be shown in sequence', function() {
        $("#qunit-fixture").append("<a class='anchor'></a>");

        var tipOne =  new HelpTip({id:"seq1",body:"I am a test 1.", isSequence: true, anchor: ".anchor"});
        var tipTwo =  new HelpTip({id:"seq2",body:"I am a test 2.", isSequence: true, anchor: ".anchor"});

        equal(Manager.sequences.length, 2,
            "Helptips to be shown in sequence is added to the list");

        this.setupSequences();

        equal(Manager.sequences.length, 2,
            "Helptips to be shown in sequence is added to the list");
        equal($(".helptip-sequence-paging", tipOne.view.$el).text(), "1/2",
            "Helptips include paging when shown in sequence");
        equal($(".helptip-close", tipOne.view.$el).length, 1,
            "Helptips include close button when shown in sequence");
        equal($(".helptip-next", tipOne.view.$el).length, 1,
            "Helptips include next button when shown in sequence");

        Manager.dismissedTipIds.push("seq2");
        this.setupSequences();

        equal(Manager.sequences.length, 1,
            "Helptips that have already been dismissed will not show");
        equal($(".helptip-sequence-paging", tipOne.view.$el).length, 0,
            "Helptips do not include paging when it is the only one");
        equal($(".helptip-close", tipOne.view.$el).length, 1,
            "Helptips include close button when it is the only one");
        equal($(".helptip-next", tipOne.view.$el).length, 0,
            "Helptips does not include next button when it is the only one");

        this.teardownSequences();
    });

    module('anchored tip with anchor selector', {
        setup: function() {
            $.fx.off = true;
            this.sandbox = sinon.sandbox.create({
                useFakeTimers: true,
                useFakeServer: true
            });
            this.clock = this.sandbox.clock;

            this.container = $("<div>").attr("id", "content").appendTo("#qunit-fixture");
            this.box = $("<div></div>").appendTo("#qunit-fixture");
            this.anchorSelectorClass = 'anchor-selector-class';
            this.anchorSelector = '.'+this.anchorSelectorClass;
            this.anchor1 = $("<a>test</a>").appendTo(this.box);
            this.anchor2 = $("<a>test</a>").appendTo(this.box);
            this.anchor1.addClass(this.anchorSelectorClass);

            this.tip = new HelpTip({body:"test " + new Date(),anchor:this.anchorSelector});
        },
        teardown: function() {
            this.tip = null;
            this.sandbox.restore();
            $.fx.off = false;
        }
    });

    test('visible when matching anchor is present after refresh', function() {
        this.tip.show();
        this.clock.tick(0);
        ok(this.tip.isVisible(), "visible because the anchor selector matches");
        this.tip.refresh();
        this.clock.tick(0);
        ok(this.tip.isVisible(), "visible because the anchor selector matches");
    });

    test('not visible when matching anchor is absent', function() {
        this.tip.show();
        this.clock.tick(0);
        ok(this.tip.isVisible(), "visible because the anchor selector matches");
        this.anchor1.removeClass(this.anchorSelectorClass);
        this.tip.refresh();
        this.clock.tick(0);
        ok(!this.tip.isVisible(), "not visible because there are no anchors matching the selector");
    });

    test('refresh on a change of anchor', function() {
        this.anchor1.addClass(this.anchorSelectorClass);
        this.tip.show();
        this.clock.tick(0);
        ok(this.tip.isVisible(), "visible because the anchor selector matches");
        this.anchor1.removeClass(this.anchorSelectorClass);
        this.anchor2.addClass(this.anchorSelectorClass);
        this.tip.refresh();
        this.clock.tick(0);
        ok(this.tip.isVisible(), "visible because the second anchor matches");
    });

    module('persistence', {
        setup: function() {
            var requests = this.requests = [];
            this.xhr = sinon.useFakeXMLHttpRequest();
            this.xhr.onCreate = function(xhr) {
                requests.push(xhr);
            }
        }, teardown: function() {
            this.xhr.restore();
        }
    });

    test('sends nothing to the server if no ID was set', function() {
        var tip = new HelpTip({body:"I have no ID-ah!"});
        tip.dismiss();
        equal(this.requests.length, 0, "no request was made to the server.");
    });

    test('POSTs the dismissal of the tip to the server', function() {
        var tip = new HelpTip({id:"testtip2",body:"I am a test."});
        tip.dismiss();
        equal(this.requests.length, 1, "A request was sent to the server");
        equal(this.requests[0].requestBody, JSON.stringify({id:"testtip2"}));
    });

});
