define('pdf-view',
  [
    'ajs',
    'backbone',
    'jquery',
    'pdf-viewer/viewer-properties',
    'pdf-viewer/pdf-viewer',
    'pdf-viewer/page-view-scroll',
    'pdf-viewer/presentation-mode',
    'pdf-viewer/pdf-link-service',
    'annotation-layer-builder',
    'BaseViewer',
    'template-store-singleton'
  ],
  function (
    AJS,
    Backbone,
    $,
    viewerProperties,
    PDFViewer,
    pageViewScroll,
    PresentationMode,
    PDFLinkService,
    AnnotationLayerBuilder,
    BaseViewer,
    templateStore
  ) {

    'use strict';

    var PDFJS = window.PDFJS;

    PDFJS.externalLinkTarget = PDFJS.LinkTarget.BLANK;

    var when = function (val) {
      if (val && val.then) {
        return val;
      }
      return new $.Deferred().resolve(val).promise();
    };

    var SCALE = {
      PAGE_WIDTH : 'page-width',
      PAGE_HEIGHT : 'page-height',
      PAGE_FIT: 'page-fit',
      AUTO: 'auto',
      PAGE_ACTUAL: 'page-actual',
      CUSTOM: 'custom'
    };

    /**
     * Create a new error from the error PDFJS throws when it fails.
     * @param  {Error} pdfjsErr PDFJS error object
     * @return {Error}
     */
    var createPdfError = function (pdfjsErr, model) {
      var err   = new Error(pdfjsErr.toString());
      var reason  = pdfjsErr.message.split(':')[0];
      switch (reason) {
        case 'PasswordException':
          err.title   = AJS.I18n.getText('cp.error.pdf.password.header');
          err.description = AJS.I18n.getText('cp.error.pdf.password.message');
          err.download  = true;
          break;
        case 'InvalidPDFException':
          err.title   = AJS.I18n.getText('cp.error.pdf.invalid.header');
          err.description = AJS.I18n.getText('cp.error.pdf.invalid.message');
          err.download  = true;
          break;
        case 'MissingPDFException':
        case 'UnexpectedResponseException':
          err.title   = AJS.I18n.getText('cp.error.pdf.missing.header');
          err.description = model.get('src');
          break;
        case 'UnknownErrorException':
        default:
          err.title   = AJS.I18n.getText('cp.error.pdf.default.header');
          err.description = model.get('src');
          break;
      }
      err.icon = 'cp-pdf-icon';
      return err;
    };

    var PDFView = BaseViewer.extend({

      id: 'cp-pdf-preview',

      tagName: 'div',

      initialize: function () {
        BaseViewer.prototype.initialize.apply(this, arguments);

        var linkService = this.linkService = new PDFLinkService();

        var viewer = this.viewer = new PDFViewer({
          container: document.getElementById('cp-pdf-preview'),
          viewer: document.getElementById('viewer'),
          linkService: linkService
        });

        linkService.setViewer(this.viewer);

        this.fileBody = document.getElementById('cp-file-body');

        this.scaleChangeListener = function scalechange (evt) {
          viewer.updateViewarea();
          this._fileViewer.getView().updatePaginationButtons();
        }.bind(this);

        $(window).on('scalechange.pdfPreviewView', this.scaleChangeListener);

        viewer.watchScroll(this.el, pageViewScroll, viewer.updateViewarea);
      },

      teardown: function () {
        BaseViewer.prototype.teardown.apply(this);
        $(window).off('scalechange.pdfPreviewView', this.scaleChangeListener);
        this._enableScroll();
        this.viewer.watchScrollEnd(this.fileBody);
        this.stopListening();

        this.viewer.close();
      },

      getBackground: function () {
        return this.$el.add('#viewer');
      },

      scrollCenter: function () {
        var $container     = $(this.viewer.el.container);
        var containerWidth   = $container.width();
        var $currentPage   = $(this.viewer.pages[this.viewer.page].el);
        var currentPageWidth = $currentPage.width();
        if (currentPageWidth < containerWidth) {
          return;
        }
        var offsetLeft = (currentPageWidth - containerWidth) / 2;
        $container.scrollLeft(offsetLeft);
      },

      zoomFit: function () {
        if (this.viewer.currentScaleValue === SCALE.PAGE_WIDTH) {
          this.viewer.setScale(this.viewer.recentScaleValue);
          this.viewer.currentScaleValue = this.viewer.recentScaleValue;
        } else {
          this.viewer.recentScaleValue = this.viewer.currentScaleValue;
          this.viewer.setScale(SCALE.PAGE_WIDTH);
          this.viewer.currentScaleValue = SCALE.PAGE_WIDTH;
        }
      },

      zoomIn: function () {
        this.viewer.zoomIn();
        this.scrollCenter();
      },

      zoomOut: function () {
        this.viewer.zoomOut();
        this.scrollCenter();
      },

      render: function () {
        this.$el.html(templateStore.get('PDFViewer.preview')());

        this.viewer.setContainer({
          container: this.el,
          viewer: this.$el.find('#viewer')[0],
          outerContainer: this.$el.find('#outerContainer')[0]
        });

        this._openViewer(this._fileViewer.getConfig().pdfTransportFactory);

        return this;
      },

      renderAnnotations: function (PinsView) {
        var annotations = this.model.get('annotations');
        var layerBuilder = new AnnotationLayerBuilder({
          annotations: annotations,
          fileViewer: this._fileViewer,
          PinsView: PinsView
        });

        this.viewer.addLayerBuilder(layerBuilder.create());

        var that = this;
        this.listenTo(annotations, 'add remove reset sync', function () {
          layerBuilder.updateAnnotations(annotations);
        });

        this.listenTo(annotations, 'selected', function (item) {
          if (item) {
            var pageNumber = item.attributes.pageNumber;
            if (pageNumber > that.viewer.pages.length) {
              pageNumber = that.viewer.pages.length;
            }
            // Jump to page, with zoomlevel 'Fit'
            if (that.viewer.pages && that.viewer.pages[pageNumber - 1]) {
              that.viewer.pages[pageNumber - 1].scrollIntoView([null, {name: 'Fit'}]);
            }
            // @TODO: Pins are no longer animated
          }
        });
      },

      _openViewer: function (pdfTransportFactory) {
        if (!window.ArrayBuffer) {
          var err = new Error();
          err.title = AJS.I18n.getText('cp.unsupported.browser.header');
          err.description = AJS.I18n.getText('cp.unsupported.browser.download.to.view');
          err.icon = 'cp-pdf-icon';
          this.trigger('viewerFail', err);
          return;
        }

        var createPdfTransportPromise = function () {
          return when(pdfTransportFactory && pdfTransportFactory(this._fileViewer.getCurrentFile()));
        }.bind(this);

        var viewerFailed = function (oldErr) {
          var err = createPdfError(oldErr, this.model);
          this.trigger('viewerFail', err);
        }.bind(this);

        var viewerSucceeded = function (pdfTransport) {

          var defaultScale = viewerProperties[
            PresentationMode.active ? 'DEFAULT_SCALE_PRESENTATION' : 'DEFAULT_SCALE'
          ];
          var passwordLayer = this._fileViewer._view.fileContentView.getLayerForName('password');
          this.viewer.openFile(this._previewSrc, defaultScale, undefined, passwordLayer, pdfTransport)
          .then(function () {
            this.trigger('viewerReady');
          }.bind(this), viewerFailed);
        }.bind(this);

        createPdfTransportPromise()
        .done(viewerSucceeded)
        .fail(viewerFailed);
      },

      goToPreviousPage: function () {
        this.viewer.goToPreviousPage();
      },

      goToNextPage: function () {
        this.viewer.goToNextPage();
      },

      hasPreviousPage: function () {
        return this.viewer.page > 1;
      },

      hasNextPage: function () {
        return this.viewer.page < this.viewer.pages.length;
      },

      _preventDefault: function (e) {
        e = e || window.event;
        if (e.preventDefault) {
          e.preventDefault();
        }
        e.returnValue = false;
      },

      _disableScroll: function () {
        window.addEventListener('DOMMouseScroll', this._preventDefault, false);
        window.onmousewheel = document.onmousewheel = this._preventDefault;
        $('#cp-pdf-preview').addClass('hide-scrollbar');
      },

      _enableScroll: function () {
        window.removeEventListener('DOMMouseScroll', this._preventDefault, false);
        window.onmousewheel = document.onmousewheel = null;
        $('#cp-pdf-preview').removeClass('hide-scrollbar');
      },

      _setPagePadding: function () {
        var pagePadding;
        if (PresentationMode.active) {
          pagePadding = {x: 0, y: 0};
        } else {
          pagePadding = {
            x: viewerProperties.SCROLLBAR_PADDING,
            y: viewerProperties.VERTICAL_PADDING
          };
          var arrowLayer = this._fileViewer.getView().fileContentView.getLayerForName('arrows');
          if (arrowLayer.showsArrow()) {
            pagePadding.x = viewerProperties.NAVIGATION_ARROW_PADDING;
          }
        }
        this.viewer.setPagePadding(pagePadding);
      },

      setupMode: function (mode, isModeChanged) {
        if (mode === 'PRESENTATION') {
          PresentationMode.active = true;
          this._disableScroll();
        } else {
          PresentationMode.active = false;
          this._enableScroll();
        }

        this._setPagePadding();

        if (isModeChanged) {
          this._scaleGraduallyToFitPage();
        }
      },

      _scaleGraduallyToFitPage: function () {
        // When browser change to fullscreen mode, the screen size is changed many times.
        // Here we scale 10 times every 100ms to make the page scaling to full screen smoothly
        var times = 0;
        var fullScreenInProgress = setInterval(function () {
          times++;
          if (times === 11) {
            clearInterval(fullScreenInProgress);
            // Set correct page view after PresentationMode changes
            this.viewer.goToPage(this.viewer.page);
          }
          this._scaleToFitPage();
        }.bind(this), 100);
      },

      _scaleToFitPage: function () {
        this.viewer.setScale(SCALE.PAGE_FIT, true);
        this.viewer.currentScaleValue = SCALE.PAGE_FIT;
      },

      annotationOptions: {
        dropTarget: '#viewer .page .annotationLayer',
        annotationCreated: function (elem) {
          var $elem = $(elem);
          return {
            pageNumber: parseInt($elem.closest('.page').attr('data-page-number'), 10)
          };
        }
      }
    });

    return PDFView;
  });
