var THREE = THREE || window.THREE;

define('3d-viewer/three-viewer', [], function () {

  function ThreeViewer () {
    this._reset();
  }

  ThreeViewer.prototype.init = function (domNode, modelPath) {
    this.domNode = domNode;

    var w = this.domNode.offsetWidth;
    var h = this.domNode.offsetHeight;

    this.clock = new THREE.Clock();
    this.scene = new THREE.Scene();
    this.sceneOverlay = new THREE.Scene();
    this.currentZoom = 50;
    this.toggleCameraType();
    this.axes = ThreeViewer.createAxes(30);
    this.sceneOverlay.add(this.axes);
    this.grid = ThreeViewer.createGrid();
    this.scene.add(this.grid);
    this.loader = ThreeViewer.loadModel(modelPath, this.scene);
    this.boundingBox = new THREE.BoundingBoxHelper();

    // FIXME: FIL-1890 unconditional scene lightning
    this.lights = ThreeViewer.createLights();
    this.scene.add(this.lights);

    this.renderer = ThreeViewer.getRenderer();
    this.renderer.setSize(w, h);
    this.renderer.setClearColor(ThreeViewer.CONST.COLOR.ECLIPSE, ThreeViewer.CONST.OPACITY.OPAQUE);
    this.domNode.appendChild(this.renderer.domElement);
    this.onWindowResizeDelegate = onWindowResize.bind(this);

    window.addEventListener('resize', this.onWindowResizeDelegate, false);

  };

  ThreeViewer.prototype.dispose = function () {
    window.removeEventListener('resize', this.onWindowResizeDelegate, false);
    this.domNode.removeChild(this.renderer.domElement);
    this._reset();
  };

  ThreeViewer.prototype.render = function () {
    if (this.renderer) {
      requestAnimationFrame(this.render.bind(this));
      this.animate();

      this.renderer.clear();
      this.renderer.render(this.scene, this.camera);
      this.renderer.clearDepth();
      this.renderer.render(this.sceneOverlay, this.camera);
    }
  };

  ThreeViewer.prototype.animate = function () {
    var delta = this.clock.getDelta();

    THREE.SEA3D.AnimationHandler.update(delta);

    THREE.AnimationHandler.update(delta);
  };

  ThreeViewer.prototype.toggleCameraType = function () {
    var w = this.domNode.offsetWidth;
    var h = this.domNode.offsetHeight;

    var oldPos = this.camera && this.camera.position;
    var isPerspectiveCamera = !this.camera || this.camera instanceof THREE.PerspectiveCamera;

    this.camera = isPerspectiveCamera
      ? new THREE.OrthographicCamera(w / -2, w / 2, h / 2, h / -2, -ThreeViewer.CONST.DISTANCE.FAR, ThreeViewer.CONST.DISTANCE.FAR)
      : new THREE.PerspectiveCamera(ThreeViewer.CONST.CAMERA.PERSPECTIVE.FOV, w / h, ThreeViewer.CONST.DISTANCE.NEAR, ThreeViewer.CONST.DISTANCE.FAR);

    if (oldPos) {
      this.camera.position.set(oldPos.x, oldPos.y, oldPos.z);
    } else {
      this.camera.position.z = this.currentZoom;
    }

    this.controls = new THREE.OrbitControls(this.camera);

  };

  ThreeViewer.prototype.toggleAxes = function () {
    this.axes.visible = !this.axes.visible;
  };

  ThreeViewer.prototype.toggleGrid = function () {
    this.grid.visible = !this.grid.visible;
  };

  ThreeViewer.prototype.toggleBoundingBox = function () {
    this.boundingBox.visible = !this.boundingBox.visible;
  };

  ThreeViewer.prototype.toggleAnimation = function () {
    if (!this.currentAnimation) return;
    if (this.currentAnimation.isPlaying) {
      this.currentAnimation.stop();
    } else {
      this.currentAnimation.play();
    }
  };

  ThreeViewer.prototype._reset = function () {
    this.scene = null;
    this.sceneOverlay = null;
    this.camera = null;
    this.controls = null;
    this.renderer = null;
    this.model = null;
    this.animations = [];
    this.currentAnimation = null;
    this.boundingBox = null;
    this.lights = null;
    this.axes = null;
    this.grid = null;
    this.domNode = null;
    this.onWindowResizeDelegate = null;
    this.currentZoom = 0;
    this.clock = null;
  };

  ThreeViewer.prototype.zoom = function (zoomFactor) {
    this.camera.zoom *= zoomFactor;
    this.camera.updateProjectionMatrix();
  };

  ThreeViewer.prototype.getZoomFactor = function () {
    return this.camera.zoom;
  };

  ThreeViewer.getRenderer = function () {
    var is3DSupported = false, renderer = null;
    try {
      var canvas = window.document.createElement('canvas');
      is3DSupported = !!canvas.getContext('webgl');
    } catch (x) {
      // not supported
    }

    if (is3DSupported) {
      renderer = new THREE.WebGLRenderer();
      renderer.setPixelRatio(window.devicePixelRatio);
    } else {
      renderer = new THREE.CanvasRenderer();
    }

    renderer.autoClear = false;
    return renderer;

  };

  ThreeViewer.createLights = function () {
    var lights = [];
    lights[0] = new THREE.PointLight(ThreeViewer.CONST.COLOR.WHITE, ThreeViewer.CONST.INTENSITY.MAX, ThreeViewer.CONST.DISTANCE.NONE);
    lights[1] = new THREE.PointLight(ThreeViewer.CONST.COLOR.WHITE, ThreeViewer.CONST.INTENSITY.MAX, ThreeViewer.CONST.DISTANCE.NONE);
    lights[2] = new THREE.PointLight(ThreeViewer.CONST.COLOR.WHITE, ThreeViewer.CONST.INTENSITY.MAX, ThreeViewer.CONST.DISTANCE.NONE);

    lights[0].position.set(0, 200, 0);
    lights[1].position.set(100, 200, 100);
    lights[2].position.set(-100, -200, -100);

    var result = new THREE.Object3D();
    var ambientLight = new THREE.AmbientLight(ThreeViewer.CONST.COLOR.BLACK);
    result.add.apply(result, [ambientLight].concat(lights));

    return result;
  };

  ThreeViewer.loadModel = function (path, container) {
    var loader = new THREE.SEA3D({
      autoPlay: true,
      container: container
    });

    loader.load(path);

    return loader;
  };

  ThreeViewer.createAxes = function (size) {
    return new THREE.AxisHelper(size);
  };

  ThreeViewer.createGrid = function () {
    var result = new THREE.GridHelper(ThreeViewer.CONST.GRID.SIZE, ThreeViewer.CONST.GRID.STEP);
    result.setColors(ThreeViewer.CONST.COLOR.NERO, ThreeViewer.CONST.COLOR.CHARCOAL);
    return result;
  };

  function onWindowResize () {
    var w = this.domNode.offsetWidth;
    var h = this.domNode.offsetHeight;

    if (this.camera instanceof THREE.PerspectiveCamera) {
      this.camera.aspect = w / h;
    } else if (this.camera instanceof THREE.OrthographicCamera) {
      this.camera.left = w / -2;
      this.camera.right = w / 2;
      this.camera.top = h / 2;
      this.camera.bottom = h / -2;
    }
    this.camera.updateProjectionMatrix();

    this.renderer.setSize(w, h);
  }

  ThreeViewer.CONST = {
    DISTANCE: {
      NONE: 0,
      NEAR: 1,
      FAR: 5000
    },
    COLOR: {
      WHITE: 0xFFFFFF,
      BLACK: 0x000000,
      CHARCOAL: 0x4A4A4A,
      NERO: 0x202020,
      ECLIPSE: 0x393939
    },
    OPACITY: {
      OPAQUE: 1.0,
      TRANSPARENT: 0.0
    },
    INTENSITY: {
      MAX: 1
    },
    GRID: {
      SIZE: 5000,
      STEP: 200
    },
    CAMERA: {
      PERSPECTIVE: {
        FOV: 75
      }
    }
  };

  return ThreeViewer;
});