(function () {
  /*global test, equal, sinon*/
  'use strict';

  var FileViewer = require('file-viewer');
  var Storage = require('storage');

  var customStorage = {};

  module('Storage');

  var fileViewer = new FileViewer({
    customStorage: {
      setItem: function (key, value) {
        customStorage[key] = value;
      },
      getItem: function (key, value) {
        return customStorage[key];
      },
      removeItem: function (key) {
        delete customStorage[key];
      }
    }
  });

  test('#Storage() sets the namespace it is created with', function () {
    // given
    var storage = new Storage(null, 'namespace');

    // then
    equal(storage._namespace, 'namespace');
  });

  test('#getItem() returns null for not existing keys', function () {
    // given
    var storage = new Storage();

    // then
    equal(storage._hasLocalStorage(), true);
    equal(storage.getItem('foo'), null);
  });

  test('#setItem() sets and #getItem() gets values for keys', function () {
    // given
    var storage = new Storage();

    var object = {
      number: 123,
      string: 'foobar',
      array: [1, 2, 3],
      bool: false
    };

    //when
    storage.setItem('foo', object);
    var result = storage.getItem('foo');
    storage.removeItem('foo');

    // then
    equal(storage._hasLocalStorage(), true);
    equal(result.number, object.number);
    equal(result.string, object.string);
    equal(result.array.toString(), object.array.toString());
    equal(result.bool, object.bool);
  });

  test('#setItem() with given expire time expires', function () {
    // given
    var storage = new Storage();

    //when
    storage.setItem('foo', 'bar', 5000);
    equal(storage._hasLocalStorage(), true);
    equal(storage.getItem('foo'), 'bar');

    // Forward the clock
    var clock = sinon.useFakeTimers(Date.now() + 5001);

    // then
    equal(storage.getItem('foo'), null);
    clock.restore();
  });

  test('#Storage customStorage behaves like localStorage', function () {
    // given
    var storage       = new Storage();
    var customStorage = new Storage(fileViewer.getConfig().customStorage);

    // when
    var object = {
      number: 123,
      string: 'foobar',
      array: [1, 2, 3],
      bool: true
    };

    storage.setItem('bar', object, 1000);
    customStorage.setItem('bar', object, 1000);

    // then
    equal(storage.getItem('foo'), customStorage.getItem('foo'));

    equal(storage.getItem('bar').number, customStorage.getItem('bar').number);
    equal(storage.getItem('bar').string, customStorage.getItem('bar').string);
    equal(storage.getItem('bar').array.toString(), customStorage.getItem('bar').array.toString());
    equal(storage.getItem('bar').bool, customStorage.getItem('bar').bool);

    var clock = sinon.useFakeTimers(Date.now() + 2000);
    equal(storage.getItem('bar'), customStorage.getItem('bar'));
    clock.restore();
  });

}());
