define('storage', ['assert'], function (assert) {
  'use strict';

  /**
   * Universal key-value store using localStorage or
   * optionally a `customStorage` object can be passed in.
   * @constructor
   * @param {Object} customStorage - Object with setItem, getItem, removeItem methods.
   * @param {Function} customStorage.setItem - Custom set method
   * @param {Function} customStorage.getItem - Custom get method
   * @param {Function} customStorage.removeItem - Custom remove method
   * @param {String} namespace - Namespace to prefix storage keys with
   */
  var Storage = function (customStorage, namespace) {
    this._namespace = namespace || '';
    this._storage = this._getStorage(customStorage);
  };

  /**
   * Return corresponding value to given key.
   * - can store arbitrary json structs for certain values
   *   (not any value, functions won't work for example)
   * - for better performance, try to have more keys and less structs
   *   (cause big structs are slow)
   * @param  {String} key
   * @return {*}
   */
  Storage.prototype.getItem = function (key) {
    var storageValue;
    key = this._namespace + key.toString();
    storageValue = new StorageValue();
    storageValue.fromJSON(this._storage.getItem(key));
    if (storageValue.isExpired()) {
      this._storage.removeItem(key);
      return;
    }
    return storageValue.value;
  };

  /**
   * Store given value for with given key.
   * @param {String} key
   * @param {*}    value
   * @param {Number} expiry - Expire n milliseconds from now
   */
  Storage.prototype.setItem = function (key, value, expiry) {
    key = this._namespace + key.toString();
    this._storage.setItem(
      key,
      new StorageValue(value, expiry).toJSON()
    );
  };

  /**
   * Remove storage entry for key.
   * @param {String} key
   */
  Storage.prototype.removeItem = function (key) {
    key = this._namespace + key.toString();
    this._storage.removeItem(key);
  };

  // Check if window.localStorage is available.
  Storage.prototype._hasLocalStorage = function () {
    var test = this._namespace + 'hasLocalStorage';

    try {
      localStorage.setItem(test, test);
      localStorage.removeItem(test);
      return true;
    } catch (e) {
      return false;
    }
  };

  // Return the proper storage object.
  Storage.prototype._getStorage = function (customStorage) {
    if (customStorage) {
      assert(
      typeof customStorage.setItem === 'function',
      'Storage custom.setItem is a function'
      );
      assert(
      typeof customStorage.getItem === 'function',
      'Storage custom.getItem is a function'
      );
      assert(
      typeof customStorage.removeItem === 'function',
      'Storage custom.removeItem is a function'
      );
      return customStorage;
    } else if (this._hasLocalStorage()) {
      return Storage.localStorage;
    }
    return Storage.noStorage;
  };

  Storage.noStorage = {
    setItem: function () {},
    getItem: function () {},
    removeItem: function () {}
  };

  Storage.localStorage = {
    setItem: function (key, value) {
      window.localStorage.setItem(key, value);
    },

    getItem: function (key) {
      return window.localStorage.getItem(key);
    },

    removeItem: function (key) {
      window.localStorage.removeItem(key);
    }
  };

  /**
   * StorageValue
   * @param {*}    value
   * @param {Number} expiry - Expire n milliseconds from now
   */
  var StorageValue = Storage.StorageValue = function (value, expiry) {
    this.value  = value;
    this._setExpiry(expiry);
  };

  StorageValue.prototype._setExpiry = function (expiry) {
    if (parseInt(expiry)) {
      this.expiry = Date.now() + expiry;
    } else {
      this.expiry = null;
    }
  };

  /**
   * Checks if the expiry time is in the past
   * @return {Boolean}
   */
  StorageValue.prototype.isExpired = function () {
    return this.expiry && Date.now() > this.expiry;
  };

  /**
   * Sets StorageValue properties from a JSON string.
   * @param  {String} stringifiedJson
   */
  StorageValue.prototype.fromJSON = function (stringifiedJson) {
    var json;
    stringifiedJson = stringifiedJson || '{}';
    json = JSON.parse(stringifiedJson);
    this.value = json.value;
    this.expiry = json.expiry;
  };

  /**
   * Return JSON string
   * @return {String}
   */
  StorageValue.prototype.toJSON = function () {
    return JSON.stringify({
      value: this.value,
      expiry: this.expiry || undefined
    });
  };

  return Storage;
});
