define('module-store', [
  'assert',
  'jquery',
  'underscore'
], function (
  assert,
  $,
  _
) {
  'use strict';

  /**
   * Provides modules by asking a previously configured backend.
   *
   * As modules can be loaded async, a promise is always returned.
   *
   * A backend is a function that accepts a module path and returns the
   * matched module. If no module is found, it returns undefined.
   *
   * @constructor
   */
  var ModuleStore = function () {
    this._backend = null;
  };

  /**
   * Checks if backend is a valid backend.
   * @param {*} backend
   * @return {bool}
   */
  ModuleStore.validBackend = function (backend) {
    return _.isFunction(backend);
  };

  /**
   * Asks its backend for the given modulePath and returns a promise.
   * @param {string} modulePath
   * @return {Promise}
   * @throws {Error} if backend is invalid
   */
  ModuleStore.prototype.get = function (modulePath) {
    assert(ModuleStore.validBackend(this._backend), 'backend is valid');
    return $.when(this._backend(modulePath));
  };

  /**
   * Sets a backend for this module store.
   * @param {function} backend
   * @throws {Error} if backend is invalid
   */
  ModuleStore.prototype.useBackend = function (backend) {
    assert(ModuleStore.validBackend(backend), 'backend is valid');
    this._backend = backend;
  };

  return ModuleStore;
});