(function () {
    'use strict';

    var $ = require('jquery');
    var Backbone = require('backbone');
    var LayerContainerView = require('layer-container-view');

    module('LayerContainerView', {
        setup: function () {
            this.contentLayerView = new LayerContainerView({});
            this.CustomView = Backbone.View.extend({});
        }
    });

    test('#addLayerView() adds a LayerView for the given name', function () {
        this.contentLayerView.addLayerView('name', this.CustomView);
        ok(this.contentLayerView.hasLayerView('name'), 'layer exists after adding');
    });

    test('#addLayerView() throws when adding another LayerView for an existing name', function () {
        this.contentLayerView.addLayerView('name', this.CustomView);
        throws(function () {
            this.contentLayerView.addLayerView('name', this.CustomView);
        }, 'throws when the same name is used another time');
    });

    test('#initializeLayers() initializes previously added LayerViews', function () {
        var initialize = sinon.spy();
        var LayerView = Backbone.View.extend({ initialize: initialize });

        this.contentLayerView.addLayerView('A', LayerView);
        this.contentLayerView.addLayerView('B', LayerView);

        this.contentLayerView.initializeLayers();
        ok(initialize.calledTwice, 'added layers are initialized');

        this.contentLayerView.addLayerView('C', LayerView);
        ok(initialize.calledThrice === false, 'layers added after initialize are ignored');
    });

    test('#initializeLayers() initializes LayerViews when predicate returns true', function () {
        var truthyPredicate = sinon.stub().returns(true);
        var truthyInitialize = sinon.spy();
        var TruthyLayerView = Backbone.View.extend({ initialize: truthyInitialize });

        var falsyPredicate = sinon.stub().returns(false);
        var falsyInitialize = sinon.spy();
        var FalsyLayerView = Backbone.View.extend({ initialize: falsyInitialize });

        this.contentLayerView.addLayerView('truthy', TruthyLayerView, { predicate: truthyPredicate });
        this.contentLayerView.addLayerView('falsy', FalsyLayerView, { predicate: falsyPredicate });
        this.contentLayerView.initializeLayers();

        ok(truthyPredicate.calledOnce, 'truthy predicate executed');
        ok(truthyInitialize.calledOnce, 'truthy layer initialized');

        ok(falsyPredicate.calledOnce, 'falsy predicate executed');
        ok(falsyInitialize.called === false, 'falsy layer not initialized');
    });

    test('#initializeLayers() passes correct parameters to layer initializer', function () {
        var initialize = sinon.spy();
        var LayerView = Backbone.View.extend({ initialize: initialize });

        this.contentLayerView.addLayerView('A', LayerView);
        this.contentLayerView.initializeLayers();

        ok(initialize.calledOnce, 'initializer is called');
        equal(initialize.args[0][0].contentLayerView, this.contentLayerView);
    });

    test('#initializeLayers() throws if layers are already initialized', function () {
        this.contentLayerView.initializeLayers();
        this.contentLayerView.teardownLayers();
        this.contentLayerView.initializeLayers();

        throws(function () {
            this.contentLayerView.initializeLayers();
        }, 'initializing layers a second time throws');
    });

    test('#initializeLayers() fires initializeLayers event', function () {
        var listener = sinon.spy();
        this.contentLayerView.once('initializeLayers', listener);
        this.contentLayerView.initializeLayers();
        ok(listener.calledOnce, 'listener was called');
    });

    test('#initializeLayers() calls render', function () {
        sinon.spy(this.contentLayerView, 'render');
        this.contentLayerView.initializeLayers();
        ok(this.contentLayerView.render.called, 'render is called');
    });

    test('#teardownLayers() calls render', function () {
        this.contentLayerView.initializeLayers();
        sinon.spy(this.contentLayerView, 'render');
        this.contentLayerView.teardownLayers();
        ok(this.contentLayerView.render.called, 'render is called');
    });

    test('#teardownLayers() tears down initialized subviews once', function () {
        var teardown = sinon.spy();
        var LayerView = Backbone.View.extend({ teardown: teardown });

        this.contentLayerView.addLayerView('A', LayerView);
        this.contentLayerView.addLayerView('B', LayerView);

        this.contentLayerView.teardownLayers();
        ok(teardown.called === false, 'ignores layers that are added, but not initialized');

        this.contentLayerView.initializeLayers();
        this.contentLayerView.teardownLayers();
        ok(teardown.calledTwice, 'all layers are torn down');

        this.contentLayerView.teardownLayers();
        ok(teardown.calledTwice, 'all layers are torn down only once');
    });

    test('#teardownLayers() also calls remove on subviews', function () {
        var remove = sinon.spy();
        var LayerView = Backbone.View.extend({ remove: remove });

        this.contentLayerView.addLayerView('A', LayerView);

        this.contentLayerView.teardownLayers();
        ok(remove.called === false, 'ignores layers that are added, but not initialized');

        this.contentLayerView.initializeLayers();
        this.contentLayerView.teardownLayers();
        ok(remove.calledOnce, 'layer is removed');
    });

    test('#teardownLayers() does not throw if layers are not initialized', function () {
        equal(this.contentLayerView.areLayersInitialized(), false);
        this.contentLayerView.teardownLayers();
        // no exception thrown at this point
    });

    test('#teardownLayers() fires teardownLayers event', function () {
        var listener = sinon.spy();
        this.contentLayerView.once('teardownLayers', listener);
        this.contentLayerView.teardownLayers();
        ok(listener.calledOnce, 'listener was called');
    });

    test('#teardownLayers() does not throw if layer has no teardown method', function () {
        var Layer = Backbone.View.extend({});
        this.contentLayerView.addLayerView('A', Layer);
        this.contentLayerView.initializeLayers();
        this.contentLayerView.teardownLayers();
        // no exception thrown at this point
        ok(true);
    });

    test('#reinitializeLayers() recreates views', function () {
        var initializeSpy = sinon.spy();
        var teardownSpy = sinon.spy();
        var Layer = Backbone.View.extend({
            initialize: initializeSpy,
            teardown: teardownSpy
        });
        this.contentLayerView.addLayerView('A', Layer);

        this.contentLayerView.initializeLayers();
        this.contentLayerView.reinitializeLayers();

        ok(teardownSpy.called, 'teardown was called');
        ok(initializeSpy.called, 'initialize was called');
        ok(teardownSpy.calledBefore(initializeSpy), 'teardown is called before initialize');
    });

    test('#isLayerInitialize() returns true if the layer is initialized', function () {
        ok(!this.contentLayerView.isLayerInitialized('A'), 'A not yet initialized');
        this.contentLayerView.initializeLayers();
        ok(!this.contentLayerView.isLayerInitialized('A'), 'A not yet initialized');
        this.contentLayerView.teardownLayers();

        this.contentLayerView.addLayerView('A', Backbone.View);
        this.contentLayerView.initializeLayers();
        ok(this.contentLayerView.isLayerInitialized('A'), 'A initialized');
    });

    test('#getLayerForName() returns an instanciated layer for the given name', function () {
        var value = {};
        var LayerViewA = Backbone.View.extend({
            initialize: function () { this.value = value; }
        });
        this.contentLayerView.addLayerView('LayerViewA', LayerViewA);
        this.contentLayerView.initializeLayers();

        equal(this.contentLayerView.getLayerForName('LayerViewA').value, value, 'returns layer instance');
    });

    test('#getLayerForName() throws if layers are not yet initialized', function () {
        var LayerViewA = function () {};
        this.contentLayerView.addLayerView('LayerViewA', LayerViewA);

        throws(function () {
            this.contentLayerView.getLayerForName('LayerViewA');
        }.bind(this), 'is not initialised');
    });

    test('#getLayerForName() throws if asking for a not yet defined name', function () {
        this.contentLayerView.initializeLayers();
        throws(function () {
            this.contentLayerView.getLayerForName('LayerViewA');
        }.bind(this), 'not yet defined');
    });

    test('#getLayerForName() throws if asking for a defined, but not initialized name', function () {
        var falsyPredicate = function () { return false; };
        var LayerViewA = function () {};
        this.contentLayerView.addLayerView('LayerViewA', LayerViewA, { predicate: falsyPredicate });
        this.contentLayerView.initializeLayers();

        throws(function () {
            this.contentLayerView.getLayerForName('LayerViewA');
        }.bind(this), 'not initialized');
    });

    test('#render() delegates to initialized subviews', function () {
        var render = sinon.spy();
        var LayerView = Backbone.View.extend({ render: render });

        this.contentLayerView.addLayerView('A', LayerView);
        this.contentLayerView.addLayerView('B', LayerView);
        this.contentLayerView.initializeLayers();

        this.contentLayerView.addLayerView('C', LayerView);

        this.contentLayerView.render();

        ok(render.called, 'calls render() on initialized subviews');
    });

    test('#render() delegates to subviews in order added if no weights given', function () {
        var renderA = sinon.spy();
        var renderB = sinon.spy();
        var LayerViewA = Backbone.View.extend({ render: renderA });
        var LayerViewB = Backbone.View.extend({ render: renderB });

        this.contentLayerView.addLayerView('BTest', LayerViewB);
        this.contentLayerView.addLayerView('A', LayerViewA);

        this.contentLayerView.initializeLayers();
        // render called automatically

        ok(renderA.called, 'render A called');
        ok(renderB.called, 'render B called');
        ok(renderB.calledBefore(renderA), 'B rendered before A');
    });

    test('#render() delegates to subviews sorted by descending weights if weights given', function () {
        var renderA = sinon.spy();
        var renderB = sinon.spy();
        var LayerViewA = Backbone.View.extend({ render: renderA });
        var LayerViewB = Backbone.View.extend({ render: renderB });

        this.contentLayerView.addLayerView('B', LayerViewB, { weight: 60 });
        this.contentLayerView.addLayerView('A', LayerViewA, { weight: 50 });

        this.contentLayerView.initializeLayers();
        // render called automatically

        ok(renderA.called, 'render A called');
        ok(renderB.called, 'render B called');
        ok(renderB.calledBefore(renderA), 'B rendered before A');
    });

    test('#render() can be called without layers beeing initialized', function () {
        equal(this.contentLayerView.areLayersInitialized(), false);
        this.contentLayerView.render();
        // no exception thrown at this point
    });

    test('#render() cleans and re-attaches all layer elements', function () {
        var LayerViewA = Backbone.View.extend({ className: 'layerA' });
        var LayerViewB = Backbone.View.extend({ className: 'layerB' });

        this.contentLayerView.addLayerView('A', LayerViewA);
        this.contentLayerView.addLayerView('B', LayerViewB);

        this.contentLayerView.initializeLayers();

        this.contentLayerView.render();
        equal(this.contentLayerView.$el.find('.layerA').length, 1, 'layer appended once');
        equal(this.contentLayerView.$el.find('.layerB').length, 1, 'layer appended once');

        this.contentLayerView.render();
        equal(this.contentLayerView.$el.find('.layerA').length, 1, 'layer appended once');
        equal(this.contentLayerView.$el.find('.layerB').length, 1, 'layer appended once');
    });

    test('#render() cleans elements without being initialized', function () {
        this.contentLayerView.$el.append('<div></div>');
        equal(this.contentLayerView.$el.children().length, 1, 'has a child');
        this.contentLayerView.render();
        equal(this.contentLayerView.$el.children().length, 0, 'has no childs');
    });

    test('#render() fires renderLayers event', function () {
        var listener = sinon.spy();
        this.contentLayerView.once('renderLayers', listener);
        this.contentLayerView.render();
        ok(listener.calledOnce, 'listener was called');
    });

}());
