define('layer-container-view', [
  'underscore',
  'backbone',
  'assert',
  'constants-dictionary'
], function (
  _,
  Backbone,
  assert,
  ConstantsDictionary
) {
  'use strict';

  // utility functions for working with layers

  var invoke = function (fn) {
    return fn();
  };

  var pick = function (property, obj) {
    return obj[property];
  };

  var pickBoundFn = function (property, obj) {
    return _.isFunction(obj[property]) && obj[property].bind(obj);
  };

  /**
   * This view manages a collection of views which can be registered with a
   * given name. This view manages the lifecycle of its subviews.
   *
   * Subviews are always the View objects themselves, not instances of them.
   *
   * Subviews have two different states: ADDED and INITIALIZED. Whenever a
   * view is registered, it starts in state ADDED and stays there until
   * #initializeLayers() is called. Then it moves to INITIALIZED and stays
   * there until #teardownLayers() is called.
   *
   * When #render() is called on the collection, only INITIALIZED subviews are
   * rendered. Subviews can provide a teardown method that will be called
   * once the view is removed.
   *
   * Optionally, you can register subviews with a predicate to tell which
   * filetypes they support. It is invoked whenever the subviews are
   * initialized.
   */
  var FileContentLayerView = Backbone.View.extend({

    /**
     * @constructor
     * @param {object} options
     */
    initialize: function (options) {
      this._layerViewsByName = new ConstantsDictionary();
      this._layerViewRegistrations = [];
      this._layers = null;
      this._fileViewer = options.fileViewer;
    },

    /**
     * Checks if a layer with the given name exists.
     * @param {string} name
     * @return {bool}
     */
    hasLayerView: function (name) {
      return this._layerViewsByName.isDefined(name);
    },

    /**
     * Adds a view as a layer with a certain, unique name. Accepts an
     * options object as third parameter.
     *
     * Keys in options:
     *  - {function} [predicate] invoked at construction
     *  - {int} [weight=0] sorts layers at construction
     *
     * @param {string} name
     * @param {Backbone.View} LayerView
     * @param {object} [options]
     * @throws Error if name is already used.
     */
    addLayerView: function (name, LayerView, options) {
      assert(!this.hasLayerView(name), 'name is unique');

      options = _.extend({
        predicate: function () { return true; },
        weight: 0
      }, options);

      this._layerViewsByName.define(name, LayerView);
      this._layerViewRegistrations.push({
        LayerView: LayerView,
        name: name,
        predicate: options.predicate,
        weight: options.weight
      });
    },

    /**
     * Checks wether layers are currently initialized.
     * @return {bool}
     */
    areLayersInitialized: function () {
      return this._layers !== null;
    },

    /**
     * Return the number of initialized layers (after applying the predicate)/
     * @return {Integer}
     */
    countInitializedLayers: function () {
      return (this._layers || []).length;
    },

    /**
     * Initializes all currently registered layers.
     * @fires initializeLayers
     * @throws Error if layers are already initialized
     */
    initializeLayers: function () {
      this.initializeLayerSubset(_.map(this._layerViewRegistrations, function (item) { return item.name; }));
    },

    /**
     * Initializes the given registered layers.
     * @param {Array} names
     * @fires initializeLayers
     * @throws Error if layers are already initialized
     */
    initializeLayerSubset: function (names) {
      assert(!this.areLayersInitialized(), 'layers are uninitialized');

      this._layers = this._layerViewRegistrations
                .filter(function (registration) {
                  var isInSubset = (names.indexOf(registration.name) !== -1);
                  return isInSubset && registration.predicate(this._fileViewer);
                }, this)
                .map(function (registration) {
                  var view = new registration.LayerView({
                    contentLayerView: this,
                    fileViewer: this._fileViewer
                  });
                  return {
                    view: view,
                    name: registration.name,
                    weight: registration.weight
                  };
                }, this);

      // sort by weight using the stable _.sortBy function to keep
      // registration order for same weights
      this._layers = _.sortBy(this._layers, function (layer) {
        return layer.weight * -1;
      });

      this.trigger('initializeLayers');

      this.render();
    },

    /**
     * Tears initialized layers down and removes them.
     * Won't throw if layers are not initialized.
     * @fires teardownLayers
     */
    teardownLayers: function () {
      if (this.areLayersInitialized()) {
        this._layers.map(_.partial(pick, 'view'))
              .map(_.partial(pickBoundFn, 'teardown'))
              .filter(_.isFunction)
              .forEach(invoke);

        this._layers.map(_.partial(pick, 'view'))
              .map(_.partial(pickBoundFn, 'remove'))
              .filter(_.isFunction)
              .forEach(invoke);

        this._layers = null;
      }

      this.trigger('teardownLayers');

      this.render();
    },

    /**
     * Utitily method. Calls teardownLayers() followed by initializeLayers().
     */
    reinitializeLayers: function () {
      this.teardownLayers();
      this.initializeLayers();
    },

    /**
     * Checks wether a layer with the given name is currently initialized.
     * @param {string} name
     * @return {bool}
     */
    isLayerInitialized: function (name) {
      if (!this.areLayersInitialized()) { return false; }

      return _.find(this._layers, function (layer) {
        return layer.name === name;
      }) ? true : false;
    },

    /**
     * Returns the instanciated LayerView object for the given name.
     * @param {string} name Name of the registered LayerView.
     * @return {layerView}
     * @throws {Error} if layer is not initialized
     */
    getLayerForName: function (name) {
      assert(this.areLayersInitialized(), 'layers are initialized');
      assert(this.hasLayerView(name), 'layer is defined');

      var layer = _.find(this._layers, function (layer) {
        return layer.name === name;
      });

      assert(layer, 'layer is initialized');

      return layer.view;
    },

    /**
     * Renders initialized layers.
     * Won't throw if layers are not initialized.
     * @fires renderLayers
     */
    render: function () {
      this.$el.empty();

      if (this.areLayersInitialized()) {
        this._layers.map(_.partial(pick, 'view'))
              .map(_.partial(pickBoundFn, 'render'))
              .forEach(invoke);

        this._layers.map(_.partial(pick, 'view'))
              .map(_.partial(pick, '$el'))
              .forEach(function ($layer) {
                this.$el.append($layer);
              }, this);
      }

      this.trigger('renderLayers');

      return this;
    }

  });

  return FileContentLayerView;
});