define('instance-manager',
  ['jquery'],
  function ($) {
    'use strict';

    /**
     * Ensures that there's only a single instance at the same time.
     * Accepts a createFn that is invoked and whose return value is
     * used as the instance.
     * @constructor
     * @param {Function} createFn
     * @param {Function} destroyFn
     */
    var InstanceManager = function (createFn, destroyFn) {
      this._createFn = createFn;
      this._destroyFn = destroyFn;
      this._instance = null;
      this._destroyDeferred = null;
    };

    /**
     * Promise a new instance. Resolves as soon as the previous one
     * is destroyed.
     * @return {Promise<Object>}
     */
    InstanceManager.prototype.create = function () {
      var args = arguments;
      var d = new $.Deferred();
      var destroyPromise = (this._destroyDeferred && this._destroyDeferred.promise()) || $.when();

      destroyPromise.then(function () {
        this._destroyDeferred = new $.Deferred();
        this._instance = this._createFn.apply(this._createFn, args);
        d.resolve(this._instance);
      }.bind(this));

      return d.promise();
    };

    /**
     * Destroy the current instance and unlock instance creation.
     */
    InstanceManager.prototype.destroy = function () {
      if (!this._destroyDeferred) {
        return;
      }
      this._destroyFn(this._instance);
      this._destroyDeferred.resolve();
    };

    return InstanceManager;

  }
);
