(function () {
    'use strict';

    var $ = require('jquery');
    var keys = require('keyboard').keys;
    var FileViewer = require('file-viewer');
    var MainView = require('MainView');
    var mainViewDefaultModel = new Backbone.Model({
        fileViewer: null,
        instanceId: 1,
        embedded: false
    });

    var File = require('file');

    var truthy = function () { return true; };
    var noop = function () {};

    module('MainView', {
        setup: function () {
            mainViewDefaultModel.set('fileViewer', new FileViewer());
        }
    });

    test('#showFile() always sets _currentFile, even when there is no file given', function () {
        var mainView = new MainView({ model: mainViewDefaultModel });
        mainView.showFile(new File());
        mainView.showFile(null);
        equal(mainView.getCurrentFile(), null);
    });

    test('#showFile() doesn‘t re-render the content when all attributes are the same', function () {
        // given
        var mainView = new MainView({ model: mainViewDefaultModel });
        var fileA = new File({ src: 'x', type: 'y' });
        var fileB = new File({ src: 'x', type: 'y' });

        mainView.showFile(fileA);
        sinon.spy(mainView.fileContentView, 'reinitializeLayers');

        // when
        mainView.showFile(fileB);

        // then
        ok(!mainView.fileContentView.reinitializeLayers.called);
    });

    test('#showFile() doesn‘t trigger cancelShow when all attributes are the same', function () {
        // given
        var mainView = new MainView({ model: mainViewDefaultModel });
        var fileA = new File({ src: 'x', type: 'y' });
        var fileB = new File({ src: 'x', type: 'y' });
        var listener = sinon.spy();

        mainView.showFile(fileA);
        mainView.on('cancelShow', listener);

        // when
        mainView.showFile(fileB);

        // then
        ok(!listener.called);
    });

    test('#showFile() doesn‘t trigger fv.fileChange when all attributes are the same', function () {
        // given
        var mainView = new MainView({ model: mainViewDefaultModel });
        var fileA = new File({ src: 'x', type: 'y' });
        var fileB = new File({ src: 'x', type: 'y' });
        var listener = sinon.spy();

        mainView.showFile(fileA);
        mainView.on('fv.fileChange', listener);

        // when
        mainView.showFile(fileB);

        // then
        ok(!listener.called);
    });

    test('#showFile() resolves with the given file when all attributes are the same', function () {
        // given
        var mainView = new MainView({ model: mainViewDefaultModel });
        var fileA = new File({ src: 'x', type: 'y' });
        var fileB = new File({ src: 'x', type: 'y' });

        mainView.showFile(fileA);

        // when
        mainView.showFile(fileB).then(function (resultFile) {
            // then
            equal(resultFile, fileB);
        });
    });

    test('#showFile() doesn‘t re-render the content as long as type and src attributes are the same', function () {
        // given
        var mainView = new MainView({ model: mainViewDefaultModel });
        var fileA = new File({ src: 'x', type: 'y', title: 'x' });
        var fileB = new File({ src: 'x', type: 'y', title: 'y' });

        mainView.showFile(fileA);
        sinon.spy(mainView.fileContentView, 'reinitializeLayers');
        sinon.spy(mainView.fileTitleView, 'reinitializePanel');

        // when
        mainView.showFile(fileB);

        // then
        ok(mainView.fileTitleView.reinitializePanel.called);
        ok(!mainView.fileContentView.reinitializeLayers.called);
    });

    test('#getViewState() returns "loading" while view is loading', function () {
        var fileViewer = mainViewDefaultModel.get('fileViewer');
        var d = new $.Deferred();
        fileViewer._viewerRegistry.get = function () {
            return function () { return d; };
        };
        var mainView = new MainView({ model: mainViewDefaultModel });
        mainView.showFile(new File());
        equal(mainView.getViewState(), 'loading');
        d.reject(new Error());
    });

    test('#getViewState() returns "fileNotFound" when file is not found', function () {
        var mainView = new MainView({ model: mainViewDefaultModel });
        expect(1);
        QUnit.stop();
        mainView.showFile(null).always(function () {
            QUnit.start();
            equal(mainView.getViewState(), 'fileNotFound');
        });
    });

    test('#getViewState() returns "viewerError" when there was a viewer error', function () {
        var fileViewer = mainViewDefaultModel.get('fileViewer');
        var viewerProvider = function () {
            return $.when(function () {
                return {once: function (ev, cb) { if (ev === 'viewerFail') {cb(new Error());} }, render:noop};
            });
        };
        fileViewer._viewerRegistry.register(truthy, viewerProvider, 0);
        var mainView = new MainView({ model: mainViewDefaultModel });
        expect(1);
        QUnit.stop();
        mainView.showFile(new File()).always(function () {
            QUnit.start();
            equal(mainView.getViewState(), 'viewerError');
        });

    });

    test('#getViewState() returns "success" when viewer is successful', function () {
        var mainView = new MainView({ model: mainViewDefaultModel });
        expect(1);
        QUnit.stop();
        mainView.showFile(new File()).always(function () {
            QUnit.start();
            equal(mainView.getViewState(), 'success');
        });
    });

    // test('pressing ESCAPE sends an analytics event', function () {
    //     // given
    //     var file = {src: 'test', type:'test/example'};
    //     var hasher = function (x) { return x.toUpperCase() };
    //     var backend = sinon.spy();
    //     var fileViewer = new FileViewer({ analyticsBackend:backend, appendTo: $('<div>') });
    //     fileViewer.analytics._hasher = hasher;

    //     // when
    //     fileViewer.open();
    //     fileViewer.updateFiles([file]);

    //     fileViewer.showFileWithQuery({src: file.src});
    //     $(document).trigger({type: 'keydown', which: keys.ESCAPE})

    //     // then
    //     ok(backend.calledOnce);
    //     var key = backend.args[0][0];
    //     var data = backend.args[0][1];
    //     equal(key, 'files.fileviewer-web.closed');
    //     equal(data.actionType, 'hotkey');
    //     equal(data.fileType, file.type);
    //     equal(data.fileId, hasher(file.src));
    //     equal(data.fileState, 'success');
    // });

    test('pressing RIGHT sends an analytics event for the next file', function () {
        // given
        var file = {src: 'test', type:'test/example'};
        var otherFile = {src: 'other', type:'other/example'};
        var hasher = function (x) { return x.toUpperCase() };
        var backend = sinon.spy();
        var fileViewer = new FileViewer({ analyticsBackend:backend, appendTo: $('<div>') });
        fileViewer.analytics._hasher = hasher;

        // when
        fileViewer.open();
        fileViewer.updateFiles([file, otherFile]);

        fileViewer.showFileWithQuery({src: otherFile.src});
        $(document).trigger({type: 'keydown', which: keys.ARROW_RIGHT});

        // then
        ok(backend.calledOnce);
        var key = backend.args[0][0];
        var data = backend.args[0][1];
        equal(key, 'files.fileviewer-web.next');
        equal(data.actionType, 'hotkey');
        equal(data.fileType, file.type);
        equal(data.fileId, hasher(file.src));
        equal(data.fileState, 'success');
    });

    test('pressing LEFT sends an analytics event for the prev file', function () {
        // given
        var file = {src: 'test', type:'test/example'};
        var otherFile = {src: 'other', type:'other/example'};
        var hasher = function (x) { return x.toUpperCase() };
        var backend = sinon.spy();
        var fileViewer = new FileViewer({ analyticsBackend:backend, appendTo: $('<div>') });
        fileViewer.analytics._hasher = hasher;

        // when
        fileViewer.open();
        fileViewer.updateFiles([file, otherFile]);

        fileViewer.showFileWithQuery({src: otherFile.src});
        $(document).trigger({type: 'keydown', which: keys.ARROW_LEFT});

        // then
        ok(backend.calledOnce);
        var key = backend.args[0][0];
        var data = backend.args[0][1];
        equal(key, 'files.fileviewer-web.prev');
        equal(data.actionType, 'hotkey');
        equal(data.fileType, file.type);
        equal(data.fileId, hasher(file.src));
        equal(data.fileState, 'success');
    });

    test('clicking the background sends an analytics event', function () {
        // given
        var file = {src: 'test', type:'test/example'};
        var hasher = function (x) { return x.toUpperCase() };
        var backend = sinon.spy();
        var fileViewer = new FileViewer({ analyticsBackend:backend, appendTo: $('<div>') });
        fileViewer.analytics._hasher = hasher;

        // when
        fileViewer.open();
        fileViewer.updateFiles([file]);

        fileViewer.showFileWithQuery({src: file.src});
        fileViewer.getView().$('.cp-error-layer').click();

        // then
        ok(backend.calledOnce);
        var key = backend.args[0][0];
        var data = backend.args[0][1];
        equal(key, 'files.fileviewer-web.closed');
        equal(data.actionType, 'element');
        equal(data.fileType, file.type);
        equal(data.fileId, hasher(file.src));
        equal(data.fileState, 'success');
    });

    // @TODO: Investigate why this test is failing on its original position
    // but not here.
    test('pressing ESCAPE sends an analytics event', function () {
        // given
        var file = {src: 'test', type:'test/example'};
        var hasher = function (x) { return x.toUpperCase() };
        var backend = sinon.spy();
        var fileViewer = new FileViewer({ analyticsBackend:backend, appendTo: $('<div>') });
        fileViewer.analytics._hasher = hasher;

        // when
        fileViewer.open();
        fileViewer.updateFiles([file]);

        fileViewer.showFileWithQuery({src: file.src});
        $(document).trigger({type: 'keydown', which: keys.ESCAPE})

        // then
        ok(backend.calledOnce);
        var key = backend.args[0][0];
        var data = backend.args[0][1];
        equal(key, 'files.fileviewer-web.closed');
        equal(data.actionType, 'hotkey');
        equal(data.fileType, file.type);
        equal(data.fileId, hasher(file.src));
        equal(data.fileState, 'success');
    });

    test('clicking Start presentation sends an analytics event', function () {
        //given
        var file = {src: 'test', type:'test/example'};
        var hasher = function (x) { return x.toUpperCase() };
        var backend = sinon.spy();
        var fileViewer = new FileViewer({ analyticsBackend:backend, appendTo: $('<div>') });
        fileViewer.analytics._hasher = hasher;

        // when
        fileViewer.open();
        fileViewer.updateFiles([file]);

        fileViewer.showFileWithQuery({src: file.src});
        fileViewer.getView().$('.cp-toolbar-presentation').click();

        // then
        ok(backend.calledOnce);
        var key = backend.args[0][0];
        var data = backend.args[0][1];
        equal(key, 'files.fileviewer-web.presentation.enter');
        equal(data.fileType, file.type);
        equal(data.fileId, hasher(file.src));
        equal(data.fileState, 'success');
    });

    test('in PRESENTATION mode, clicking Exit presentation sends an analytics event', function () {
        //given
        var file = {src: 'test', type:'test/example'};
        var hasher = function (x) { return x.toUpperCase() };
        var backend = sinon.spy();
        var fileViewer = new FileViewer({ analyticsBackend:backend, appendTo: $('<div>') });
        fileViewer.analytics._hasher = hasher;

        // when
        fileViewer.open();
        fileViewer.updateFiles([file]);

        fileViewer.showFileWithQuery({src: file.src});
        fileViewer.changeMode('PRESENTATION');
        fileViewer.getView().$('.cp-toolbar-presentation-exit').click();

        // then
        ok(backend.calledOnce);
        var key = backend.args[0][0];
        var data = backend.args[0][1];
        equal(key, 'files.fileviewer-web.presentation.exit');
        equal(data.actionType, 'button');
        equal(data.fileType, file.type);
        equal(data.fileId, hasher(file.src));
        equal(data.fileState, 'success');
    });

    test('in PRESENTATION mode, pressing ESC sends an analytics event', function () {
        //given
        var file = {src: 'test', type:'test/example'};
        var hasher = function (x) { return x.toUpperCase() };
        var backend = sinon.spy();
        var fileViewer = new FileViewer({ analyticsBackend:backend, appendTo: $('<div>') });
        fileViewer.analytics._hasher = hasher;

        // when
        fileViewer.open();
        fileViewer.updateFiles([file]);

        fileViewer.showFileWithQuery({src: file.src});
        fileViewer.changeMode('PRESENTATION');
        $(document).trigger({type: 'fullscreenchange'});

        // then
        ok(backend.calledOnce);
        var key = backend.args[0][0];
        var data = backend.args[0][1];
        equal(key, 'files.fileviewer-web.presentation.exit');
        equal(data.actionType, 'hotkey');
        equal(data.fileType, file.type);
        equal(data.fileId, hasher(file.src));
        equal(data.fileState, 'success');
    });

    test('in PRESENTATION mode, pressing RIGHT sends an analytics event for the next file', function () {
        // given
        var file = {src: 'test', type:'test/example'};
        var otherFile = {src: 'other', type:'other/example'};
        var hasher = function (x) { return x.toUpperCase() };
        var backend = sinon.spy();
        var fileViewer = new FileViewer({ analyticsBackend:backend, appendTo: $('<div>') });
        fileViewer.analytics._hasher = hasher;

        // when
        fileViewer.open();
        fileViewer.updateFiles([file, otherFile]);

        fileViewer.showFileWithQuery({src: otherFile.src});
        fileViewer.changeMode('PRESENTATION');
        $(document).trigger({type: 'keydown', which: keys.ARROW_RIGHT})

        // then
        ok(backend.calledOnce);
        var key = backend.args[0][0];
        var data = backend.args[0][1];
        equal(key, 'files.fileviewer-web.next');
        equal(data.mode, 'PRESENTATION');
        equal(data.actionType, 'hotkey');
        equal(data.fileType, file.type);
        equal(data.fileId, hasher(file.src));
        equal(data.fileState, 'success');
    });

    test('in PRESENTATION mode, pressing LEFT sends an analytics event for the prev file', function () {
        // given
        var file = {src: 'test', type:'test/example'};
        var otherFile = {src: 'other', type:'other/example'};
        var hasher = function (x) { return x.toUpperCase() };
        var backend = sinon.spy();
        var fileViewer = new FileViewer({ analyticsBackend:backend, appendTo: $('<div>') });
        fileViewer.analytics._hasher = hasher;

        // when
        fileViewer.open();
        fileViewer.updateFiles([file, otherFile]);

        fileViewer.showFileWithQuery({src: otherFile.src});
        fileViewer.changeMode('PRESENTATION');
        $(document).trigger({type: 'keydown', which: keys.ARROW_LEFT})

        // then
        ok(backend.calledOnce);
        var key = backend.args[0][0];
        var data = backend.args[0][1];
        equal(key, 'files.fileviewer-web.prev');
        equal(data.mode, 'PRESENTATION');
        equal(data.actionType, 'hotkey');
        equal(data.fileType, file.type);
        equal(data.fileId, hasher(file.src));
        equal(data.fileState, 'success');
    });

}());
