(function (FileViewer) {
    'use strict';

    // use FileViewer's internal module system
    var define  = FileViewer.define;
    var require = FileViewer.require;

define('minimode/MinimodePanel',
  [
    'jquery',
    'ajs',
    'backbone',
    'minimode/ThumbnailView',
    'template-store-singleton'
  ],
  function (
    $,
    AJS,
    Backbone,
    FileThumbnailView,
    templateStore
  ) {
    'use strict';

    var FileMinimodeView = Backbone.View.extend({

      id: 'cp-footer-minimode',

      initialize: function (options) {
        this.subviews = [];
        this._fileViewer = options.fileViewer;
        this._panelView = options.panelView;
        this.listenTo(this.collection, 'add reset', this.render);
        this.listenTo(this._panelView, 'renderPanel', this._forceChromeRepaint);
        this.listenTo(this._panelView, 'renderPanel', this.scrollToSelected);
        this.$minimode = $(templateStore.get('minimode')());
        this.$minimode.appendTo(this.$el);
      },

      closeOldSubviews: function () {
        while (this.subviews.length > 0) {
          var view = this.subviews.pop();
          view.remove();
          view.unbind();
        }
      },

      render: function () {
        this.closeOldSubviews();

        this.collection.each(function (model) {
          var view = new FileThumbnailView({
            model: model,
            fileViewer: this._fileViewer,
            panelView: this._panelView
          });
          this.subviews.push(view);
          $(view.render().el).appendTo(this.$minimode);
        }, this);

        return this;
      },

      scrollToSelected: function () {
        var file = this._fileViewer.getCurrentFile();
        this.subviews.forEach(function (view) {
          if (view.model === file) {
            var topPos = view.$el.get(0).offsetTop - 59;
            if ((topPos) && this.$el.scrollTop !== topPos) {
              this.$el.find('#cp-thumbnails').scrollTop(topPos);
            }
          }
        }.bind(this));
      },

      _forceChromeRepaint: function () {
        // Chrome doesn't respect the 100% height on images once the container is resized.
        var $img = $('#cp-img');
        if ($img.length) {
          var $preview = $img.closest('#cp-image-preview'),
            left = $preview.scrollLeft(),
            top = $preview.scrollTop();
          $img.css('display', 'none').height();
          $img.css('display', 'inline-block');
          $preview.scrollLeft(left);
          $preview.scrollTop(top);
        }
      }

    });

    return FileMinimodeView;
  }
);

define('minimode/minimodePlugin', [
  'minimode/MinimodeToggle',
  'minimode/MinimodePanel'
], function (
  MinimodeToggle,
  MinimodePanel
) {
  'use strict';

  var minimodePlugin = function (fileViewer) {
    var fileView = fileViewer.getView();
    var sinkView = fileView.fileSinkView;
    var metaView = fileView.fileMetaView;

    if (!fileViewer.getConfig().enableMiniMode) {
      return;
    }

    metaView.addLayerView('minimodeToggle', MinimodeToggle, {
      predicate: MinimodeToggle.predicate
    });
    sinkView.addPanelView('minimode', MinimodePanel);
  };

  return minimodePlugin;
});
define('minimode/MinimodeToggle', [
  'backbone', 'template-store-singleton'
], function (Backbone, templateStore) {
  'use strict';

  var MinimodeToggle = Backbone.View.extend({

    events: {
      'click #cp-files-label': '_toggleMinimode'
    },

    initialize: function (options) {
      this._fileViewer = options.fileViewer;
      this._sinkView = this._fileViewer.getView().fileSinkView;
    },

    render: function () {
      this.$el.html(templateStore.get('minimodeBanner')());
      this._setShowAllFilesVisible();
      return this;
    },

    _toggleMinimode: function (event) {
      event.preventDefault();

      var analytics = this._fileViewer.analytics;

      if (this._sinkView.isPanelInitialized('minimode')) {
        this._sinkView.teardownPanel('minimode');
        analytics.send('files.fileviewer-web.minimode.closed');
      } else {
        this._sinkView.initializePanel('minimode');
        analytics.send('files.fileviewer-web.minimode.opened');
      }

      this._setShowAllFilesVisible();
    },

    _setShowAllFilesVisible: function () {
      var visible = this._sinkView.isPanelInitialized('minimode');
      this.$('.cp-files-collapser.up').toggleClass('hidden', visible);
      this.$('.cp-files-collapser.down').toggleClass('hidden', !visible);
    }

  }, {

    predicate: function (fileViewer) {
      return fileViewer._fileState.collection.length > 1;
    }

  });

  return MinimodeToggle;
});

define('minimode/ThumbnailView',
  [
    'ajs',
    'backbone',
    'jquery',
    'underscore',
    'file-types',
    'icon-utils',
    'template-store-singleton'
  ],
  function (
    AJS,
    Backbone,
    $,
    _,
    fileTypes,
    iconUtils,
    templateStore
  ) {
    'use strict';

    var ThumbnailView = Backbone.View.extend({

      className: 'cp-thumbnail',

      tagName: 'li',

      events: {
        'click' : 'jumpToFile'
      },

      initialize: function (options) {
        this._fileViewer = options.fileViewer;
        this.listenTo(this.model, 'change', this.render);
        this.listenTo(options.panelView, 'renderPanel', this.setSelected);
      },

      jumpToFile: function (event) {
        event.preventDefault();

        this._fileViewer.showFileWithCID(this.model.cid)
        .then(function () {
          var contentView = this._fileViewer.getView().fileContentView;
          var currentViewer;
          if (contentView.isLayerInitialized('content')) {
            currentViewer = contentView.getLayerForName('content')._viewer;
            currentViewer && currentViewer.play && currentViewer.play();
          }
        }.bind(this))
        .always(
          this._fileViewer.analytics.fn('files.fileviewer-web.minimode.thumbnail.clicked')
        );
      },

      setSelected: function () {
        // this may not be the same as file being shown, e.g., a different version of file is shown
        var file = this._fileViewer._fileState.getCurrent();
        if (file === this.model) {
          this.$el.addClass('selected');
        } else if (this.$el.hasClass('selected')) {
          this.$el.removeClass('selected');
        }
      },

      onThumbLoadError: function (ev) {
        var el = $(ev.target);
        el.parent().removeClass('has-thumbnail');
        el.remove();
      },

      render: function () {
        var type = this.model.get('type'),
          thumbnailSrc = this.model.get('thumbnail'),
          isImage = fileTypes.isImage(type);

        var generateThumbnail = this._fileViewer.getConfig().generateThumbnail;

        var $thumbnail = $(templateStore.get('placeholderThumbnail')({
          iconClass: iconUtils.getCssClass(type),
          title: this.model.get('title')
        }));

        this.$el.empty().append($thumbnail);

        if (thumbnailSrc && generateThumbnail) {
          generateThumbnail(this.model).done(function (thumbSrc) {
            $thumbnail.replaceWith(templateStore.get('thumbnail')({
              iconClass: iconUtils.getCssClass(type),
              thumbnailSrc: thumbSrc,
              title: this.model.get('title')
            }));
            this.$el.find('img').error(this.onThumbLoadError);
          }.bind(this));
        } else if (isImage || thumbnailSrc) {
          $thumbnail.replaceWith(templateStore.get('thumbnail')({
            iconClass: iconUtils.getCssClass(type),
            thumbnailSrc: thumbnailSrc || this.model.get('src'),
            title: this.model.get('title')
          }));
          this.$el.find('img').error(this.onThumbLoadError);
        }

        return this;
      }

    });

    return ThumbnailView;
  });

(function () {
  'use strict';
  var FileViewer = require('file-viewer');
  var minimodePlugin = require('minimode/minimodePlugin');
  FileViewer.registerPlugin('minimode', minimodePlugin);
}());

}(function () {
  var FileViewer;

    if (typeof module !== "undefined" && ('exports' in module)) {
      FileViewer = require('./fileviewer.js');
    } else if (window.require) {
      FileViewer = window.FileViewer;
    } else {
      FileViewer = window.FileViewer;
    }

    return FileViewer;
}()));
