(function (FileViewer) {
    'use strict';

    // use FileViewer's internal module system
    var define  = FileViewer.define;
    var require = FileViewer.require;

define('image-preloader/imagePreloaderPlugin', [
  'jquery'
], function (
  $
) {
  'use strict';

  var imagePreloaderPlugin = function (fileViewer) {
    var delayTimeOut,
      delay      = 500,
      config = fileViewer.getConfig(),
      numberNextImages = config.preloadImagesAfterCurrent,
      numberPrevImages = config.preloadImagesBeforeCurrent;

    /**
     * Check if given MimeType is an image MimeType
     * @param  {String}  type - MimeType string
     * @return {Boolean}
     */
    function isImage (type) {
      return /^image\/.*/i.test(type);
    }

    /**
     * Return an array of files next to the current one
     * @param  {Number}   n  - Number of adjacent
     * @param  {Function} fn - Function to calculate the adjacent direction
     * @return {Array}
     */
    function getAdjacentImages (n, fn) {
      var res   = [],
        current = fileViewer._fileState.get('currentFileIndex'),
        length  = fileViewer._fileState.collection.length,
        index;
      // No adjacent images if `current` is not within `length`
      if (current > length || current < 0) {
        return [];
      }
      // Let n never be greater than collection length minus current file
      n = Math.min(length - 1, n);
      while (n > 0) {
        index = fn(n, current, length);
        res.push(fileViewer._fileState.collection.get(index));
        n -= 1;
      }
      return res.reverse();
    }

    /**
     * Get adjacent `n` images to the right
     * @param  {Number} n
     * @return {Array}
     */
    function getNextImages (n) {
      return getAdjacentImages(n, function (n, current, length) {
        var newIndex = current + n;
        return newIndex % length;
      });
    }

    /**
     * Get adjacent `n` images to the left
     * @param  {Number} n
     * @return {Array}
     */
    function getPrevImages (n) {
      return getAdjacentImages(n, function (n, current, length) {
        var newIndex = current - n;
        if (newIndex < 0) {
          newIndex = length - Math.abs(newIndex);
        }
        return newIndex;
      });
    }

    /**
     * Once the image is done loading the next image will be preloaded.
     * @param  {jQuery} $el   - Image element triggering the handled event
     * @param  {Array}  files
     */
    function handleImageEvent ($el, files) {
      preloadImages(files);
      $el.off('load');
      $el.off('error');
    }

    /**
     * Create an image tag to preload first image from `files` array.
     * @param  {Array} files
     */
    function preloadImages (files) {
      var file = files.shift();
      if (!file || !isImage(file.get('type'))) {
        return;
      }
      var nextImage = new Image();
      if (files.length) {
        var $el = $(nextImage);
        $el.on('load', handleImageEvent.bind(null, $el, files));
        $el.on('error', handleImageEvent.bind(null, $el, files));
      }
      nextImage.src = file.get('src');
    }

    /**
     * Start preloading images on `fv.changeFile` after `delay` ms
     */
    function startPreloading () {
      clearTimeout(delayTimeOut);
      delayTimeOut = setTimeout(function () {
        preloadImages(getNextImages(numberNextImages));
        preloadImages(getPrevImages(numberPrevImages));
      }, delay);
    }

    // Start preloading once the current file is done
    fileViewer.on('fv.showFile', startPreloading);
    fileViewer.on('fv.showFileError', startPreloading);
  };

  return imagePreloaderPlugin;
});

(function () {
  'use strict';
  var FileViewer = require('file-viewer');
  var imagePreloaderPlugin = require('image-preloader/imagePreloaderPlugin');
  FileViewer.registerPlugin('imagePreloader', imagePreloaderPlugin);
}());

}(function () {
  var FileViewer;

    if (typeof module !== "undefined" && ('exports' in module)) {
      FileViewer = require('./fileviewer.js');
    } else if (window.require) {
      FileViewer = window.FileViewer;
    } else {
      FileViewer = window.FileViewer;
    }

    return FileViewer;
}()));
