(function (FileViewer) {
    'use strict';

    // use FileViewer's internal module system
    var define  = FileViewer.define;
    var require = FileViewer.require;

define('annotation-layer-builder',
  [
    'pdf-viewer/viewer-properties',
    'jquery'
  ],
  function (
    viewerProperties,
    $
    ) {
    var AnnotationLayerBuilder = function (opts) {
      this.annotations = opts.annotations;
      this._fileViewer = opts.fileViewer;
      this.PinsView = opts.PinsView;
      this.annotationLayers = [];
    };

    AnnotationLayerBuilder.prototype.create = function () {
      var that = this;

      function AnnotationLayer (options) {
        this.layerDiv = options.layerDiv;
        this.pageIdx = options.pageIndex;
        this.lastScrollSource = options.lastScrollSource || null;
        this.viewport = options.viewport;
        this.isViewerInPresentationMode = options.isViewerInPresentationMode;
        this.renderTimer = null;
        that.annotationLayers.push(this);

        this.pinsView = new that.PinsView({
          fileViewer: that._fileViewer,
          container: $(this.layerDiv),
          filter: function (annotation) {
            return annotation.get('pageNumber') === this.pageIdx + 1;
          }.bind(this),
          collection: that.annotations
        });

        this.layerDiv.className = 'annotationLayer';
      }

      AnnotationLayer.prototype = {
        render: function () {
          this.pinsView.render();
        },

        setupRenderLayoutTimer: function () {
          // Schedule renderLayout() if the user has been scrolling,
          // otherwise run it right away.
          var self = this;
          var lastScroll = (this.lastScrollSource === null ?
            0 : this.lastScrollSource.lastScroll);

          if (Date.now() - lastScroll > viewerProperties.RENDER_DELAY) { // Render right away
            this.render();
          } else { // Schedule
            if (this.renderTimer) {
              clearTimeout(this.renderTimer);
            }
            this.renderTimer = setTimeout(function () {
              self.setupRenderLayoutTimer();
            }, viewerProperties.RENDER_DELAY);
          }
        }
      };
      return AnnotationLayer;
    };

    AnnotationLayerBuilder.prototype.updateAnnotations = function (annotations) {
      this.annotations = annotations;
      this.annotationLayers.forEach(function (layer) {
        layer.render();
      });
    };

    return AnnotationLayerBuilder;
  }
);

/* Copyright 2012 Mozilla Foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * Extracted from PDFJS viewer.js by Atlassian to be AMD compatible
 * and to remove unneeded features
 */
define('pdf-viewer/cache',
  [
    'pdf-viewer/viewer-properties'
  ],
  function (
    viewerProperties
    ) {
    var Cache = function cacheCache (size) {
      var data = [];
      this.push = function cachePush (view) {
        var i = data.indexOf(view);
        if (i >= 0) {
          data.splice(i, 1);
        }
        data.push(view);
        if (data.length > size) {
          data.shift().destroy();
        }
      };
      this.resize = function (newSize) {
        size = newSize;
        while (data.length > size) {
          data.shift().destroy();
        }
      };
    };

    return new Cache(viewerProperties.DEFAULT_CACHE_SIZE);
  });

/* Copyright 2012 Mozilla Foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * Extracted from PDFJS viewer.js by Atlassian to be AMD compatible
 * and to remove unneeded features
 */
define('pdf-viewer/page-view-scroll',
  [],
  function () {
    return {};
  });
/* Copyright 2012 Mozilla Foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * Extracted from PDFJS viewer.js by Atlassian to be AMD compatible
 * and to remove unneeded features
 */
define('pdf-viewer/utils',
  [],
  function () {

    /**
     *  Approximates float number as a fraction using Farey sequence (max order
     *  of 8).
     *  @param {number} x - Positive float number.
     *  @returns {Array} Estimated fraction: the first array item is a numerator,
     *                   the second one is a denominator.
     */
    var approximateFraction = function roundToDivide (x) {
      // Fast paths for int numbers or their inversions.
      if (Math.floor(x) === x) {
        return [x, 1];
      }
      var xinv = 1 / x;
      var limit = 8;
      if (xinv > limit) {
        return [1, limit];
      } else  if (Math.floor(xinv) === xinv) {
        return [1, xinv];
      }

      var x_ = x > 1 ? xinv : x;
      // a/b and c/d are neighbours in Farey sequence.
      var a = 0, b = 1, c = 1, d = 1;
      // Limiting search to order 8.

      /* eslint-disable no-constant-condition */
      while (true) {
        // Generating next term in sequence (order of q).
        var p = a + c, q = b + d;
        if (q > limit) {
          break;
        }
        if (x_ <= p / q) {
          c = p; d = q;
        } else {
          a = p; b = q;
        }
      }
      /* eslint-enable no-constant-condition */

      // Select closest of the neighbours to x.
      if (x_ - a / b < c / d - x_) {
        return x_ === x ? [a, b] : [b, a];
      } else {
        return x_ === x ? [c, d] : [d, c];
      }
    };

    var roundToDivide = function roundToDivide (x, div) {
      var r = x % div;
      return r === 0 ? x : Math.round(x - r + div);
    };

    /**
     * Returns scale factor for the canvas. It makes sense for the HiDPI displays.
     * @return {Object} The object with horizontal (sx) and vertical (sy)
     scales. The scaled property is set to false if scaling is
     not required, true otherwise.
     */
    var getOutputScale = function getOutputScale (ctx) {
      var devicePixelRatio = window.devicePixelRatio || 1;
      var backingStoreRatio = ctx.webkitBackingStorePixelRatio ||
        ctx.mozBackingStorePixelRatio ||
        ctx.msBackingStorePixelRatio ||
        ctx.oBackingStorePixelRatio ||
        ctx.backingStorePixelRatio || 1;
      var pixelRatio = devicePixelRatio / backingStoreRatio;
      return {
        sx: pixelRatio,
        sy: pixelRatio,
        scaled: pixelRatio != 1
      };
    };

    return {
      approximateFraction: approximateFraction,
      roundToDivide: roundToDivide,
      getOutputScale: getOutputScale
    };
  });

/* Copyright 2012 Mozilla Foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * Extracted from PDFJS viewer.js by Atlassian to be AMD compatible
 * and to remove unneeded features
 */
define('pdf-viewer/page-view',
  [
    'pdf-viewer/utils',
    'pdf-viewer/viewer-properties',
    'pdf-viewer/rendering-states',
    'pdf-viewer/presentation-mode',
    'pdf-viewer/text-layer-builder',
    'pdf-viewer/cache'
  ],
  function (
    viewerUtils,
    viewerProperties,
    renderingStates,
    PresentationMode,
    TextLayerBuilder,
    cache
  ) {

    var PDFJS = window.PDFJS;
    var CustomStyle = PDFJS.CustomStyle;

    /**
     * Scrolls specified element into view of its parent.
     * element {Object} The element to be visible.
     * containerEl {Object} The elements scroll parent.
     * spot {Object} An object with optional top and left properties,
     *         specifying the offset from the top left edge.
     */
    function scrollIntoView (element, containerEl, spot) {
      // Assuming offsetParent is available (it's not available when viewer is in
      // hidden iframe or object). We have to scroll: if the offsetParent is not set
      // producing the error. See also animationStartedClosure.
      var parent = containerEl;
      var offsetY = element.offsetTop + element.clientTop;
      var offsetX = element.offsetLeft + element.clientLeft;
      if (!parent) {
        console.error('container element is not set -- cannot scroll');
        return;
      }

      while (parent.clientHeight === parent.scrollHeight) {
        if (parent.dataset._scaleY) {
          offsetY /= parent.dataset._scaleY;
          offsetX /= parent.dataset._scaleX;
        }
        offsetY += parent.offsetTop;
        offsetX += parent.offsetLeft;
        parent = parent.offsetParent;
        if (!parent) {
          return; // no need to scroll
        }
      }
      if (spot) {
        if (spot.top !== undefined) {
          offsetY += spot.top;
        }
        if (spot.left !== undefined) {
          offsetX += spot.left;
          parent.scrollLeft = offsetX;
        }
      }
      parent.scrollTop = offsetY;
    }

    var PageView = function pageView (container, id, scale, defaultViewport, parentViewer) {
      this.id = id;

      this.parentViewer = parentViewer;

      this.rotation = 0;
      this.scale = scale || 1.0;
      this.viewport = defaultViewport;
      this.pdfPageRotate = defaultViewport.rotation;

      this.renderingState = renderingStates.INITIAL;
      this.resume = null;

      this.textLayer = null;

      this.layers = [];

      this.zoomLayer = null;

      this.annotationLayer = null; // custom layer for Confluence annotations pins
      this.linkLayer = null; // contains links refered to as AnnotationLayer in PDFJS

      var anchor = document.createElement('a');
      anchor.name = '' + this.id;

      var div = this.el = document.createElement('div');
      div.id = 'pageContainer' + this.id;
      div.setAttribute('data-page-number', this.id);
      div.className = 'page';
      div.style.width = Math.floor(this.viewport.width) + 'px';
      div.style.height = Math.floor(this.viewport.height) + 'px';

      container.appendChild(anchor);
      container.appendChild(div);

      this.setPdfPage = function pageViewSetPdfPage (pdfPage) {
        this.pdfPage = pdfPage;
        this.pdfPageRotate = pdfPage.rotate;
        var totalRotation = (this.rotation + this.pdfPageRotate) % 360;
        this.viewport = pdfPage.getViewport(this.scale * viewerProperties.CSS_UNITS, totalRotation);
        this.stats = pdfPage.stats;
        this.reset();
      };

      this.destroy = function pageViewDestroy () {
        this.zoomLayer = null;
        this.reset();
        if (this.pdfPage) {
          this.pdfPage.destroy();
        }
      };

      this.reset = function pageViewReset (keepAnnotations, keepLinks) {
        if (this.renderTask) {
          this.renderTask.cancel();
        }
        this.resume = null;
        this.renderingState = renderingStates.INITIAL;

        div.style.width = Math.floor(this.viewport.width) + 'px';
        div.style.height = Math.floor(this.viewport.height) + 'px';

        var childNodes = div.childNodes;
        for (var i = div.childNodes.length - 1; i >= 0; i--) {
          var node = childNodes[i];
          if ((this.zoomLayer && this.zoomLayer === node) ||
            (keepAnnotations && this.annotationLayer === node) ||
            (keepLinks && this.linkLayer === node)) {
            continue;
          }
          div.removeChild(node);
        }
        div.removeAttribute('data-loaded');

        if (keepAnnotations) {
          if (this.annotationLayer) {
            // Hide annotationLayer until all elements are resized
            // so they are not displayed on the already-resized page
            this.annotationLayer.setAttribute('hidden', 'true');
          }
        } else {
          this.annotationLayer = null;
        }

        if (!keepLinks) {
          this.linkLayer = null;
        }

        delete this.canvas;
        this.loadingIconDiv = document.createElement('div');
        this.loadingIconDiv.className = 'loadingIcon';
        div.appendChild(this.loadingIconDiv);
      };

      this.update = function pageViewUpdate (scale, rotation) {
        this.scale = scale || this.scale;

        if (typeof rotation !== 'undefined') {
          this.rotation = rotation;
        }

        var totalRotation = (this.rotation + this.pdfPageRotate) % 360;
        this.viewport = this.viewport.clone({
          scale: this.scale * viewerProperties.CSS_UNITS,
          rotation: totalRotation
        });

        if (viewerProperties.USE_ONLY_CSS_ZOOM && this.canvas) {
          this.cssTransform(this.canvas);
          return;
        } else if (this.canvas && !this.zoomLayer) {
          this.zoomLayer = this.canvas.parentNode;
          this.zoomLayer.style.position = 'absolute';
        }
        if (this.zoomLayer) {
          this.cssTransform(this.zoomLayer.firstChild);
        }
        this.reset(true, true);
      };

      this.cssTransform = function pageCssTransform (canvas) {
        // Scale canvas, canvas wrapper, and page container.
        var width = this.viewport.width;
        var height = this.viewport.height;
        canvas.style.width = canvas.parentNode.style.width = div.style.width =
          Math.floor(width) + 'px';
        canvas.style.height = canvas.parentNode.style.height = div.style.height =
          Math.floor(height) + 'px';
        // The canvas may have been originally rotated, so rotate relative to that.
        var relativeRotation = this.viewport.rotation - canvas._viewport.rotation;
        var absRotation = Math.abs(relativeRotation);
        var scaleX = 1, scaleY = 1;
        if (absRotation === 90 || absRotation === 270) {
          // Scale x and y because of the rotation.
          scaleX = height / width;
          scaleY = width / height;
        }
        var cssTransform = 'rotate(' + relativeRotation + 'deg) ' +
          'scale(' + scaleX + ',' + scaleY + ')';
        CustomStyle.setProp('transform', canvas, cssTransform);

        if (this.textLayer) {
          // Rotating the text layer is more complicated since the divs inside the
          // the text layer are rotated.
          // TODO: This could probably be simplified by drawing the text layer in
          // one orientation then rotating overall.
          var textRelativeRotation = this.viewport.rotation -
            this.textLayer.viewport.rotation;
          var textAbsRotation = Math.abs(textRelativeRotation);
          var scale = (width / canvas.width);
          if (textAbsRotation === 90 || textAbsRotation === 270) {
            scale = width / canvas.height;
          }
          var textLayerDiv = this.textLayer.textLayerDiv;
          var transX, transY;
          switch (textAbsRotation) {
            case 0:
              transX = transY = 0;
              break;
            case 90:
              transX = 0;
              transY = '-' + textLayerDiv.style.height;
              break;
            case 180:
              transX = '-' + textLayerDiv.style.width;
              transY = '-' + textLayerDiv.style.height;
              break;
            case 270:
              transX = '-' + textLayerDiv.style.width;
              transY = 0;
              break;
            default:
              console.error('Bad rotation value.');
              break;
          }
          CustomStyle.setProp('transform', textLayerDiv,
            'rotate(' + textAbsRotation + 'deg) ' +
              'scale(' + scale + ', ' + scale + ') ' +
              'translate(' + transX + ', ' + transY + ')');
          CustomStyle.setProp('transformOrigin', textLayerDiv, '0% 0%');
        }

        setupInternalLinks(div, this.pdfPage, this.viewport);
      };

      Object.defineProperty(this, 'width', {
        get: function PageView_getWidth () {
          return this.viewport.width;
        },
        enumerable: true
      });

      Object.defineProperty(this, 'height', {
        get: function PageView_getHeight () {
          return this.viewport.height;
        },
        enumerable: true
      });

      this.getPagePoint = function pageViewGetPagePoint (x, y) {
        return this.viewport.convertToPdfPoint(x, y);
      };

      this.scrollIntoView = function pageViewScrollIntoView (dest) {
        if (PresentationMode.active) {
          if (this.parentViewer.page !== this.id) {
            // Avoid breaking parentViewer.getVisiblePages in presentation mode.
            this.parentViewer.page = this.id;
          }
          dest = null;
          this.parentViewer.setScale(this.parentViewer.currentScaleValue, true, true);
        }
        if (!dest) {
          scrollIntoView(div, this.parentViewer.el.container);
          return;
        }

        var x = 0, y = 0;
        var width = 0, height = 0, widthScale, heightScale;
        var changeOrientation = (this.rotation % 180 === 0 ? false : true);
        var pageWidth = (changeOrientation ? this.height : this.width) /
          this.scale / viewerProperties.CSS_UNITS;
        var pageHeight = (changeOrientation ? this.width : this.height) /
          this.scale / viewerProperties.CSS_UNITS;
        var scale = 0;
        if (!dest[1]) dest[1] = '';
        switch (dest[1].name) {
          case 'XYZ':
            x = dest[2];
            y = dest[3];
            scale = dest[4];
            // If x and/or y coordinates are not supplied, default to
            // _top_ left of the page (not the obvious bottom left,
            // since aligning the bottom of the intended page with the
            // top of the window is rarely helpful).
            x = x !== null ? x : 0;
            y = y !== null ? y : pageHeight;
            break;
          case 'Fit':
          case 'FitB':
            scale = 'page-fit';
            break;
          case 'FitH':
          case 'FitBH':
            y = dest[2];
            scale = 'page-width';
            break;
          case 'FitV':
          case 'FitBV':
            x = dest[2];
            width = pageWidth;
            height = pageHeight;
            scale = 'page-height';
            break;
          case 'FitR':
            x = dest[2];
            y = dest[3];
            width = dest[4] - x;
            height = dest[5] - y;
            widthScale = (this.parentViewer.viewer.clientWidth - viewerProperties.VERTICAL_PADDING) /
              width / viewerProperties.CSS_UNITS;
            heightScale = (this.parentViewer.viewer.clientHeight - viewerProperties.VERTICAL_PADDING) /
              height / viewerProperties.CSS_UNITS;
            scale = Math.min(Math.abs(widthScale), Math.abs(heightScale));
            break;
          default:
            return;
        }

        if (scale && scale !== this.parentViewer.currentScale) {
          this.parentViewer.setScale(scale, true, true);
        } else if (this.parentViewer.currentScale === viewerProperties.UNKNOWN_SCALE) {
          this.parentViewer.setScale(viewerProperties.DEFAULT_SCALE, true, true);
        }

        if (scale === 'page-fit' && !dest[4]) {
          scrollIntoView(div, this.parentViewer.el.container);
          return;
        }

        var boundingRect = [
          this.viewport.convertToViewportPoint(x, y),
          this.viewport.convertToViewportPoint(x + width, y + height)
        ];
        var left = Math.min(boundingRect[0][0], boundingRect[1][0]);
        var top = Math.min(boundingRect[0][1], boundingRect[1][1]);

        scrollIntoView(div, this.parentViewer.el.container, { left: left, top: top });
      };

      this.getTextContent = function pageviewGetTextContent () {
        return this.parentViewer.getPage(this.id).then(function (pdfPage) {
          return pdfPage.getTextContent();
        });
      };

      this.draw = function pageviewDraw (callback) {
        var pdfPage = this.pdfPage;
        // It's important that linkLayer and annotationLayer stay on top
        // of the other layers. In case they don't exist insertBefore with
        // referenceNode = null will append.
        var beforeLayer = this.annotationLayer || this.linkLayer || null;

        if (this.pagePdfPromise) {
          return;
        }
        if (!pdfPage) {
          var promise = this.parentViewer.getPage(this.id);
          promise.then(function (pdfPage) {
            delete this.pagePdfPromise;
            this.setPdfPage(pdfPage);
            this.draw(callback);
          }.bind(this));
          this.pagePdfPromise = promise;
          return;
        }

        if (this.renderingState !== renderingStates.INITIAL) {
          console.error('Must be in new state before drawing');
        }

        this.renderingState = renderingStates.RUNNING;

        var viewport = this.viewport;
        // Wrap the canvas so if it has a css transform for highdpi the overflow
        // will be hidden in FF.
        var canvasWrapper = document.createElement('div');
        canvasWrapper.style.width = div.style.width;
        canvasWrapper.style.height = div.style.height;
        canvasWrapper.classList.add('canvasWrapper');

        var canvas = document.createElement('canvas');
        canvas.id = 'page' + this.id;
        // Keep the canvas hidden until the first draw callback, or until drawing
        // is complete when `!this.renderingQueue`, to prevent black flickering.
        canvas.setAttribute('hidden', 'hidden');
        var isCanvasHidden = true;
        canvasWrapper.appendChild(canvas);

        div.insertBefore(canvasWrapper, beforeLayer);

        this.canvas = canvas;

        canvas.mozOpaque = true;
        var ctx = canvas.getContext('2d', {alpha: false});
        var outputScale = viewerUtils.getOutputScale(ctx);

        if (viewerProperties.USE_ONLY_CSS_ZOOM) {
          var actualSizeViewport = viewport.clone({ scale: viewerProperties.CSS_UNITS });
          // Use a scale that will make the canvas be the original intended size
          // of the page.
          outputScale.sx *= actualSizeViewport.width / viewport.width;
          outputScale.sy *= actualSizeViewport.height / viewport.height;
          outputScale.scaled = true;
        }

        var sfx = viewerUtils.approximateFraction(outputScale.sx);
        var sfy = viewerUtils.approximateFraction(outputScale.sy);
        canvas.width = viewerUtils.roundToDivide(viewport.width * outputScale.sx, sfx[0]);
        canvas.height = viewerUtils.roundToDivide(viewport.height * outputScale.sy, sfy[0]);
        canvas.style.width = viewerUtils.roundToDivide(viewport.width, sfx[1]) + 'px';
        canvas.style.height = viewerUtils.roundToDivide(viewport.height, sfy[1]) + 'px';
        // Add the viewport so it's known what it was originally drawn with.
        canvas._viewport = viewport;

        var textLayerDiv = null;
        if (!PDFJS.disableTextLayer) {
          textLayerDiv = document.createElement('div');
          textLayerDiv.className = 'textLayer';
          textLayerDiv.style.width = canvas.style.width;
          textLayerDiv.style.height = canvas.style.height;

          div.insertBefore(textLayerDiv, beforeLayer);
        }
        var textLayer = this.textLayer =
          textLayerDiv ? new TextLayerBuilder({
            textLayerDiv: textLayerDiv,
            pageIndex: this.id - 1,
            lastScrollSource: parentViewer,
            viewport: this.viewport,
            isViewerInPresentationMode: PresentationMode.active
          }) : null;

        if (!this.linkLayer) {
          // Adding linkLayer before layerBuilders to ensure proper
          // order of the layers.
          var linkLayerDiv = document.createElement('div');
          linkLayerDiv.className = 'linkLayer';
          div.appendChild(linkLayerDiv);
          this.linkLayer = linkLayerDiv;
        }

        // when in presentation mode, we don't want to render any layers (i.e annotation) on top of pdf pages
        if (parentViewer.layerBuilders && !PresentationMode.active) {
          for (var i = 0; i < parentViewer.layerBuilders.length; i++) {
            var layerDiv = document.createElement('div');
            layerDiv.style.width = canvas.style.width;
            layerDiv.style.height = canvas.style.height;

            var LayerBuilder = parentViewer.layerBuilders[i];
            var layer = new LayerBuilder({
              layerDiv: layerDiv,
              pageIndex: this.id - 1,
              lastScrollSource: parentViewer,
              viewport: this.viewport,
              isViewerInPresentationMode: PresentationMode.active
            });
            this.layers.push(layer);
            div.appendChild(layerDiv);
            layer.setupRenderLayoutTimer && layer.setupRenderLayoutTimer();
          }
        }

        // Rendering area

        var self = this;
        function pageViewDrawCallback (error) {
          // The renderTask may have been replaced by a new one, so only remove the
          // reference to the renderTask if it matches the one that is triggering
          // this callback.
          if (renderTask === self.renderTask) {
            self.renderTask = null;
          }

          if (error === 'cancelled') {
            return;
          }

          self.renderingState = renderingStates.FINISHED;

          if (isCanvasHidden) {
            self.canvas.removeAttribute('hidden');
            isCanvasHidden = false;
          }

          if (self.loadingIconDiv) {
            div.removeChild(self.loadingIconDiv);
            delete self.loadingIconDiv;
          }

          if (self.zoomLayer) {
            div.removeChild(self.zoomLayer);
            self.zoomLayer = null;
          }

          // @todo: send event specifying the error

          if (self.onAfterDraw) {
            self.onAfterDraw();
          }

          cache.push(self);

          var event = document.createEvent('CustomEvent');
          event.initCustomEvent('pagerender', true, true, {
            pageNumber: pdfPage.pageNumber
          });
          div.dispatchEvent(event);

          callback();
        }

        var transform = !outputScale.scaled ? null :
          [outputScale.sx, 0, 0, outputScale.sy, 0, 0];

        var renderContext = {
          canvasContext: ctx,
          transform: transform,
          viewport: this.viewport,
          textLayer: textLayer,
          // intent: 'default', // === 'display'
          continueCallback: function pdfViewcContinueCallback (cont) {
            if (self.parentViewer.highestPriorityPage !== 'page' + self.id) {
              self.renderingState = renderingStates.PAUSED;
              self.resume = function resumeCallback () {
                self.renderingState = renderingStates.RUNNING;
                cont();
              };
              return;
            }
            if (isCanvasHidden) {
              self.canvas.removeAttribute('hidden');
              isCanvasHidden = false;
            }
            cont();
          }
        };

        var renderTask = this.renderTask = this.pdfPage.render(renderContext);

        this.renderTask.promise.then(
          function pdfPageRenderCallback () {
            pageViewDrawCallback(null);
            if (textLayer) {
              self.getTextContent().then(
                function textContentResolved (textContent) {
                  textLayer.setTextContent(textContent);
                }
              );
            }
          },
          function pdfPageRenderError (error) {
            pageViewDrawCallback(error);
          }
        );

        setupInternalLinks(div, pdfPage, this.viewport);
        div.setAttribute('data-loaded', true);
      };

      var setupInternalLinks = function setupInternalLinks (pageDiv, pdfPage, viewport) {
        var self = this;
        var pdfViewer = self.parentViewer;

        pdfPage.getAnnotations().then(function (annotationsData) {
          var PDFJS = window.PDFJS;
          viewport = viewport.clone({ dontFlip: true });

          if (self.linkLayer && self.linkLayer.childElementCount > 0) {
            PDFJS.AnnotationLayer.update({
              viewport: viewport,
              div: self.linkLayer,
              page: pdfPage,
              annotations: annotationsData,
              linkService: pdfViewer.linkService
            });
          } else {
            if (annotationsData.length === 0) {
              return;
            }

            if (!self.linkLayer) {
              self.linkLayer = document.createElement('div');
              self.linkLayer.className = 'linkLayer';
              pageDiv.appendChild(self.linkLayer);
            }

            self.linkLayer.innerHTML = '';
            PDFJS.AnnotationLayer.render({
              viewport: viewport,
              div: self.linkLayer,
              page: pdfPage,
              annotations: annotationsData,
              linkService: pdfViewer.linkService
            });
          }
        });
      }.bind(this);
    };

    return PageView;
  });

define('pdf-viewer/pdf-link-service',
  ['url'],
  function (urlHelper) {

    'use strict';

    /* global Promise */
    // At a point where PDFFJS is available its own Promise polyfill should be
    // available

    /**
     * @constructs PDFLinkService
     */
    function PDFLinkService () {
      this.baseUrl = null;
      this.pdfDocument = null;
      this.pdfViewer = null;
      this.pdfHistory = null;

      this._pagesRefCache = null;
    }

    PDFLinkService.prototype = {
      setDocument: function PDFLinkService_setDocument (pdfDocument, baseUrl) {
        this.baseUrl = baseUrl;
        this.pdfDocument = pdfDocument;
        this._pagesRefCache = Object.create(null);
      },

      setViewer: function PDFLinkService_setViewer (pdfViewer) {
        this.pdfViewer = pdfViewer;
      },

      setHistory: function PDFLinkService_setHistory (pdfHistory) {
        this.pdfHistory = pdfHistory;
      },

      /**
       * @returns {number}
       */
      get pagesCount () {
        return this.pdfDocument.numPages;
      },

      /**
       * @returns {number}
       */
      get page () {
        return this.pdfViewer.currentPageNumber;
      },

      /**
       * @param {number} value
       */
      set page (value) {
        this.pdfViewer.currentPageNumber = value;
      },

      /**
       * @param dest - The PDF destination object.
       */
      navigateTo: function PDFLinkService_navigateTo (dest) {
        var destString = '';
        var self = this;

        var goToDestination = function (destRef) {
          // dest array looks like that: <page-ref> </XYZ|FitXXX> <args..>
          var pageNumber = destRef instanceof Object ?
          self._pagesRefCache[destRef.num + ' ' + destRef.gen + ' R'] :
          (destRef + 1);
          if (pageNumber) {
            if (pageNumber > self.pagesCount) {
              pageNumber = self.pagesCount;
            }
            self.pdfViewer.scrollPageIntoView(pageNumber, dest);

            if (self.pdfHistory) {
              // Update the browsing history.
              self.pdfHistory.push({
                dest: dest,
                hash: destString,
                page: pageNumber
              });
            }
          } else {
            self.pdfDocument.getPageIndex(destRef).then(function (pageIndex) {
              var pageNum = pageIndex + 1;
              var cacheKey = destRef.num + ' ' + destRef.gen + ' R';
              self._pagesRefCache[cacheKey] = pageNum;
              goToDestination(destRef);
            });
          }
        };

        var destinationPromise;
        if (typeof dest === 'string') {
          destString = dest;
          destinationPromise = this.pdfDocument.getDestination(dest);
        } else {
          destinationPromise = Promise.resolve(dest);
        }
        destinationPromise.then(function (destination) {
          dest = destination;
          if (!(destination instanceof Array)) {
            return; // invalid destination
          }
          goToDestination(destination[0]);
        });
      },

      /**
       * @param dest - The PDF destination object.
       * @returns {string} The hyperlink to the PDF object.
       */
      getDestinationHash: function PDFLinkService_getDestinationHash (dest) {
        if (typeof dest === 'string') {
          return this.getAnchorUrl('#' + escape(dest));
        }
        if (dest instanceof Array) {
          var destRef = dest[0]; // see navigateTo method for dest format
          var pageNumber = destRef instanceof Object ?
            this._pagesRefCache[destRef.num + ' ' + destRef.gen + ' R'] :
            (destRef + 1);
          if (pageNumber) {
            var pdfOpenParams = this.getAnchorUrl('#page=' + pageNumber);
            var destKind = dest[1];
            if (typeof destKind === 'object' && 'name' in destKind &&
              destKind.name === 'XYZ') {
              var scale = (dest[4] || this.pdfViewer.currentScaleValue);
              var scaleNumber = parseFloat(scale);
              if (scaleNumber) {
                scale = scaleNumber * 100;
              }
              pdfOpenParams += '&zoom=' + scale;
              if (dest[2] || dest[3]) {
                pdfOpenParams += ',' + (dest[2] || 0) + ',' + (dest[3] || 0);
              }
            }
            return pdfOpenParams;
          }
        }
        return this.getAnchorUrl('');
      },

      /**
       * Prefix the full url on anchor links to make sure that links are resolved
       * relative to the current URL instead of the one defined in <base href>.
       * @param {String} anchor The anchor hash, including the #.
       * @returns {string} The hyperlink to the PDF object.
       */
      getAnchorUrl: function PDFLinkService_getAnchorUrl (anchor) {
        return (this.baseUrl || '') + anchor;
      },

      /**
       * @param {string} hash
       */
      setHash: function PDFLinkService_setHash (hash) {
        if (hash.indexOf('=') >= 0) {
          var params = urlHelper.parseQueryString(hash);
          // borrowing syntax from 'Parameters for Opening PDF Files'
          if ('nameddest' in params) {
            if (this.pdfHistory) {
              this.pdfHistory.updateNextHashParam(params.nameddest);
            }
            this.navigateTo(params.nameddest);
            return;
          }
          var pageNumber, dest;
          if ('page' in params) {
            pageNumber = (params.page | 0) || 1;
          }
          if ('zoom' in params) {
            // Build the destination array.
            var zoomArgs = params.zoom.split(','); // scale,left,top
            var zoomArg = zoomArgs[0];
            var zoomArgNumber = parseFloat(zoomArg);

            if (zoomArg.indexOf('Fit') === -1) {
              // If the zoomArg is a number, it has to get divided by 100. If it's
              // a string, it should stay as it is.
              dest = [null, { name: 'XYZ' },
                zoomArgs.length > 1 ? (zoomArgs[1] | 0) : null,
                zoomArgs.length > 2 ? (zoomArgs[2] | 0) : null,
                (zoomArgNumber ? zoomArgNumber / 100 : zoomArg)];
            } else {
              if (zoomArg === 'Fit' || zoomArg === 'FitB') {
                dest = [null, { name: zoomArg }];
              } else if ((zoomArg === 'FitH' || zoomArg === 'FitBH') ||
                   (zoomArg === 'FitV' || zoomArg === 'FitBV')) {
                dest = [null, { name: zoomArg },
                  zoomArgs.length > 1 ? (zoomArgs[1] | 0) : null];
              } else if (zoomArg === 'FitR') {
                if (zoomArgs.length !== 5) {
                  console.error('PDFLinkService_setHash: ' +
                      'Not enough parameters for \'FitR\'.');
                } else {
                  dest = [null, { name: zoomArg },
                  (zoomArgs[1] | 0), (zoomArgs[2] | 0),
                  (zoomArgs[3] | 0), (zoomArgs[4] | 0)];
                }
              } else {
                console.error('PDFLinkService_setHash: \'' + zoomArg +
                    '\' is not a valid zoom value.');
              }
            }
          }
          if (dest) {
            this.pdfViewer.scrollPageIntoView(pageNumber || this.page, dest);
          } else if (pageNumber) {
            this.page = pageNumber; // simple page
          }
          if ('pagemode' in params) {
            var event = document.createEvent('CustomEvent');
            event.initCustomEvent('pagemode', true, true, {
              mode: params.pagemode
            });
            this.pdfViewer.container.dispatchEvent(event);
          }
        } else if (/^\d+$/.test(hash)) { // page number
          this.page = hash;
        } else { // named destination
          if (this.pdfHistory) {
            this.pdfHistory.updateNextHashParam(unescape(hash));
          }
          this.navigateTo(unescape(hash));
        }
      },

      /**
       * @param {string} action
       */
      executeNamedAction: function PDFLinkService_executeNamedAction (action) {
        // See PDF reference, table 8.45 - Named action
        switch (action) {
          case 'GoBack':
            if (this.pdfHistory) {
              this.pdfHistory.back();
            }
            break;

          case 'GoForward':
            if (this.pdfHistory) {
              this.pdfHistory.forward();
            }
            break;

          case 'NextPage':
            this.page++;
            break;

          case 'PrevPage':
            this.page--;
            break;

          case 'LastPage':
            this.page = this.pagesCount;
            break;

          case 'FirstPage':
            this.page = 1;
            break;

          default:
            break; // No action according to spec
        }

        var event = document.createEvent('CustomEvent');
        event.initCustomEvent('namedaction', true, true, {
          action: action
        });
        this.pdfViewer.container.dispatchEvent(event);
      },

      /**
       * @param {number} pageNum - page number.
       * @param {Object} pageRef - reference to the page.
       */
      cachePageRef: function PDFLinkService_cachePageRef (pageNum, pageRef) {
        var refStr = pageRef.num + ' ' + pageRef.gen + ' R';
        this._pagesRefCache[refStr] = pageNum;
      }
    };

    return PDFLinkService;
  });

define('pdf-view',
  [
    'ajs',
    'backbone',
    'jquery',
    'pdf-viewer/viewer-properties',
    'pdf-viewer/pdf-viewer',
    'pdf-viewer/page-view-scroll',
    'pdf-viewer/presentation-mode',
    'pdf-viewer/pdf-link-service',
    'annotation-layer-builder',
    'BaseViewer',
    'template-store-singleton'
  ],
  function (
    AJS,
    Backbone,
    $,
    viewerProperties,
    PDFViewer,
    pageViewScroll,
    PresentationMode,
    PDFLinkService,
    AnnotationLayerBuilder,
    BaseViewer,
    templateStore
  ) {

    'use strict';

    var PDFJS = window.PDFJS;

    PDFJS.externalLinkTarget = PDFJS.LinkTarget.BLANK;

    var when = function (val) {
      if (val && val.then) {
        return val;
      }
      return new $.Deferred().resolve(val).promise();
    };

    var SCALE = {
      PAGE_WIDTH : 'page-width',
      PAGE_HEIGHT : 'page-height',
      PAGE_FIT: 'page-fit',
      AUTO: 'auto',
      PAGE_ACTUAL: 'page-actual',
      CUSTOM: 'custom'
    };

    /**
     * Create a new error from the error PDFJS throws when it fails.
     * @param  {Error} pdfjsErr PDFJS error object
     * @return {Error}
     */
    var createPdfError = function (pdfjsErr, model) {
      var err   = new Error(pdfjsErr.toString());
      var reason  = pdfjsErr.message.split(':')[0];
      switch (reason) {
        case 'PasswordException':
          err.title   = AJS.I18n.getText('cp.error.pdf.password.header');
          err.description = AJS.I18n.getText('cp.error.pdf.password.message');
          err.download  = true;
          break;
        case 'InvalidPDFException':
          err.title   = AJS.I18n.getText('cp.error.pdf.invalid.header');
          err.description = AJS.I18n.getText('cp.error.pdf.invalid.message');
          err.download  = true;
          break;
        case 'MissingPDFException':
        case 'UnexpectedResponseException':
          err.title   = AJS.I18n.getText('cp.error.pdf.missing.header');
          err.description = model.get('src');
          break;
        case 'UnknownErrorException':
        default:
          err.title   = AJS.I18n.getText('cp.error.pdf.default.header');
          err.description = model.get('src');
          break;
      }
      err.icon = 'cp-pdf-icon';
      return err;
    };

    var PDFView = BaseViewer.extend({

      id: 'cp-pdf-preview',

      tagName: 'div',

      initialize: function () {
        BaseViewer.prototype.initialize.apply(this, arguments);

        var linkService = this.linkService = new PDFLinkService();

        var viewer = this.viewer = new PDFViewer({
          container: document.getElementById('cp-pdf-preview'),
          viewer: document.getElementById('viewer'),
          linkService: linkService
        });

        linkService.setViewer(this.viewer);

        this.fileBody = document.getElementById('cp-file-body');

        this.scaleChangeListener = function scalechange (evt) {
          viewer.updateViewarea();
          this._fileViewer.getView().updatePaginationButtons();
        }.bind(this);

        $(window).on('scalechange.pdfPreviewView', this.scaleChangeListener);

        viewer.watchScroll(this.el, pageViewScroll, viewer.updateViewarea);
      },

      teardown: function () {
        BaseViewer.prototype.teardown.apply(this);
        $(window).off('scalechange.pdfPreviewView', this.scaleChangeListener);
        this._enableScroll();
        this.viewer.watchScrollEnd(this.fileBody);
        this.stopListening();

        this.viewer.close();
      },

      getBackground: function () {
        return this.$el.add('#viewer');
      },

      scrollCenter: function () {
        var $container     = $(this.viewer.el.container);
        var containerWidth   = $container.width();
        var $currentPage   = $(this.viewer.pages[this.viewer.page].el);
        var currentPageWidth = $currentPage.width();
        if (currentPageWidth < containerWidth) {
          return;
        }
        var offsetLeft = (currentPageWidth - containerWidth) / 2;
        $container.scrollLeft(offsetLeft);
      },

      zoomFit: function () {
        if (this.viewer.currentScaleValue === SCALE.PAGE_WIDTH) {
          this.viewer.setScale(this.viewer.recentScaleValue);
          this.viewer.currentScaleValue = this.viewer.recentScaleValue;
        } else {
          this.viewer.recentScaleValue = this.viewer.currentScaleValue;
          this.viewer.setScale(SCALE.PAGE_WIDTH);
          this.viewer.currentScaleValue = SCALE.PAGE_WIDTH;
        }
      },

      zoomIn: function () {
        this.viewer.zoomIn();
        this.scrollCenter();
      },

      zoomOut: function () {
        this.viewer.zoomOut();
        this.scrollCenter();
      },

      render: function () {
        this.$el.html(templateStore.get('PDFViewer.preview')());

        this.viewer.setContainer({
          container: this.el,
          viewer: this.$el.find('#viewer')[0],
          outerContainer: this.$el.find('#outerContainer')[0]
        });

        this._openViewer(this._fileViewer.getConfig().pdfTransportFactory);

        return this;
      },

      renderAnnotations: function (PinsView) {
        var annotations = this.model.get('annotations');
        var layerBuilder = new AnnotationLayerBuilder({
          annotations: annotations,
          fileViewer: this._fileViewer,
          PinsView: PinsView
        });

        this.viewer.addLayerBuilder(layerBuilder.create());

        var that = this;
        this.listenTo(annotations, 'add remove reset sync', function () {
          layerBuilder.updateAnnotations(annotations);
        });

        this.listenTo(annotations, 'selected', function (item) {
          if (item) {
            var pageNumber = item.attributes.pageNumber;
            if (pageNumber > that.viewer.pages.length) {
              pageNumber = that.viewer.pages.length;
            }
            // Jump to page, with zoomlevel 'Fit'
            if (that.viewer.pages && that.viewer.pages[pageNumber - 1]) {
              that.viewer.pages[pageNumber - 1].scrollIntoView([null, {name: 'Fit'}]);
            }
            // @TODO: Pins are no longer animated
          }
        });
      },

      _openViewer: function (pdfTransportFactory) {
        if (!window.ArrayBuffer) {
          var err = new Error();
          err.title = AJS.I18n.getText('cp.unsupported.browser.header');
          err.description = AJS.I18n.getText('cp.unsupported.browser.download.to.view');
          err.icon = 'cp-pdf-icon';
          this.trigger('viewerFail', err);
          return;
        }

        var createPdfTransportPromise = function () {
          return when(pdfTransportFactory && pdfTransportFactory(this._fileViewer.getCurrentFile()));
        }.bind(this);

        var viewerFailed = function (oldErr) {
          var err = createPdfError(oldErr, this.model);
          this.trigger('viewerFail', err);
        }.bind(this);

        var viewerSucceeded = function (pdfTransport) {

          var defaultScale = viewerProperties[
            PresentationMode.active ? 'DEFAULT_SCALE_PRESENTATION' : 'DEFAULT_SCALE'
          ];
          var passwordLayer = this._fileViewer._view.fileContentView.getLayerForName('password');
          this.viewer.openFile(this._previewSrc, defaultScale, undefined, passwordLayer, pdfTransport)
          .then(function () {
            this.trigger('viewerReady');
          }.bind(this), viewerFailed);
        }.bind(this);

        createPdfTransportPromise()
        .done(viewerSucceeded)
        .fail(viewerFailed);
      },

      goToPreviousPage: function () {
        this.viewer.goToPreviousPage();
      },

      goToNextPage: function () {
        this.viewer.goToNextPage();
      },

      hasPreviousPage: function () {
        return this.viewer.page > 1;
      },

      hasNextPage: function () {
        return this.viewer.page < this.viewer.pages.length;
      },

      _preventDefault: function (e) {
        e = e || window.event;
        if (e.preventDefault) {
          e.preventDefault();
        }
        e.returnValue = false;
      },

      _disableScroll: function () {
        window.addEventListener('DOMMouseScroll', this._preventDefault, false);
        window.onmousewheel = document.onmousewheel = this._preventDefault;
        $('#cp-pdf-preview').addClass('hide-scrollbar');
      },

      _enableScroll: function () {
        window.removeEventListener('DOMMouseScroll', this._preventDefault, false);
        window.onmousewheel = document.onmousewheel = null;
        $('#cp-pdf-preview').removeClass('hide-scrollbar');
      },

      _setPagePadding: function () {
        var pagePadding;
        if (PresentationMode.active) {
          pagePadding = {x: 0, y: 0};
        } else {
          pagePadding = {
            x: viewerProperties.SCROLLBAR_PADDING,
            y: viewerProperties.VERTICAL_PADDING
          };
          var arrowLayer = this._fileViewer.getView().fileContentView.getLayerForName('arrows');
          if (arrowLayer.showsArrow()) {
            pagePadding.x = viewerProperties.NAVIGATION_ARROW_PADDING;
          }
        }
        this.viewer.setPagePadding(pagePadding);
      },

      setupMode: function (mode, isModeChanged) {
        if (mode === 'PRESENTATION') {
          PresentationMode.active = true;
          this._disableScroll();
        } else {
          PresentationMode.active = false;
          this._enableScroll();
        }

        this._setPagePadding();

        if (isModeChanged) {
          this._scaleGraduallyToFitPage();
        }
      },

      _scaleGraduallyToFitPage: function () {
        // When browser change to fullscreen mode, the screen size is changed many times.
        // Here we scale 10 times every 100ms to make the page scaling to full screen smoothly
        var times = 0;
        var fullScreenInProgress = setInterval(function () {
          times++;
          if (times === 11) {
            clearInterval(fullScreenInProgress);
            // Set correct page view after PresentationMode changes
            this.viewer.goToPage(this.viewer.page);
          }
          this._scaleToFitPage();
        }.bind(this), 100);
      },

      _scaleToFitPage: function () {
        this.viewer.setScale(SCALE.PAGE_FIT, true);
        this.viewer.currentScaleValue = SCALE.PAGE_FIT;
      },

      annotationOptions: {
        dropTarget: '#viewer .page .annotationLayer',
        annotationCreated: function (elem) {
          var $elem = $(elem);
          return {
            pageNumber: parseInt($elem.closest('.page').attr('data-page-number'), 10)
          };
        }
      }
    });

    return PDFView;
  });

/* Copyright 2012 Mozilla Foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * Extracted from PDFJS viewer.js by Atlassian to be AMD compatible
 * and to remove unneeded features
 */
define('pdf-viewer/pdf-viewer',
  [
    'underscore',
    'jquery',
    'ajs',
    'pdf-viewer/viewer-properties',
    'pdf-viewer/page-view',
    'pdf-viewer/rendering-states',
    'pdf-viewer/presentation-mode',
    'pdf-viewer/cache',
    'pdf-viewer/page-view-scroll'
  ],
  function (
    _,
    $,
    AJS,
    viewerProperties,
    PageView,
    renderingStates,
    PresentationMode,
    cache,
    pageViewScroll
  ) {
    /* global Promise */
    // At a point where PDFFJS is available its own Promise polyfill should be
    // available
    var PDFJS = window.PDFJS;

    function PDFViewer (options) {
      PDFJS.UnsupportedManager.listen(_.bind(PDFViewer.prototype.fallback, PDFViewer));
      PDFJS.verbosity = PDFJS.VERBOSITY_LEVELS.errors;
      PDFJS.cMapPacked = true;
      PDFJS.disableAutoFetch = true;
      this.lastScroll = 0;
      this.layerBuilders = [];
      this.pages = [];
      this._pagePadding = {
        x: viewerProperties.SCROLLBAR_PADDING,
        y: viewerProperties.VERTICAL_PADDING
      };
      this.linkService = options.linkService;
    }

    // Helper function to keep track whether a div was scrolled up or down and
    // then call a callback.
    PDFViewer.prototype.watchScroll = function pdfViewWatchScroll (viewAreaElement, state, callback) {
      state.down = true;
      state.lastY = viewAreaElement.scrollTop;
      var viewer = this;
      $(viewAreaElement).on('scroll.pdfViewer', function webViewerScroll (evt) {
        viewer.lastScroll = Date.now();
        var currentY = viewAreaElement.scrollTop;
        var lastY = state.lastY;
        if (currentY > lastY) {
          state.down = true;
        } else if (currentY < lastY) {
          state.down = false;
        }
        // else do nothing and use previous value
        state.lastY = currentY;
        callback.call(viewer);
      });
    };

    PDFViewer.prototype.watchScrollEnd = function pdfViewWatchScrollEnd (viewAreaElement) {
      $(viewAreaElement).off('scroll.pdfViewer');
    };

    PDFViewer.prototype.setContainer = function (args) {
      this.el = {
        container: args.container,
        viewer: args.viewer,
        outerContainer: args.outerContainer
      };
    };

    PDFViewer.prototype.addLayerBuilder = function (layerBuilder) {
      if (layerBuilder) {
        this.layerBuilders.push(layerBuilder);
      }
    };

    PDFViewer.prototype._setScaleUpdatePages = function pdfView_setScaleUpdatePages (
      newScale, newValue, resetAutoSettings, noScroll) {
      this.currentScaleValue = newValue;
      if (newScale === this.currentScale) {
        return;
      }
      for (var i = 0, ii = this.pages.length; i < ii; i++) {
        this.pages[i].update(newScale);
      }
      this.currentScale = newScale;

      if (!noScroll) {
        var page = this.page, dest;
        if (this.currentPosition && !viewerProperties.IGNORE_CURRENT_POSITION_ON_ZOOM) {
          page = this.currentPosition.page;
          dest = [null, { name: 'XYZ' }, this.currentPosition.left,
            this.currentPosition.top, null];
        }
        this.pages[page - 1].scrollIntoView(dest);
      }

      // CONFDEV-27407: This event shouldn't be thrown on the window, but should be tracked internally and registered via an API PDFViewer.prototype.on
      var event = document.createEvent('UIEvents');
      event.initUIEvent('scalechange', false, false, window, 0);
      event.scale = newScale;
      event.resetAutoSettings = resetAutoSettings;
      window.dispatchEvent(event);
    };

    /**
     * Set the viewers page padding
     * @param {Object}   padding
     * @param {Number}   padding.x [VERTICAL_PADDING] - Horizontal padding
     * @param {Number}   padding.y [SCROLLBAR_PADDING] - Vertical padding
     */
    PDFViewer.prototype.setPagePadding = function (padding) {
      this._pagePadding = {
        x: parseInt(padding.x, 10) || 0,
        y: parseInt(padding.y, 10) || 0
      };
    };

    PDFViewer.prototype.setScale = function pdfViewSetScale (value, resetAutoSettings, noScroll) {
      if (value === 'custom') {
        return;
      }
      var scale = parseFloat(value);

      if (scale > 0) {
        this._setScaleUpdatePages(scale, value, true, noScroll);
      } else {
        var currentPage = this.pages[this.page - 1];
        if (!currentPage) {
          return;
        }
        var hPadding = this._pagePadding.x;
        var vPadding = this._pagePadding.y;
        var pageWidthScale = (this.el.container.clientWidth - hPadding) /
          currentPage.width * currentPage.scale;
        var pageHeightScale = (this.el.container.clientHeight - vPadding) /
          currentPage.height * currentPage.scale;
        switch (value) {
          case 'page-actual':
            scale = 1;
            break;
          case 'page-width':
            scale = pageWidthScale;
            break;
          case 'page-height':
            scale = pageHeightScale;
            break;
          case 'page-fit':
            // Resolve 'page-fit' to actual scale mode
            scale = Math.min(pageWidthScale, pageHeightScale);
            if (scale === pageWidthScale) {
              value = 'page-width';
            } else {
              value = 'page-height';
            }
            break;
          case 'auto':
            scale = Math.min(viewerProperties.MAX_AUTO_SCALE, pageWidthScale);
            break;
          default:
            console.error('pdfViewSetScale: \'' + value +
              '\' is an unknown zoom value.');
            return;
        }
        this._setScaleUpdatePages(scale, value, resetAutoSettings, noScroll);
      }
    };

    PDFViewer.prototype.zoomIn = function pdfViewZoomIn (ticks) {
      var newScale = this.currentScale;
      do {
        newScale = (newScale * viewerProperties.DEFAULT_SCALE_DELTA).toFixed(2);
        newScale = Math.ceil(newScale * 10) / 10;
        newScale = Math.min(viewerProperties.MAX_SCALE, newScale);
      } while (--ticks && newScale < viewerProperties.MAX_SCALE);
      this.setScale(newScale, true);
    };

    PDFViewer.prototype.zoomOut = function pdfViewZoomOut (ticks) {
      var newScale = this.currentScale;
      do {
        newScale = (newScale / viewerProperties.DEFAULT_SCALE_DELTA).toFixed(2);
        newScale = Math.floor(newScale * 10) / 10;
        newScale = Math.max(viewerProperties.MIN_SCALE, newScale);
      } while (--ticks && newScale > viewerProperties.MIN_SCALE);
      this.setScale(newScale, true);
    };

    PDFViewer.prototype.openFile = function (url, scale, password, passwordLayer, pdfDataRangeTransport, args) {
      var parameters = {};
      if (typeof url === 'string') { // URL
        parameters.url = url;
      } else if (url && 'byteLength' in url) { // ArrayBuffer
        parameters.data = url;
      }

      if (args) {
        for (var prop in args) {
          parameters[prop] = args[prop];
        }
      }

      var self = this;
      self.loading = true;
      self.downloadComplete = false;

      self._pdfLoadingTask = PDFJS.getDocument(parameters, pdfDataRangeTransport);

      self._pdfLoadingTask.onProgress = function getDocumentProgress (progressData) {
        self.progress(progressData.loaded / progressData.total);
      };

      self._pdfLoadingTask.onPassword = function passwordNeeded (updatePassword, reason) {
        passwordLayer.showPasswordInput(reason, updatePassword);
      };

      return self._pdfLoadingTask.promise.then(
        function getDocumentCallback (pdfDocument) {
          var loadedPromise = self.load(pdfDocument, scale);
          self.loading = false;
          return loadedPromise;
        },
        function getDocumentError (message, exception) {
          // CONFDEV-27407: Extract loading response out of viewer on('loadingerror')
          self.loading = false;
          return new Promise(function (resolve, reject) {
            reject(exception || new Error(message));
          });
        }
      );
    };

    PDFViewer.prototype.close = function () {
      if (!this._pdfLoadingTask) { return; }

      this._pdfLoadingTask.then(function () {
        this.cleanup();

        this.pdfDocument.destroy();
        this.pdfDocument = null;

        // CONFDEV-27408: The code below is likely a sign of a leak suspect
        clearTimeout(this.idleTimeout);
        for (var i = 0; i < this.pages.length; i++) {
          this.pages[i].destroy();
        }
        this.pages = [];

        var container = this.el.viewer;
        while (container.hasChildNodes()) {
          container.removeChild(container.lastChild);
        }
      }.bind(this), function (error) {
        // We try to keep expected errors out of the console so
        // we ignore errors about missing PDFs in the #close() method
        if (error.name !== 'MissingPDFException') {
          throw (error);
        }
      });

      this._pdfLoadingTask.destroy();
      this.pdfLoadingTask = null;
    };

    PDFViewer.prototype.load = function (pdfDocument, scale) {
      var self = this;
      var isOnePageRenderedResolved = false;
      var resolveOnePageRendered = null;

      var onePageRendered = new Promise(function (resolve) {
        resolveOnePageRendered = resolve;
      });

      function bindOnAfterDraw (pageView) {
        pageView.onAfterDraw = function pdfViewLoadOnAfterDraw () {
          if (!isOnePageRenderedResolved) {
            isOnePageRenderedResolved = true;
            resolveOnePageRendered();
          }
        };
      }

      this.pdfDocument = pdfDocument;
      var baseDocumentUrl = null;
      this.linkService.setDocument(pdfDocument, baseDocumentUrl);

      var downloadedPromise = pdfDocument.getDownloadInfo().then(function () {
        self.downloadComplete = true;
        var outerContainer = self.el.outerContainer;
        outerContainer.classList.remove('loadingInProgress');
      });

      var pagesCount = pdfDocument.numPages;

      // @todo: send event specifying the pageCount and other metadata

      var pages = this.pages = [];
      self.pagesRefMap = {};

      var firstPagePromise = pdfDocument.getPage(1);
      var container = this.el.viewer;

      var parentViewer = this;

      // Fetch a single page so we can get a viewport that will be the default
      // viewport for all pages
      firstPagePromise.then(function (pdfPage) {
        var viewport = pdfPage.getViewport(1.0 * viewerProperties.CSS_UNITS);
        for (var pageNum = 1; pageNum <= pagesCount; ++pageNum) {
          var viewportClone = viewport.clone();
          var pageView = new PageView(container, pageNum, 0, viewportClone, parentViewer);
          bindOnAfterDraw(pageView);
          pages.push(pageView);
        }

        // Fetch all the pages since the viewport is needed before printing
        // starts to create the correct size canvas. Wait until one page is
        // rendered so we don't tie up too many resources early on.
        onePageRendered.then(function () {
          if (!PDFJS.disableAutoFetch) {
            for (var pageNum = 1; pageNum <= pagesCount; ++pageNum) {
              pdfDocument.getPage(pageNum).then(function (pageNum, pdfPage) {
                var pageView = pages[pageNum - 1];
                if (!pageView.pdfPage) {
                  pageView.setPdfPage(pdfPage);
                }
                var refStr = pdfPage.ref.num + ' ' + pdfPage.ref.gen + ' R';
                self.pagesRefMap[refStr] = pageNum;
                self.linkService.cachePageRef(pageNum, pdfPage.ref);
              }.bind(null, pageNum));
            }
          }
        });

        downloadedPromise.then(function () {
          // CONFDEV-27407: This is where our load event (non document) should be fired on('pdfloaded')
          var event = document.createEvent('CustomEvent');
          event.initCustomEvent('documentload', true, true, {});
          window.dispatchEvent(event);
        });

        return downloadedPromise;
      });

      Promise.all([firstPagePromise]).then(function resolved () {
        self.setInitialView(null, scale);
      }, function rejected (errorMsg) {

        firstPagePromise.then(function () {
          self.setInitialView(null, scale);
        });
      });

      self.destinationsPromise = pdfDocument.getDestinations();
      self.destinationsPromise.then(function (destinations) {
        self.destinations = destinations;
      });

      pdfDocument.getMetadata().then(function (data) {
        /* eslint-disable no-console*/
        var info = data.info, metadata = data.metadata;
        self.documentInfo = info;
        self.metadata = metadata;

        // Provides some basic debug information
        if (PDFJS.pdfBug) {
          var debugMsg = [];
          debugMsg.push('PDF Fingerprint: ' + pdfDocument.fingerprint);
          debugMsg.push('Version: ' + info.PDFFormatVersion);
          debugMsg.push('Producer: ' + (info.Producer || '-').trim());
          debugMsg.push('Creator: ' + (info.Creator || '-').trim());
          debugMsg.push('PDF.js: ' + (PDFJS.version || '-'));
          console.info(debugMsg.join('; '));
        }

        var pdfTitle;
        if (metadata && metadata.has('dc:title')) {
          pdfTitle = metadata.get('dc:title');
        }

        if (!pdfTitle && info && info['Title']) {
          pdfTitle = info['Title'];
        }

        if (PDFJS.pdfBug && pdfTitle) {
          console.info(pdfTitle);
        }

        if (info.IsAcroFormPresent) {
          console.warn('Warning: AcroForm/XFA is not supported');
          self.fallback(PDFJS.UNSUPPORTED_FEATURES.forms);
        }
        /* eslint-enable no-console*/
      });
    };


    PDFViewer.prototype.setInitialView = function pdfViewSetInitialView (storedHash, scale) {
      // Reset the current scale, as otherwise the page's scale might not get
      // updated if the zoom level stayed the same.
      this.currentScale = 0;
      this.currentScaleValue = null;
      // Reset the current position when loading a new file,
      // to prevent displaying the wrong position in the document.
      var currentPage = this.pages[0];
      var topLeft = currentPage.getPagePoint((this.el.viewer.scrollLeft - currentPage.x),
        (this.el.viewer.scrollTop - currentPage.y));
      var intLeft = Math.round(topLeft[0]);
      var intTop = Math.round(topLeft[1]);
      this.currentPosition = { page: 1, left: intLeft, top: intTop };

      this.page = 1;
      this.setScale(scale, true);

      if (this.currentScale === viewerProperties.UNKNOWN_SCALE) {
        // Scale was not initialized: invalid bookmark or scale was not specified.
        // Setting the default one.
        this.setScale(viewerProperties.DEFAULT_SCALE, true);
      }
    };

    PDFViewer.prototype.renderHighestPriority = function pdfViewRenderHighestPriority () {
      if (this.idleTimeout) {
        clearTimeout(this.idleTimeout);
        this.idleTimeout = null;
      }

      // Pages have a higher priority than thumbnails, so check them first.
      var visiblePages = this.getVisiblePages();
      var pageView = this.getHighestPriority(visiblePages, this.pages,
        pageViewScroll.down);
      if (pageView) {
        this.renderView(pageView, 'page');
        return;
      }

      var that = this;
      this.idleTimeout = setTimeout(function () {
        that.cleanup();
      }, viewerProperties.CLEANUP_TIMEOUT);
    };

    PDFViewer.prototype.cleanup = function pdfViewCleanup () {
      for (var i = 0, ii = this.pages.length; i < ii; i++) {
        if (this.pages[i] &&
          this.pages[i].renderingState !== renderingStates.FINISHED) {
          this.pages[i].reset();
        }
      }
      this.pdfDocument.cleanup();
    };

    PDFViewer.prototype.getHighestPriority = function pdfViewGetHighestPriority (visible, views,
                                 scrolledDown) {
      // The state has changed figure out which page has the highest priority to
      // render next (if any).
      // Priority:
      // 1 visible pages
      // 2 if last scrolled down page after the visible pages
      // 2 if last scrolled up page before the visible pages
      var visibleViews = visible.views;

      var numVisible = visibleViews.length;
      if (numVisible === 0) {
        return false;
      }
      for (var i = 0; i < numVisible; ++i) {
        var view = visibleViews[i].view;
        if (!this.isViewFinished(view)) {
          return view;
        }
      }

      // All the visible views have rendered, try to render next/previous pages.
      if (scrolledDown) {
        var nextPageIndex = visible.last.id;
        // ID's start at 1 so no need to add 1.
        if (views[nextPageIndex] && !this.isViewFinished(views[nextPageIndex])) {
          return views[nextPageIndex];
        }
      } else {
        var previousPageIndex = visible.first.id - 2;
        if (views[previousPageIndex] &&
          !this.isViewFinished(views[previousPageIndex])) {
          return views[previousPageIndex];
        }
      }
      // Everything that needs to be rendered has been.
      return false;
    };

    PDFViewer.prototype.isViewFinished = function pdfViewIsViewFinished (view) {
      return view.renderingState === renderingStates.FINISHED;
    };

    // Render a page or thumbnail view. This calls the appropriate function based
    // on the views state. If the view is already rendered it will return false.
    PDFViewer.prototype.renderView = function pdfViewRender (view, type) {
      var state = view.renderingState;
      switch (state) {
        case renderingStates.FINISHED:
          return false;
        case renderingStates.PAUSED:
          this.highestPriorityPage = type + view.id;
          view.resume();
          break;
        case renderingStates.RUNNING:
          this.highestPriorityPage = type + view.id;
          break;
        case renderingStates.INITIAL:
          this.highestPriorityPage = type + view.id;
          view.draw(this.renderHighestPriority.bind(this));
          break;
      }
      return true;
    };

    PDFViewer.prototype.getVisibleElements = function pdfViewGetVisibleElements (scrollEl, views, sortByVisibility) {
      var top = scrollEl.scrollTop, bottom = top + scrollEl.clientHeight;
      var left = scrollEl.scrollLeft, right = left + scrollEl.clientWidth;

      var visible = [], view;
      var currentHeight, viewHeight, hiddenHeight, percentHeight;
      var currentWidth, viewWidth;
      for (var i = 0, ii = views.length; i < ii; ++i) {
        view = views[i];
        currentHeight = view.el.offsetTop + view.el.clientTop;
        viewHeight = view.el.clientHeight;
        if ((currentHeight + viewHeight) < top) {
          continue;
        }
        if (currentHeight > bottom) {
          break;
        }
        currentWidth = view.el.offsetLeft + view.el.clientLeft;
        viewWidth = view.el.clientWidth;
        if ((currentWidth + viewWidth) < left || currentWidth > right) {
          continue;
        }
        hiddenHeight = Math.max(0, top - currentHeight) +
          Math.max(0, currentHeight + viewHeight - bottom);
        percentHeight = ((viewHeight - hiddenHeight) * 100 / viewHeight) | 0;

        visible.push({ id: view.id, x: currentWidth, y: currentHeight,
          view: view, percent: percentHeight });
      }

      var first = visible[0];
      var last = visible[visible.length - 1];

      if (sortByVisibility) {
        visible.sort(function (a, b) {
          var pc = a.percent - b.percent;
          if (Math.abs(pc) > 0.001) {
            return -pc;
          }
          return a.id - b.id; // ensure stability
        });
      }
      return {first: first, last: last, views: visible};
    };

    PDFViewer.prototype.getVisiblePages = function pdfViewGetVisiblePages () {
      if (!PresentationMode.active) {
        return this.getVisibleElements(this.el.container, this.pages, true);
      } else {
        // The algorithm in getVisibleElements doesn't work in all browsers and
        // configurations when presentation mode is active.
        var visible = [];
        var currentPage = this.pages[this.page - 1];
        visible.push({ id: currentPage.id, view: currentPage });
        return { first: currentPage, last: currentPage, views: visible };
      }
    };

    PDFViewer.prototype.updateViewarea = function () {
      var visible = this.getVisiblePages();
      if (!visible || visible.views.length === 0) {
        return;
      }
      var visiblePages = visible.views;

      var suggestedCacheSize = Math.max(viewerProperties.DEFAULT_CACHE_SIZE,
        2 * visiblePages.length + 1);
      cache.resize(suggestedCacheSize);

      this.renderHighestPriority(visible);

      var currentId = this.page;
      var firstPage = visible.first;

      for (var i = 0, ii = visiblePages.length, stillFullyVisible = false;
         i < ii; ++i) {
        var page = visiblePages[i];

        if (page.percent < 100) {
          break;
        }
        if (page.id === this.page) {
          stillFullyVisible = true;
          break;
        }
      }

      if (!stillFullyVisible) {
        currentId = visiblePages[0].id;
      }

      if (!PresentationMode.active) {
        this.updateViewarea.inProgress = true; // used in 'set page'
        this.page = currentId;
        this.updateViewarea.inProgress = false;
      }

      var pageNumber = firstPage.id;
      var currentPage = this.pages[pageNumber - 1];
      var container = this.el.container;
      var topLeft = currentPage.getPagePoint((container.scrollLeft - firstPage.x),
        (container.scrollTop - firstPage.y));
      var intLeft = Math.round(topLeft[0]);
      var intTop = Math.round(topLeft[1]);

      if (PresentationMode.active || PresentationMode.switchInProgress) {
        this.currentPosition = null;
      } else {
        this.currentPosition = { page: pageNumber, left: intLeft, top: intTop };
      }
    };

    PDFViewer.prototype.getPage = function pdfViewGetPage (n) {
      return this.pdfDocument.getPage(n);
    };

    PDFViewer.prototype.fallback = function () {
      //CONFDEV-27407: This should be a handler registered via PDFViewer.protoype.on
    };

    PDFViewer.prototype.progress = function () {
      //CONFDEV-27407: This should be a handler registered via PDFViewer.protoype.on
    };

    PDFViewer.prototype.goToNextPage = function () {
      this.goToPage(this.page + 1);
    };

    PDFViewer.prototype.goToPreviousPage = function () {
      this.goToPage(this.page - 1);
    };

    PDFViewer.prototype.goToPage = function (pageNumber) {
      if (this.pages && this.pages[pageNumber - 1]) {
        this.pages[pageNumber - 1].scrollIntoView([null, {name: 'Fit'}]);
      }
    };

    PDFViewer.prototype.scrollPageIntoView = function (pageNumber, dest) {
      if (!this.pdfDocument) {
        return;
      }

      var pageView = this.pages[pageNumber - 1];

      if (this.isInPresentationMode) {
        if (this._currentPageNumber !== pageView.id) {
          // Avoid breaking getVisiblePages in presentation mode.
          this.currentPageNumber = pageView.id;
          return;
        }
        dest = null;
        // Fixes the case when PDF has different page sizes.
        this._setScale(this._currentScaleValue, true);
      }
      if (!dest) {
        pageView.scrollIntoView(pageView.div, this.el.viewer);
        return;
      }

      var x = 0, y = 0;
      var width = 0, height = 0, widthScale, heightScale;
      var changeOrientation = (pageView.rotation % 180 === 0 ? false : true);
      var pageWidth = (changeOrientation ? pageView.height : pageView.width) /
      pageView.scale / viewerProperties.CSS_UNITS;
      var pageHeight = (changeOrientation ? pageView.width : pageView.height) /
      pageView.scale / viewerProperties.CSS_UNITS;
      var scale = 0;
      switch (dest[1].name) {
        case 'XYZ':
          x = dest[2];
          y = dest[3];
          scale = dest[4];
          // If x and/or y coordinates are not supplied, default to
          // _top_ left of the page (not the obvious bottom left,
          // since aligning the bottom of the intended page with the
          // top of the window is rarely helpful).
          x = x !== null ? x : 0;
          y = y !== null ? y : pageHeight;
          break;
        case 'Fit':
        case 'FitB':
          scale = 'page-fit';
          break;
        case 'FitH':
        case 'FitBH':
          y = dest[2];
          scale = 'page-width';
          // According to the PDF spec, section 12.3.2.2, a `null` value in the
          // parameter should maintain the position relative to the new page.
          if (y === null && this._location) {
            x = this._location.left;
            y = this._location.top;
          }
          break;
        case 'FitV':
        case 'FitBV':
          x = dest[2];
          width = pageWidth;
          height = pageHeight;
          scale = 'page-height';
          break;
        case 'FitR':
          x = dest[2];
          y = dest[3];
          width = dest[4] - x;
          height = dest[5] - y;
          var hPadding = this.removePageBorders ? 0 : viewerProperties.SCROLLBAR_PADDING;
          var vPadding = this.removePageBorders ? 0 : viewerProperties.VERTICAL_PADDING;

          widthScale = (this.container.clientWidth - hPadding) /
          width / viewerProperties.CSS_UNITS;
          heightScale = (this.container.clientHeight - vPadding) /
          height / viewerProperties.CSS_UNITS;
          scale = Math.min(Math.abs(widthScale), Math.abs(heightScale));
          break;
        default:
          return;
      }

      if (scale && scale !== this._currentScale) {
        this.currentScaleValue = scale;
      } else if (this._currentScale === viewerProperties.UNKNOWN_SCALE) {
        this.currentScaleValue = viewerProperties.DEFAULT_SCALE_VALUE;
      }

      if (scale === 'page-fit' && !dest[4]) {
        pageView.scrollIntoView(pageView.div, this.el.viewer);
        return;
      }

      var boundingRect = [
        pageView.viewport.convertToViewportPoint(x, y),
        pageView.viewport.convertToViewportPoint(x + width, y + height)
      ];
      var left = Math.min(boundingRect[0][0], boundingRect[1][0]);
      var top = Math.min(boundingRect[0][1], boundingRect[1][1]);

      pageView.scrollIntoView(pageView.div, this.el.viewer, { left: left, top: top });
    };

    return PDFViewer;
  }
);

/* Copyright 2012 Mozilla Foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * Extracted from PDFJS viewer.js by Atlassian to be AMD compatible
 * and to remove unneeded features
 */
define('pdf-viewer/presentation-mode',
  [],
  function () {
    // placeholder until we add presentation mode support
    return {};
  });
/* Copyright 2012 Mozilla Foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * Extracted from PDFJS viewer.js by Atlassian to be AMD compatible
 * and to remove unneeded features
 */
define('pdf-viewer/rendering-states',
  [],
  function () {
    var RenderingStates = {
      INITIAL: 0,
      RUNNING: 1,
      PAUSED: 2,
      FINISHED: 3
    };

    return RenderingStates;
  }
);
/* Copyright 2012 Mozilla Foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * Extracted from PDFJS viewer.js by Atlassian to be AMD compatible
 * and to remove unneeded features
 */
define('pdf-viewer/text-layer-builder',
  [
    'pdf-viewer/viewer-properties'
  ],
  function (
    viewerProperties
    ) {

    var PDFJS = window.PDFJS;
    var CustomStyle = PDFJS.CustomStyle;

    /**
     * TextLayerBuilder provides text-selection functionality for the PDF.
     * It does this by creating overlay divs over the PDF text. These divs
     * contain text that matches the PDF text they are overlaying. This object
     * also provides a way to highlight text that is being searched for.
     */
    function TextLayerBuilder (options) {
      this.textLayerDiv = options.textLayerDiv;
      this.layoutDone = false;
      this.divContentDone = false;
      this.pageIdx = options.pageIndex;
      this.matches = [];
      this.lastScrollSource = options.lastScrollSource || null;
      this.viewport = options.viewport;
      this.isViewerInPresentationMode = options.isViewerInPresentationMode;
      this.textDivs = [];
      this.findController = window.PDFFindController || null;
    }

    TextLayerBuilder.prototype = {
      renderLayer: function TextLayerBuilder_renderLayer () {
        var textLayerFrag = document.createDocumentFragment();
        var textDivs = this.textDivs;
        var textDivsLength = textDivs.length;
        var canvas = document.createElement('canvas');
        var ctx = canvas.getContext('2d');

        // No point in rendering many divs as it would make the browser
        // unusable even after the divs are rendered.
        if (textDivsLength > viewerProperties.MAX_TEXT_DIVS_TO_RENDER) {
          return;
        }

        for (var i = 0; i < textDivsLength; i++) {
          var textDiv = textDivs[i];
          if (textDiv.dataset.isWhitespace !== undefined) {
            continue;
          }

          ctx.font = textDiv.style.fontSize + ' ' + textDiv.style.fontFamily;
          var width = ctx.measureText(textDiv.textContent).width;
          if (width > 0) {
            textLayerFrag.appendChild(textDiv);
            var textScale = textDiv.dataset.canvasWidth / width;
            var rotation = textDiv.dataset.angle;
            var transform = 'scale(' + textScale + ', 1)';
            transform = 'rotate(' + rotation + 'deg) ' + transform;
            CustomStyle.setProp('transform', textDiv, transform);
            CustomStyle.setProp('transformOrigin', textDiv, '0% 0%');
          }
        }

        this.textLayerDiv.appendChild(textLayerFrag);
        this.renderingDone = true;
      },

      setupRenderLayoutTimer:
        function TextLayerBuilder_setupRenderLayoutTimer () {
          // Schedule renderLayout() if the user has been scrolling,
          // otherwise run it right away.
          var self = this;
          var lastScroll = (this.lastScrollSource === null ?
            0 : this.lastScrollSource.lastScroll);

          if (Date.now() - lastScroll > viewerProperties.RENDER_DELAY) { // Render right away
            this.renderLayer();
          } else { // Schedule
            if (this.renderTimer) {
              clearTimeout(this.renderTimer);
            }
            this.renderTimer = setTimeout(function () {
              self.setupRenderLayoutTimer();
            }, viewerProperties.RENDER_DELAY);
          }
        },

      appendText: function TextLayerBuilder_appendText (geom, styles) {
        var style = styles[geom.fontName];
        var textDiv = document.createElement('div');
        this.textDivs.push(textDiv);
        if (!/\S/.test(geom.str)) {
          textDiv.dataset.isWhitespace = true;
          return;
        }
        var tx = PDFJS.Util.transform(this.viewport.transform, geom.transform);
        var angle = Math.atan2(tx[1], tx[0]);
        if (style.vertical) {
          angle += Math.PI / 2;
        }
        var fontHeight = Math.sqrt((tx[2] * tx[2]) + (tx[3] * tx[3]));
        var fontAscent = (style.ascent ? style.ascent * fontHeight :
          (style.descent ? (1 + style.descent) * fontHeight : fontHeight));

        textDiv.style.position = 'absolute';
        textDiv.style.left = (tx[4] + (fontAscent * Math.sin(angle))) + 'px';
        textDiv.style.top = (tx[5] - (fontAscent * Math.cos(angle))) + 'px';
        textDiv.style.fontSize = fontHeight + 'px';
        textDiv.style.fontFamily = style.fontFamily;

        textDiv.textContent = geom.str;
        textDiv.dataset.fontName = geom.fontName;
        textDiv.dataset.angle = angle * (180 / Math.PI);
        if (style.vertical) {
          textDiv.dataset.canvasWidth = geom.height * this.viewport.scale;
        } else {
          textDiv.dataset.canvasWidth = geom.width * this.viewport.scale;
        }
      },

      setTextContent: function TextLayerBuilder_setTextContent (textContent) {
        this.textContent = textContent;

        var textItems = textContent.items;
        for (var i = 0, len = textItems.length; i < len; i++) {
          this.appendText(textItems[i], textContent.styles);
        }
        this.divContentDone = true;
        this.setupRenderLayoutTimer();
      }

    };
    return TextLayerBuilder;

  }
);

/* Copyright 2012 Mozilla Foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * Extracted from PDFJS viewer.js by Atlassian to be AMD compatible
 * and to remove unneeded features
 */
define('pdf-viewer/viewer-properties',
  [],
  function () {
    return {
      DEFAULT_SCALE: 'auto',
      DEFAULT_SCALE_PRESENTATION: 'page-fit',
      DEFAULT_SCALE_DELTA: 1.1,
      DEFAULT_CACHE_SIZE: 10,
      UNKNOWN_SCALE: 0,
      CSS_UNITS: 96.0 / 72.0,
      SCROLLBAR_PADDING: 40,
      VERTICAL_PADDING: 20,
      NAVIGATION_ARROW_PADDING: 160,
      MAX_AUTO_SCALE: 1.25,
      MIN_SCALE: 0.25,
      MAX_SCALE: 4.0,
      USE_ONLY_CSS_ZOOM: false,
      CLEANUP_TIMEOUT: 30000,
      IGNORE_CURRENT_POSITION_ON_ZOOM: false,
      RENDER_DELAY: 200, // ms
      MAX_TEXT_DIVS_TO_RENDER: 100000
    };
  });

}(function () {
  var FileViewer;

    if (typeof module !== "undefined" && ('exports' in module)) {
      FileViewer = require('./fileviewer.js');
    } else if (window.require) {
      FileViewer = window.FileViewer;
    } else {
      FileViewer = window.FileViewer;
    }

    return FileViewer;
}()));
