'use strict';

var configureGrunt = function (grunt) {

  /**
   * Loads grunt jobs from js file and parses the job tasks into grunts config
   *
   * @param  {String} file path to config file
   */
  var loadJobs = function (file) {
    var data = require(file);
    Object.keys(data.tasks).forEach(function (key) {
      if (!config.hasOwnProperty(key)) {
        config[key] = {};
      }
      config[key][data.name] = data.tasks[key];
    });
  };

  // Load local tasks
  grunt.loadTasks('support/grunt');
  // Dynamically load npm grunt modules
  require('matchdep').filterDev('grunt-*').forEach(grunt.loadNpmTasks);

  var config = {
    pkg: grunt.file.readJSON('package.json'),
    versionString: 'UNDEFINED_VERSION',
    versionDate: new Date().toUTCString(),
    versionSet: false,
    bin: {
      javaHome: process.env.JAVA_HOME + '/bin/java',
      atlassianSoy: 'support/atlassian-soy-cli-support-3.3.1-jar-with-dependencies.jar',
      closureCompiler: './support/closure-compiler.jar',
      jsDox: 'node_modules/.bin/jsdox'
    },
    paths: {
      build: 'build',
      dist: 'dist',
      pdfjs: 'vendor/pdfjs'
    },
    cssmin: {
      options: {
        shorthandCompacting: false,
        roundingPrecision: -1
      }
    },

    exec: {
      clean: {
        command: 'rm -rf <%= paths.build %>',
        stdout: false,
        stderr: false
      },
      cleandist: {
        command: 'rm -rf <%= paths.dist %>/*',
        stdout: false,
        stderr: false
      }
    },

    i18nproperties: {
      en: {
        src: 'lib/i18n/translations-en.properties',
        dest: '<%= paths.build %>/translations.en.js'
      }
    },

    copy: {
      example: {
        files: [
          {
            expand: true,
            src: [
              'vendor/aui-dist/**',
              'node_modules/jquery/dist/jquery.min.js',
              'node_modules/soyutils/soyutils_nogoog.js',
              'node_modules/underscore/underscore.js',
              'node_modules/backbone/backbone.js',
              'node_modules/video.js/dist/video-js/**',
              'node_modules/videojs-youtube/dist/vjs.youtube.js'
            ],
            dest: 'examples/bare_page/vendor/'
          },
          {
            expand: true,
            src: [
              'vendor/**'
            ],
            dest: 'examples/bare_page/'
          },
          {
            expand: true,
            flatten: true,
            src: [
              '<%= paths.build %>/fileviewer.all.*'
            ],
            dest: 'examples/bare_page/build/'
          }
        ]
      },
      dist: {
        files: [
          {
            expand: true,
            flatten: true,
            src: [
              '<%= paths.build %>/fileviewer*',
              '<%= paths.build %>/translations.*.js'
            ],
            dest: '<%= paths.dist%>/'
          }
        ]
      },
    },

    closure: {
      pdfjs: {
        files: {
          '<%= paths.pdfjs %>/pdf.min.js': '<%= paths.pdfjs %>/pdf.js',
          '<%= paths.pdfjs %>/pdf.worker.min.js': '<%= paths.pdfjs %>/pdf.worker.js'
        }
      }
    },

    watch: {
      grunt: {
        files: [
          'package.json',
          'Gruntfile.js'
        ]
      },
      all: {
        files: [
          'vendor/*',
          ],
          tasks: [
            'example'
          ]
      },
      core: {
        files: [
          'lib/**/*',
          'lib/viewers/!image-view/*',
          'lib/viewers/!pdf-view/*',
          'lib/viewers/!video-view/*',
          'support/wrapper-*.js',
        ],
        tasks: [
          'example'
        ]
      },
      imageviewer: {
        files: 'lib/viewers/image-view/*',
        tasks: [
          'imageviewer',
          'concat:all',
          'usebanner:all',
          'copy:example',
        ]
      },
      documentviewer: {
        files: 'lib/viewers/pdf-view/*',
        tasks: [
          'documentviewer',
          'concat:all',
          'usebanner:all',
          'copy:example',
        ]
      },
      threedviewer: {
        files: 'lib/viewers/3d-view/*',
        tasks: [
          'threedviewer',
          'concat:all',
          'usebanner:all',
          'copy:example'
        ]
      },
      videoviewer: {
        files: 'lib/viewers/video-view/*',
        tasks: [
          'videoviewer',
          'concat:all',
          'usebanner:all',
          'copy:example',
        ]
      },
      minimode: {
        files: 'lib/plugins/minimode/*',
        tasks: [
          'minimode',
          'concat:all',
          'usebanner:all',
          'copy:example',
        ]
      },
      metabar: {
        files: 'lib/plugins/metabar/*',
        tasks: [
          'metabar',
          'concat:all',
          'usebanner:all',
          'copy:example',
        ]
      },
      watermark: {
        files: 'lib/plugins/watermark/*',
        tasks: [
          'watermark',
          'concat:all',
          'usebanner:all',
          'copy:example',
        ]
      }
    },

    replace: {
      version: {
        options: {
          patterns: [
            {
              match: '__FILEVIEWER_VERSION__',
              replacement: '<%= versionString %>'
            },
          ]
        },
        files: [
          {
            src: ['<%= paths.build %>/fileviewer.js'],
            dest: '<%= paths.build %>/fileviewer.js'
          }
        ]
      }
    }

  };

  loadJobs('./support/grunt/jobs.all.js');
  loadJobs('./support/grunt/jobs.core.js');
  loadJobs('./support/grunt/jobs.imageviewer.js');
  loadJobs('./support/grunt/jobs.documentviewer.js');
  loadJobs('./support/grunt/jobs.3dviewer.js');
  loadJobs('./support/grunt/jobs.videoviewer.js');
  loadJobs('./support/grunt/jobs.minimode.js');
  loadJobs('./support/grunt/jobs.metabar.js');
  loadJobs('./support/grunt/jobs.watermark.js');
  loadJobs('./support/grunt/jobs.apidocs.js');
  loadJobs('./support/grunt/jobs.hipchat.js');

  grunt.initConfig(config);

  grunt.registerTask('default', [
    'example'
  ]);

  grunt.registerTask('clean', [
    'exec:clean'
  ]);

  grunt.registerTask('cleandist', [
    'exec:cleandist'
  ]);

  grunt.registerTask('pdfjsmin', [
    'closure:pdfjs'
  ]);

  grunt.registerTask('apidocs', [
    'mkdir:apidocs',
    'exec:apidocs'
  ]);

  grunt.registerTask('all', [
    'prepare',
    'clean',
    'core',
    'imageviewer',
    'documentviewer',
    'threedviewer',
    'videoviewer',
    'minimode',
    'metabar',
    'watermark',
    'concat:all',
    'usebanner:all'
  ]);

  grunt.registerTask('example', [
    'all',
    'copy:example'
  ]);

  grunt.registerTask('dist', [
    'clean',
    'cleandist',
    'example',
    'apidocs',
    'copy:dist',
    'clean',
    'uglify:all',
    'cssmin:all',
    'usebanner:all',
    'dist-message'
  ]);

  grunt.registerTask('core', [
    'mkdir:core',
    'copy:core',
    'less:core',
    'concat:core',
    'translate:core',
    'i18nproperties:en',
    'replace:version'
  ]);

  grunt.registerTask('imageviewer', [
    'mkdir:imageviewer',
    'copy:imageviewer',
    'translate:imageviewer',
    'less:imageviewer',
    'concat:imageviewer'
  ]);

  grunt.registerTask('documentviewer', [
    'mkdir:documentviewer',
    'copy:documentviewer',
    'translate:documentviewer',
    'less:documentviewer',
    'concat:documentviewer'
  ]);

  grunt.registerTask('threedviewer', [
    'mkdir:threedviewer',
    'copy:threedviewer',
    'translate:threedviewer',
    'less:threedviewer',
    'concat:threedviewer'
  ]);

  grunt.registerTask('videoviewer', [
    'mkdir:videoviewer',
    'copy:videoviewer',
    'translate:videoviewer',
    'less:videoviewer',
    'concat:videoviewer'
  ]);

  grunt.registerTask('minimode', [
    'mkdir:minimode',
    'copy:minimode',
    'translate:minimode',
    'less:minimode',
    'concat:minimode'
  ]);

  grunt.registerTask('metabar', [
    'mkdir:metabar',
    'copy:metabar',
    'translate:metabar',
    'less:metabar',
    'concat:metabar'
  ]);

  grunt.registerTask('watermark', [
    'mkdir:watermark',
    'copy:watermark',
    'translate:watermark',
    'less:watermark',
    'concat:watermark'
  ]);

  grunt.registerTask('hipchat', [
    'all',
    'mkdir:hipchat',
    'concat:hipchat',
    'uglify:hipchat',
    'cssmin:hipchat',
    'copy:hipchat'
  ]);

  grunt.registerTask('dist-message', 'A sample task that logs stuff.', function (arg1, arg2) {
    grunt.log.writeln();
    grunt.log.writeln(('** FileViewer ' + config.versionString)['yellow'].bold);
    if (config.versionSet) {
      grunt.log.writeln(('** Version package.json updated')['yellow'].bold);
      grunt.log.writeln();
      grunt.log.writeln(('** Remember to update CHANGELOG.md before releasing :)')['yellow'].bold);
    }
  });
};

module.exports = configureGrunt;
