define('jira/richeditor/wiki-adapter/wiki-toolbar-operation', [
    'underscore',
    'jquery',
    'jira/util/logger',
    'jira/richeditor/wiki-adapter/editor-operation',
    'jira/util/events',
    'jira/editor/constants'
], function (
    _,
    $,
    logger,
    EditorOperation,
    JIRAEvents,
    EditorConstants
) {
    var WikiToolbarOperation = function ($el, $wikiTextarea, params) {
        this.$el = $el;
        this.$wikiTextarea = $wikiTextarea;
        this.params = params || {};

        this.supportedIcons = [
            ":)", ":(", ":P", ":D", ";)", "(y)", "(n)", "(i)", "(/)", "(x)", "(!)", "(+)", "(-)", "(?)", "(on)", "(off)", "(*)", "(*r)", "(*g)", "(*b)", "(*y)", "(flag)", "(flagoff)"
        ];

        this.supportedParams = ['data-color'];
    };

    WikiToolbarOperation.prototype.getElement = function () {
        return this.$el;
    };

    WikiToolbarOperation.prototype.getName = function () {
        if (this._isIcon()) {
            return "icon";
        }
        if (this._isAttachment()) {
            return "attachment"
        }
        return this.$el.attr('data-operation');
    };

    WikiToolbarOperation.prototype._isIcon = function () {
        var name = this.$el.attr('data-operation');
        return this.supportedIcons.indexOf(name) > -1;
    };

    WikiToolbarOperation.prototype._isColor = function () {
        return 'color' === this.$el.attr('data-operation');
    };

    WikiToolbarOperation.prototype._isAttachment = function () {
        return this.$el.filter(
                ".wiki-edit-dropdown-image, " +
                ".wiki-edit-dropdown-attachment"
            ).length > 0;
    };

    WikiToolbarOperation.prototype.getParams = function (additionalParams) {
        var params = {};
        additionalParams = additionalParams || {};

        this.supportedParams.forEach(function (attribute) {
            var attrValue = this.$el.attr(attribute);
            if (attrValue) {
                params[attribute] = attrValue;
            }
        }.bind(this));
        if (this._isIcon()) {
            params.icon = this.$el.attr('data-operation');
        }
        if (this._isColor()) {
            params.color = this.$el.is('.wiki-edit-operation-color') ? this.$el.find('.wiki-edit-color-indicator').css("background-color") : this.$el.attr('data-color');
        }

        if (this._isAttachment()) {
            if (additionalParams.attachment) {
                params.attachment = additionalParams.attachment;
            } else {
                var triggerElement = additionalParams.$attachmentElement;
                var filename = additionalParams.attachmentName;
                var attachmentType = additionalParams.attachmentType;
                if (triggerElement && filename && !triggerElement.is(".wiki-attachment-browse")) {
                    var href = this.getHrefFromDom(filename, triggerElement, triggerElement.is(":not([data-attachment-type])"));
                    var attachmentElement = this._getAttachmentRendered(filename, href, attachmentType);
                    if (attachmentElement) {
                        params.attachment = attachmentElement;
                    }
                }
            }
        }

        params = _.extend(params, this.params);

        return params;
    };

    WikiToolbarOperation.prototype._getAttachmentRendered = function (filename, href, attachmentType) {
        if (attachmentType === 'image') {
            if (!href) {
                return undefined;
            }
            var imgThumb = href.replace('/attachment/', '/thumbnail/');
            return JIRA.Editor.Tags.Templates.attachedImage({
                source: imgThumb,
                filename: filename
            });
        } else {
            return JIRA.Editor.Tags.Templates.attachedFile({
                title: filename,
                href: href,
                filename: filename
            });
        }
    },

    WikiToolbarOperation.prototype.getShortcut = function () {
        var shortcut = this.$el.attr('data-shortcut');
        if (shortcut) {
            // ctrl is treated as meta by tinymce
            return shortcut.replace('command+', 'meta+').replace('ctrl+', 'meta+');
        }
    };

    WikiToolbarOperation.prototype.getEditorOperation = function (additionalParams) {
        return new EditorOperation(this.getName(), this.getParams(additionalParams), this.getShortcut());
    };

    WikiToolbarOperation.prototype.execute = function (editor, params) {
        if (!editor.isVisible()) {
            return;
        }

        params = params || {};
        var element = params.element || this.getElement();
        if (this._isAttachment()) {
            if (element.is('[data-attachment-type]')) {
                var editorOperation = this.getEditorOperation({
                    attachmentName: element.find('.wiki-edit-attachment-name').text(),
                    attachmentType: element.attr('data-attachment-type'),
                    $attachmentElement: element
                });
                if (editor.executeOperation(editorOperation.getName(), editorOperation.getParams())) {
                    logger.trace("jira.editor.operation.executed");
                }
            }
        } else {
            var editorOperation = this.getEditorOperation();
            if (editor.executeOperation(editorOperation.getName(), editorOperation.getParams())) {
                logger.trace("jira.editor.operation.executed");
            }
        }
    };

    WikiToolbarOperation.prototype.getHrefFromDom = function (filename, $button, useScope) {
        var dialog = $button.closest('.jira-dialog');
        var $scope = $(document);
        if (useScope) {
            $scope = (dialog && dialog.length > 0) ? $(dialog[0]) : $(document);
        }
        var ar = $scope.find('a.attachment-title, .attachment-title a').filter(function (idx, el) {
            return $(el).text() === filename;
        });

        var href;
        if (ar.length === 0) {
            ar = $scope.find('span.upload-progress-bar__file-name').filter(function (idx, el) {
                return $(el).text() === filename;
            });
            if (ar.length === 0) {
                return;
            }

            href = $(ar[0].parentElement).find('img').attr('src');
            if (!href) {
                href = $(ar[0]).closest('.attachments-upload-progress-bar').find('img').attr('src');
            }
            if (!href) {
                href = $(ar[0]).closest('.temp-upload-progress-bar__parent-container').find('img').attr('src');
            }
        } else {
            href = $(ar[0]).attr('href');
        }

        return href;
    };

    WikiToolbarOperation.prototype._attachmentReceived = function (editor, e, params) {
        if (!editor.isVisible()) {
            return;
        }
        var determineFileType = function (fileName) {
            return fileName.match(EditorConstants.SupportedImagesRegexp) ? 'image' : 'file';
        };

        if (params && params.files && params.files.length === 1 && params.wikiTextfield) {
            if (this.$wikiTextarea.get(0) === params.wikiTextfield) {
                var file = params.files[0];
                var href;
                var attachmentType = determineFileType(file.name);
                if (attachmentType === 'image') {
                    href = window.URL.createObjectURL(file);
                }

                var attachmentType = this._getAttachmentRendered(file.name, href, attachmentType);
                var editorOperation = this.getEditorOperation({attachment: attachmentType});
                if (editor.executeOperation(editorOperation.getName(), editorOperation.getParams())) {
                    logger.trace("jira.editor.operation.executed");
                }
            }
        }
    };

    WikiToolbarOperation.prototype.bindEvents = function (editor) {
        if (this._isAttachment() && !this._eventHandler) {
            this._eventHandler = this._attachmentReceived.bind(this, editor);
            JIRAEvents.bind("wikiAttachmentForPageReceived.operationAttachment", this._eventHandler);
        }
    };

    WikiToolbarOperation.prototype.unbindEvents = function () {
        if (this._isAttachment() && this._eventHandler) {
            JIRAEvents.unbind("wikiAttachmentForPageReceived.operationAttachment", this._eventHandler);
        }
    };

    return WikiToolbarOperation;
});
