define('jira/richeditor/wiki-adapter/wiki-toolbar-button', [
    'jira/richeditor/wiki-adapter/selection',
    'jquery'
], function (selection,
             $
) {
    var EVENT_NAMESPACE = ".richEditorEvents";

    var WikiToolbarButtonBasic = function (wikiOperation, $wikiTextarea, contextManager) {
        this.bindButton = function (editor) {
            if (!wikiOperation.getElement()) {
                throw 'No toolbar element found';
            }
            this.$uiEl = wikiOperation.getElement();
            this.editor = editor;

            this.bindSaveSelectionAndCopyText(wikiOperation);

            var execute = function (e) {
                e.preventDefault();
                if (this.isDisabled()) {
                    return;
                }
                wikiOperation.execute(editor, {element: $(e.currentTarget)});
                this.restoreSelection();
            }.bind(this);

            wikiOperation.bindEvents(editor);
            if (wikiOperation._isAttachment()) {
                this.$uiEl.on('click' + EVENT_NAMESPACE, 'a', execute);
            } else {
                this.$uiEl.on('click' + EVENT_NAMESPACE, execute);
            }

            var editorOperation = wikiOperation.getEditorOperation();
            editor.bindOperationSelectedListener(editorOperation.getName(), this.setSelected.bind(this));
            this.addShortcut(editor, wikiOperation.getShortcut());

            contextManager.on("change:all change:" + mapOperationNameToTag(wikiOperation.getName()), function (e) {
                this.setDisabled(e.disableState);
            }, this);

            editor.on('tabs:changed', function (isText) {
                if (isText) {
                    this.setSelected(false);
                }
            }.bind(this));
        };

        this.setSelected = function (selected) {
            if (this.isDisabled()) {
                return;
            }

            if (!!selected !== this.isSelected()) {
                this.$uiEl.attr('aria-pressed', selected ? 'true' : null);
            }
        };

        this.isSelected = function() {
            return !!this.$uiEl.attr('aria-pressed');
        };

        this.destroy = function () {
            if (this.$uiEl) {
                this.setDisabled(false);
                this.$uiEl.off('click' + EVENT_NAMESPACE);
                this.$uiEl.unbind('click' + EVENT_NAMESPACE);
            }
            wikiOperation.unbindEvents();
        };

        this.addShortcut = function (editor, shortcut) {
            if (shortcut) {
                var that = this;
                editor.addShortcut(shortcut, function beforeEditorOperation() {
                    that.$uiEl.click();
                });
            }
        };

        this.bindSaveSelectionAndCopyText = function () {
            var type = 'click';
            this.$uiEl.bind(type + EVENT_NAMESPACE, this.saveSelectionAndCopyText.bind(this));
            var currentBindings = this.$uiEl.data('events')[type] || [];
            var handler = currentBindings.pop();
            currentBindings.splice(0, 0, handler);
        };

        this.saveSelectionAndCopyText = function () {
            if (!this.editor.isVisible() || this.isDisabled()) {
                return;
            }

            this.saveSelection();
            var selectedText = this.editor.getSelectedContent(true);
            $wikiTextarea.val(selectedText);
            $wikiTextarea.select();
        };

        this.saveSelection = function saveSelection() {
            this.savedRange = selection.save();
        };

        this.restoreSelection = function restoreSelection() {
            if (this.savedRange) {
                selection.restore(this.savedRange);
                this.savedRange = undefined;
            }
        };

        this.isDisabled = function isDisabled() {
            return this.$uiEl.attr("aria-disabled") === "true";
        };

        this.setDisabled = function (disabled) {
            this.$uiEl.attr("aria-disabled", disabled.toString());
            if (disabled) {
                this.$uiEl.addClass('disabled').css('pointer-events', 'none');
            } else {
                this.$uiEl.removeClass('disabled').css('pointer-events', 'auto');
            }
        };
    };

    function mapOperationNameToTag(opName) {
        var map = {
            'link': 'a',
            'link-mail': 'a',
            'link-anchor': 'a'
        };
        return map[opName] ? map[opName] : opName;
    }

    return WikiToolbarButtonBasic;
});