AJS.test.require(["com.atlassian.jira.plugins.jira-editor-plugin:resources"], function() {
    var $ = require('jquery');
    var Backbone = require("backbone");
    var _ = require("underscore");
    var WikiToolbarButton;

    var WIKI_OPERATION_NAME = 'some-wiki-operation';

    module("wiki-toolbar-button", {
        setup: function() {
            this.mockedContext = AJS.test.mockableModuleContext();
            this.sandbox = sinon.sandbox.create({ useFakeServer: true });
            this.EVENT_NAMESPACE = ".richEditorEvents";

            this.element = $('<div>');
            this.sandbox.spy(this.element, 'on');
            this.sandbox.spy(this.element, 'off');
            this.sandbox.spy(this.element, "attr");

            this.editorOperation = {};
            this.editorOperation.getName = this.sandbox.stub().returns("operation1");
            this.editorOperation.getParams = this.sandbox.stub().returns({});

            this.wikiOperation = {};
            this.wikiOperation.getElement = this.sandbox.stub().returns(this.element);
            this.wikiOperation.getName = this.sandbox.stub().returns(WIKI_OPERATION_NAME);
            this.wikiOperation.execute = this.sandbox.spy();
            this.wikiOperation.getEditorOperation = this.sandbox.stub().returns(this.editorOperation);
            this.wikiOperation.getShortcut = this.sandbox.stub().returns(undefined);
            this.wikiOperation._isAttachment = this.sandbox.stub().returns(false);
            this.wikiOperation.bindEvents = this.sandbox.spy();
            this.wikiOperation.unbindEvents = this.sandbox.spy();

            this.wikiOperationWithShortcut = {};
            this.wikiOperationWithShortcut.getElement = this.sandbox.stub().returns(this.element);
            this.wikiOperationWithShortcut.getName = this.sandbox.stub().returns(WIKI_OPERATION_NAME);
            this.wikiOperationWithShortcut.execute = this.sandbox.spy();
            this.wikiOperationWithShortcut.getEditorOperation = this.sandbox.stub().returns(this.editorOperation);
            this.wikiOperationWithShortcut.getShortcut = this.sandbox.stub().returns("shortcut1");
            this.wikiOperationWithShortcut._isAttachment = this.sandbox.stub().returns(false);
            this.wikiOperationWithShortcut.bindEvents = this.sandbox.spy();
            this.wikiOperationWithShortcut.unbindEvents = this.sandbox.spy();

            this.wikiOperationAttachment = {};
            this.wikiOperationAttachment.getElement = this.sandbox.stub().returns(this.element);
            this.wikiOperationAttachment.getName = this.sandbox.stub().returns(WIKI_OPERATION_NAME);
            this.wikiOperationAttachment.execute = this.sandbox.spy();
            this.wikiOperationAttachment.getEditorOperation = this.sandbox.stub().returns(this.editorOperation);
            this.wikiOperationAttachment.getShortcut = this.sandbox.stub().returns(undefined);
            this.wikiOperationAttachment._isAttachment = this.sandbox.stub().returns(true);
            this.wikiOperationAttachment.bindEvents = this.sandbox.spy();
            this.wikiOperationAttachment.unbindEvents = this.sandbox.spy();

            this.editor = {};
            this.editor.bindOperationSelectedListener = this.sandbox.spy();
            this.editor.addShortcut = this.sandbox.spy();
            this.editor.on = this.sandbox.spy();

            this.$wikiTextarea = $("<textarea>");
            this.contextManager = _.extend({}, Backbone.Events);

            WikiToolbarButton = this.mockedContext.require('jira/richeditor/wiki-adapter/wiki-toolbar-button');
        },
        teardown: function () {
            this.sandbox.restore();
        }
    });

    test("Cannot bind button without dom element", function (assert) {
        var wikiOperation = {};
        assert["throws"](
            function () {
                new WikiToolbarButton(wikiOperation, this.$wikiTextarea, this.contextManager).bindButton({});
            }.bind(this),
            'Constructor should throw when there is no dom element provided'
        );
    });

    test("Click event handler is bound on dom element", function () {
        new WikiToolbarButton(this.wikiOperation, this.$wikiTextarea, this.contextManager).bindButton(this.editor);

        sinon.assert.calledWith(this.element.on, 'click'+this.EVENT_NAMESPACE);
        sinon.assert.calledWith(this.editor.on, 'tabs:changed');

        sinon.assert.notCalled(this.element.off);
    });

    test("Events are bound", function () {
        new WikiToolbarButton(this.wikiOperation, this.$wikiTextarea, this.contextManager).bindButton(this.editor);

        sinon.assert.calledOnce(this.wikiOperation.bindEvents);
        sinon.assert.calledWith(this.wikiOperation.bindEvents, this.editor);
    });

    test("Click event handler is bound on child 'a' dom element", function () {
        new WikiToolbarButton(this.wikiOperationAttachment, this.$wikiTextarea, this.contextManager).bindButton(this.editor);

        sinon.assert.calledWith(this.element.on, 'click'+this.EVENT_NAMESPACE, 'a');
        sinon.assert.calledWith(this.editor.on, 'tabs:changed');

        sinon.assert.notCalled(this.element.off);
    });

    test("Operation selected listener is bound", function () {
        new WikiToolbarButton(this.wikiOperation, this.$wikiTextarea, this.contextManager).bindButton(this.editor);

        sinon.assert.calledOnce(this.editor.bindOperationSelectedListener);
        sinon.assert.calledWith(this.editor.bindOperationSelectedListener, "operation1");
        sinon.assert.calledWith(this.editor.on, 'tabs:changed');
    });

    test("Shortcut is added if present", function () {
        new WikiToolbarButton(this.wikiOperationWithShortcut, this.$wikiTextarea, this.contextManager).bindButton(this.editor);

        sinon.assert.calledOnce(this.editor.addShortcut);
        sinon.assert.calledWith(this.editor.on, 'tabs:changed');
    });

    test("Shortcut is not added if not present", function () {
        new WikiToolbarButton(this.wikiOperation, this.$wikiTextarea, this.contextManager).bindButton(this.editor);

        sinon.assert.notCalled(this.editor.addShortcut);
        sinon.assert.calledWith(this.editor.on, 'tabs:changed');
    });

    test("Destroy function should clean up resources", function () {
        var button = new WikiToolbarButton(this.wikiOperationWithShortcut, this.$wikiTextarea, this.contextManager);
        button.bindButton(this.editor);
        button.destroy();

        sinon.assert.calledWith(this.element.off, 'click'+this.EVENT_NAMESPACE);
        sinon.assert.calledOnce(this.wikiOperationWithShortcut.unbindEvents);
        sinon.assert.calledWith(this.editor.on, 'tabs:changed');
    });

    test("Should enable button when context manager triggers a 'change:all' event", function () {
        new WikiToolbarButton(this.wikiOperation, this.$wikiTextarea, this.contextManager).bindButton(this.editor);

        this.contextManager.trigger("change:all", {
            disableState: false
        });

        sinon.assert.calledWith(this.element.attr, "aria-disabled", "false");
        sinon.assert.calledWith(this.editor.on, 'tabs:changed');
    });

    test("Should disable button when context manager triggers a 'change:all' event", function () {
        new WikiToolbarButton(this.wikiOperation, this.$wikiTextarea, this.contextManager).bindButton(this.editor);

        this.contextManager.trigger("change:all", {
            disableState: true
        });

        sinon.assert.calledWith(this.element.attr, "aria-disabled", "true");
        sinon.assert.calledWith(this.editor.on, 'tabs:changed');
    });

    test("Should enable button when context manager triggers a specific event", function () {
        new WikiToolbarButton(this.wikiOperation, this.$wikiTextarea, this.contextManager).bindButton(this.editor);

        this.contextManager.trigger("change:" + WIKI_OPERATION_NAME, {
            disableState: false
        });

        sinon.assert.calledWith(this.element.attr, "aria-disabled", "false");
        sinon.assert.calledWith(this.editor.on, 'tabs:changed');
    });

    test("Should disable button when context manager triggers a specific event", function () {
        new WikiToolbarButton(this.wikiOperation, this.$wikiTextarea, this.contextManager).bindButton(this.editor);

        this.contextManager.trigger("change:" + WIKI_OPERATION_NAME, {
            disableState: true
        });

        sinon.assert.calledWith(this.element.attr, "aria-disabled", "true");
        sinon.assert.calledWith(this.editor.on, 'tabs:changed');
    });

    test("Should not react when context manager triggers an unrelated event", function () {
        new WikiToolbarButton(this.wikiOperation, this.$wikiTextarea, this.contextManager).bindButton(this.editor);

        this.contextManager.trigger("change:fake-operation", {
            disableState: true
        });

        sinon.assert.notCalled(this.element.attr);
        sinon.assert.calledWith(this.editor.on, 'tabs:changed');
    });
});