define("jira/richeditor/wiki-adapter/wiki-toolbar-adapter", [
    "underscore",
    "jira/richeditor/wiki-adapter/toolbar-parser",
    "jira/richeditor/wiki-adapter/wiki-toolbar-button",
    "jira/richeditor/editor-supported-operations"
], function (
    _,
    toolbarParser,
    WikiToolbarButtonBasic,
    supportedOperations
) {
    "use strict";

    /**
     *
     * @param editor - Jira editor instance
     * @param $toolbarParent Element containing existing toolbar
     * @constructor
     */
    var WikiToolbarAdapter = function (editor, $toolbarParent, $wikiTextarea) {
        this.supportedOperations = supportedOperations.getWikiOperations();
        var allWikiToolbarOperations = toolbarParser($toolbarParent, $wikiTextarea);
        this._overrideWikiFallbackOperations(allWikiToolbarOperations, editor, $wikiTextarea);
        this.buttons = this.buildButtons(editor, $wikiTextarea, allWikiToolbarOperations);
        this.bindButtonsToToolbar(this.buttons, $toolbarParent, $wikiTextarea, editor);
        this.dropdowns = this._getDropdowns($toolbarParent);

        editor.contextManager.on("change:all", this._disableStateChanged.bind(this));
    };

    WikiToolbarAdapter.prototype.destroy = function () {
        this.buttons.forEach(function (button) {
            button.destroy();
        });
    };

    /**
     * builds behaviour of buttons - what happens on onclick, how to get state etc.
     */
    WikiToolbarAdapter.prototype.buildButtons = function (editor, $wikiTextarea, allWikiOperations) {
        var contextManager = editor.contextManager;
        var buttons = allWikiOperations.map(function (wikiOperation) {
            if (this.supportedOperations.indexOf(wikiOperation.getName()) > -1) {
                return new WikiToolbarButtonBasic(wikiOperation, $wikiTextarea, contextManager);
            }
        }.bind(this)).filter(_.identity);

        //remove supported buttons
        var fallbackWikiOperations = this._filterFallbackWikiOperations(allWikiOperations);

        fallbackWikiOperations.forEach(function (wikiOperation) {
            buttons.push(new WikiToolbarButtonBasic(wikiOperation, $wikiTextarea, contextManager));
        });

        return buttons;
    };

    WikiToolbarAdapter.prototype._overrideWikiFallbackOperations = function (allWikiOperatons, editor, $wikiTextarea) {
        var fallbackWikiOperations = this._filterFallbackWikiOperations(allWikiOperatons);

        fallbackWikiOperations.forEach(function (wikiOperation) {
            editor.addOperationOverride(wikiOperation.getName(), function (args) {
                var wikiText = $wikiTextarea.val();
                wikiText = wikiText.replace(/(?:\r\n|\r|\n)/g, '<br />');
                editor.executeOperation('editorReplaceContent', {content: wikiText});
            });
        });
    };

    WikiToolbarAdapter.prototype._filterFallbackWikiOperations = function (allWikiOperations) {
        return allWikiOperations.filter(function (wikiOperation) {
            return this.supportedOperations.indexOf(wikiOperation.getName()) < 0
        }.bind(this));
    };

    /**
     * Binds passed buttons to toolbar
     * @param buttons Array of buttons to bind to toolbar.
     */
    WikiToolbarAdapter.prototype.bindButtonsToToolbar = function (buttons, $toolbar, $wikiTextarea, editor) {
        buttons = buttons || [];

        // bind supported new operations
        buttons.forEach(function bindButton(button) {
            button.bindButton(editor);
        });
    };

    WikiToolbarAdapter.prototype._disableStateChanged = function (params) {
        this.dropdowns.attr("aria-disabled", params.disableState.toString());
    };

    WikiToolbarAdapter.prototype._getDropdowns = function ($toolbar) {
        return $toolbar.find(".aui-dropdown2-trigger");
    };

    return WikiToolbarAdapter;
});