define("jira/editor/undomanager-adapter/visual-adapter", [
    "jira/editor/constants",
    "jira/editor/marionette",
    "jquery"
], function (
    EditorConstants,
    Marionette,
    $
) {
    "use strict";

    var VisualAdapter = Marionette.Object.extend({
        initialize: function initialize(options) {
            this.id = this.determineId(options.richEditorEl);
            this.visualEditorInstance = options.richEditorEl.getInstance();
            this.visualEditor = this.visualEditorInstance.editor;
            this.visualUndoManager = this.visualEditor.undoManager;

            // It's needed for ignoring subsequent event coming from observed UndoManager
            //  when the action was triggered explicitly in "undo" or "redo" method.
            this.ongoingInternalOperation = false;

            this.attachEvents(this.visualEditor);
        },
        undo: function undo() {
            try {
                this.ongoingInternalOperation = true;
                this.visualUndoManager.undo();
            } finally {
                this.ongoingInternalOperation = false;
            }
        },
        redo: function redo() {
            try {
                this.ongoingInternalOperation = true;
                this.visualUndoManager.redo();
            } finally {
                this.ongoingInternalOperation = false;
            }
        },
        hasUndo: function hasUndo() {
            return this.visualUndoManager.hasUndo();
        },
        hasRedo: function hasRedo() {
            return this.visualUndoManager.hasRedo();
        },
        clear: function clear() {
            this.visualUndoManager.clear();
        },

        add: function add(level) {
            this.visualEditorInstance.setContent(level.content, null, true);
            this.visualUndoManager.add(level);
        },
        getData: function getData() {
            return this.visualUndoManager.data;
        },
        getId: function getId() {
            return this.id;
        },

        attachEvents: function attachEvents(editor) {
            editor.on('AddUndo', function(e) {
                if (!this.ongoingInternalOperation) {
                    this.triggerMethod("action", EditorConstants.EventTypes.ADD, EditorConstants.EventSources.UNKNOWN);
                }
            }.bind(this));
            editor.on('Undo', function(e) {
                if (!this.ongoingInternalOperation) {
                    this.triggerMethod("action", EditorConstants.EventTypes.UNDO, EditorConstants.EventSources.SHORTCUT);
                }
            }.bind(this));
            editor.on('Redo', function(e) {
                if (!this.ongoingInternalOperation) {
                    this.triggerMethod("action", EditorConstants.EventTypes.REDO, EditorConstants.EventSources.SHORTCUT);
                }
            }.bind(this));
            editor.on('ClearUndos', function(e) {
                if (!this.ongoingInternalOperation) {
                    this.triggerMethod("action", EditorConstants.EventTypes.CLEAR, EditorConstants.EventSources.UNKNOWN);
                }
            }.bind(this));
        },
        determineId: function determineId(richEditorEl) {
            var $textarea = richEditorEl.$textarea;
            var issueKey = $(richEditorEl).data('renderParams').issueKey || '';
            var fieldName = $textarea.attr('name');
            var fieldId = $textarea.attr('id');

            return [issueKey, fieldName, fieldId].join('.');
        }
    });

    return VisualAdapter;
});