define("jira/editor/ui/toggle-tabs", [
    "jira/editor/constants",
    "jira/editor/analytics",
    "jira/editor/toggle",
    "jira/lib/class",
    "backbone",
    "aui/tabs",
    "underscore",
    "jquery"
], function(
    Constants,
    Analytics,
    editorToggle,
    Class,
    Backbone,
    tabs,
    _,
    $
) {
    var Modes = Constants.Modes;

    var ToggleTabs = Class.extend({

        /**
         * @param element
         */
        init: function(element) {
            this.$parent = $(element).parent('.wiki-edit-content');

            this.rendered = new $.Deferred();

            editorToggle.getMode().always(function(mode) {
                this.renderTabs(mode);
            }.bind(this));

            this.hideTooltip = _.debounce(this.hideTooltip, 750);
        },


        renderTabs: function(mode) {
            this.$tabs = $(aui.tabs({
                menuItems: [
                    { text: AJS.I18n.getText('jira.editor.toggle.visual'), url: '#', mode: Modes.VISUAL },
                    { text: AJS.I18n.getText('jira.editor.toggle.text'), url: '#', mode: Modes.TEXT }
                ].map(function(item) {
                    return _.extend(item, { url: '#', extraAttributes: 'data-mode="' + item.mode + '"', isActive: mode === item.mode })
                }),
                paneContent: '',
                extraClasses: 'editor-toggle-tabs'
            }));

            this.$tabs.appendTo(this.$parent)
                .find('.menu-item').on('tabSelect', this.onTabSelect.bind(this));

            this.$tooltip = $(aui.labels.label({ text: '', extraClasses: 'editor-toggle-tooltip' }))
                .appendTo(this.$parent);

            tabs.setup();

            this.rendered.resolve(this.$tabs);
        },

        getTab: function () {
            return this.$tabs.find(".menu-item").filter(function (idx, item) {
                return $(item).data("mode") === Modes.VISUAL;
            });
        },

        onTabSelect: function(event, context) {
            var mode = context.tab.parent(".menu-item").data("mode");
            if (mode) {
                this.trigger("change", mode);
                editorToggle.setMode(mode);


                this.showTooltip(mode);
                this.hideTooltip();

                Analytics.sendEvent("editor.instance.switch.change");
                Analytics.sendEvent("bundled.editor.instance.switch.change");
            }
        },

        getMode: function() {
            var deferred = new $.Deferred();

            this.rendered.then(function($tabs) {
                var mode = $tabs.find(".menu-item.active-tab").data("mode");

                if (mode) {
                    deferred.resolve(mode);
                } else {
                    deferred.reject();
                }
            }.bind(this));

            return deferred.promise();
        },

        setMode: function(mode) {
            this.rendered.then(function($tabs) {
                $tabs.find(".menu-item").removeClass("active-tab").filter(function(idx, el) {
                    return $(el).data("mode") === mode;
                }).addClass("active-tab");

                this.showTooltip(mode);
                this.hideTooltip();
            }.bind(this));
        },

        enable: function() {
            this.$tabs.removeClass('aui-tabs-disabled').find('a').attr("tabindex", -1);
        },

        disable: function() {
            this.$tabs.addClass('aui-tabs-disabled').find('a').removeAttr("tabindex");
        },

        showTooltip: function(mode) {
            var modeText = mode === Modes.VISUAL
                ? AJS.I18n.getText('jira.editor.toggle.mode.visual')
                : AJS.I18n.getText('jira.editor.toggle.mode.text');
            this.$tooltip
                .text(modeText)
                .addClass('visible');
            // we need aui-blanket to prevent inline edit close during mode toggle
            if (this.$tabs.parents().is('.inline-edit-fields')) {
                this.$tooltip.append('<span class="aui-blanket"></span>');
            }
        },

        hideTooltip: function() {
            this.$tooltip.removeClass('visible');
            this.$tooltip.find('.aui-blanket').remove();
        },

        isVisible: function() {
            if (!this.$tabs) {
                return;
            }

            // element not present in DOM - rect will be null
            var rect = this.$tabs[0].getBoundingClientRect();
            // element not visible - elementFromPoint either null or is covered
            return rect && $(document.elementFromPoint(rect.left + rect.width / 2, rect.top + rect.height / 2)).parents().is(this.$tabs);
        }

    }).extend(Backbone.Events);

    return ToggleTabs;
});