define("jira/editor/plugins/wiki-autoformat", [
    "jira/editor/tinymce",
    "jira/editor/analytics",
    'jquery',
    'jira/util/version',
    'jira/editor/html-renderer',
    'jira/util/logger'
], function (
    tinymce,
    Analytics,
    $,
    Version,
    htmlRenderer,
    logger
) {
    var wikiAutoformat = {
        /**
         * An internal map of regexes. Expose for testing.
         */

        _REGEXES_EMOTICON: {
            SMILE: /\B(:-?)$/,
            SAD: /\B(:-?)$/,
            CHEEKY: /\B(:-?)$/,
            CHEEKY_2: /\B(:-?)$/,
            LAUGH: /\B(:-?)$/,
            WINK: /\B(;-?)$/,
            THUMBS_UP: /\B(\(y)$/,
            THUMBS_DOWN: /\B(\(n)$/,
            INFORMATION: /\B(\(i)$/,
            TICK: /\B(\(\/)$/,
            CROSS: /\B(\(x)$/,
            WARNING: /\B(\(!)$/,
            PLUS: /\B(\(\+)$/,
            MINUS: /\B(\(-)$/,
            QUESTION: /\B(\(\?)$/,
            LIGHT_ON: /\B(\(on)$/,
            LIGHT_OFF: /\B(\(off)$/,
            YELLOW_STAR: /\B(\(\*)$/,
            YELLOW_STAR_2: /\B(\(\*y)$/,
            RED_STAR: /\B(\(\*r)$/,
            GREEN_STAR: /\B(\(\*g)$/,
            BLUE_STAR: /\B(\(\*b)$/,
            FLAG: /\B(\(flag)$/,
            FLAG_OFF: /\B(\(flagoff)$/,
            HEART: /\B(<)$/,
            BROKEN_HEAR: /\B(<\/)$/
        }
    };

    var rangeAfterAutoFormat;

    function encodeDomNode(domNode){
        return $('<div>').append( domNode.cloneNode(true) ).html();
    }
    function getStartContainer(lengthOfMatch, currentTextNode, currentOffset) {
        var siblingParent;
        var siblingIndex;

        if (!currentTextNode) {
            throw new Error("text node is null");
        }
        if (currentTextNode.nodeType !== Node.TEXT_NODE) {
            currentTextNode = currentTextNode.childNodes[currentOffset - 1];
            currentOffset = currentTextNode.length;
        }

        for (var ps = currentTextNode, runningOffset = currentOffset; ps && ps.nodeType === Node.TEXT_NODE; ps = ps.previousSibling) {
            if (runningOffset === -1) {
                runningOffset = ps.nodeValue.length;
            }

            if (runningOffset > lengthOfMatch) {
                return {
                    container: ps,
                    offset: runningOffset - lengthOfMatch
                };
            } else if (runningOffset === lengthOfMatch) {
                siblingIndex = 0;
                siblingParent = ps.parentNode;
                while (ps = ps.previousSibling) {
                    siblingIndex++;
                }
                return {
                    container: siblingParent,
                    offset: siblingIndex
                }
            } else {
                lengthOfMatch -= runningOffset;
                runningOffset = -1;
            }
        }

        return null;
    }

    /**
     * Assuming | is the caret marker the caret could be positioned as such:
     * <p><textNode>content of text node|</textNode></p>
     *                                  ^
     * or (less obviously) as such:
     * <p><textNode>content of text node</textNode>|</p>
     *                                             ^
     */
    function getTextFromPreviousSiblingTextNodes(range) {
        var currentNode
        var currentOffset;

        if (!range || !range.collapsed) {
            throw new Error("range is null or not collapsed");
        }
        currentNode = range.startContainer;
        currentOffset = range.startOffset;

        if (currentNode.nodeType === Node.ELEMENT_NODE && currentOffset > 0) {
            //descend into the relevent child node
            currentNode = currentNode.childNodes[range.startOffset - 1];
            if (currentNode.nodeType === Node.TEXT_NODE) {
                //It's a text node so we'll use it.
                currentOffset = currentNode.nodeValue.length;
            } else {
                //We're working under the assumption that the start container will never be
                // higher than textnode.parentNode.
                return "";
            }
        } else if (currentNode.nodeType !== Node.TEXT_NODE) {
            //not a text node or element, we can assume "".
            return "";
        }

        var result = currentNode.nodeValue.substring(0, currentOffset);

        for (var ps = currentNode.previousSibling; ps && ps.nodeType === Node.TEXT_NODE; ps = ps.previousSibling) {
            result = ps.nodeValue + result;
        }

        return result;
    }

    /**
     * @param {RegExp} regex
     * @param {Function} fragmentCreator
     * @param {boolean=} swallowTriggerChar
     * @param {string|Array.string=} forbiddenContexts
     * @returns {{handles: handles, execute: execute}}
     */
    function createHandler(regex, fragmentCreator, swallowTriggerChar, forbiddenContexts) {
        return {
            handles: function (ed) {
                if (typeof forbiddenContexts === 'string') {
                    forbiddenContexts = [forbiddenContexts];
                } else if (!forbiddenContexts || !forbiddenContexts.length) {
                    forbiddenContexts = [];
                }

                var result = false;
                var range = ed.selection.getRng(true);
                var candidateTextNode = range.commonAncestorContainer || {};

                if (!range.collapsed) {
                    return false;
                }

                var contextManager = ed.contextManager;
                // CONFDEV-4376: Autoformat should be disabled in preformatted text.
                if (contextManager.isPreContextActive()) {
                    Analytics.sendEvent('editor.instance.autoformat.prevented', {context: 'pre'});
                    return false;
                }

                for (var i = 0, l = forbiddenContexts.length; i < l; ++i) {
                    if (contextManager.isContextActive(forbiddenContexts[i])) {
                        Analytics.sendEvent('editor.instance.autoformat.prevented', {context: forbiddenContexts[i]});
                        return false;
                    }
                }

                result = regex.test(getTextFromPreviousSiblingTextNodes(range));

                return result;
            },
            execute: function (ed, nativeEvent) {
                var range;
                var combinedText;
                var matchGroups;
                var matchGroupsOffset = 1;
                var startContainerData;
                var commonAncestor;
                var charCode = getCharCode(nativeEvent);

                if (charCode === 32) {
                    ed.execCommand('mceInsertContent', false, '&nbsp;'); //spaces don't stick around for undos (it's tragic, I know)
                } else {
                    ed.execCommand('mceInsertContent', false, String.fromCharCode(charCode));
                }
                range = ed.selection.getRng(true);
                combinedText = getTextFromPreviousSiblingTextNodes(range);

                //CONFDEV-4771 Just for tables, add a space at the end and change the offset for matchGroups
                //This fixes and IE8 issue where the table autoformat didn't always work
                if(combinedText[combinedText.length-1] === '|') {
                    combinedText += ' ';
                    matchGroupsOffset = 0;
                }

                matchGroups = regex.exec(combinedText.substring(0, combinedText.length-1)); // regexes need to work for handles() where trigger character has not been appended
                startContainerData = getStartContainer(matchGroups[1].length+matchGroupsOffset, range.commonAncestorContainer, range.startOffset);
                range.setStart(startContainerData.container, startContainerData.offset);
                commonAncestor = $(range.commonAncestorContainer);
                ed.selection.setRng(range); //we have to set the editors selection now that we have modified the range to have text selected

                fragmentCreator(matchGroups, ed.selection.getRng(true));
                rangeAfterAutoFormat = ed.selection.getRng(true);

                if (swallowTriggerChar) {
                    nativeEvent.preventDefault();
                    nativeEvent.stopPropagation();
                    tinymce.dom.Event.cancel(nativeEvent);
                    // CONFDEV-2503 - cancelling the event stops the browser from scrolling to the new content
                    // so manually scroll there.
                    //AJS.Rte.showElement(rangeAfterAutoFormat.startContainer);
                    return false;
                }
            }
        };
    }

    function getCharCode (nativeEvent) {
        return $.browser.msie ? nativeEvent.keyCode : nativeEvent.which;
    }

    function HandlerManager () {
        this.handlers = {};
    }
    HandlerManager.prototype = {
        registerHandler: function (triggerCharCode, handler) {
            if (!this.handlers[triggerCharCode]) {
                this.handlers[triggerCharCode] = [];
            }

            this.handlers[triggerCharCode].push(handler);
        },
        executeHandlers: function (triggerCharCode, ed, nativeEvent) {
            var result = true;
            $.each(this.handlers[triggerCharCode] || [], function (i, handler) {
                if (handler.handles(ed)) {
                    result = handler.execute(ed, nativeEvent);
                    logger.trace("jira.editor.wiki.autoformatted");
                    return false; // signal end of iteration
                }
            });

            return result;
        }
    };

    function initAutoFormat(ed) {
        var handlerManager = new HandlerManager();

        /* Emoticons */

        /** Create an Emoticon object and register it with the handlerManager */
        function Emoticon(triggerChar, handlerRegex, emoticon, title, image, secret) {
            var triggerCharCode = triggerChar.charCodeAt(0);
            var imagePath = AJS.contextPath() + "/images/icons/emoticons/" + image;
            var handler;

            handler = createHandler(handlerRegex, function () {
                var img = ed.dom.createHTML("img", {
                    "src": imagePath,
                    "alt": ed.getLang(title),
                    "title": ed.getLang(title),
                    "border": 0,
                    "class": "emoticon emoticon-" + emoticon,
                    "data-emoticon-name": emoticon,
                    "align": "absmiddle"
                });
                ed.execCommand('mceInsertContent', false, img, {skip_undo: true});
            }, true);
            this.imagePath = imagePath;
            handlerManager.registerHandler(triggerCharCode, handler);
        }

        var emoteExt = Version.isGreaterThanOrEqualTo("7.0") ? "png" : "gif";
        var emoticons = [
            new Emoticon(")", wikiAutoformat._REGEXES_EMOTICON.SMILE, "smile", "emotions_dlg.smile", "smile." + emoteExt),
            new Emoticon("(", wikiAutoformat._REGEXES_EMOTICON.SAD, "sad", "emotions_dlg.sad", "sad." + emoteExt),
            new Emoticon("P", wikiAutoformat._REGEXES_EMOTICON.CHEEKY, "tongue", "emotions_dlg.tongue", "tongue." + emoteExt),
            new Emoticon("p", wikiAutoformat._REGEXES_EMOTICON.CHEEKY_2, "cheeky", "emotions_dlg.tongue", "tongue." + emoteExt),
            new Emoticon("D", wikiAutoformat._REGEXES_EMOTICON.LAUGH, "biggrin", "emotions_dlg.biggrin", "biggrin." + emoteExt),
            new Emoticon(")", wikiAutoformat._REGEXES_EMOTICON.WINK, "wink", "emotions_dlg.wink", "wink." + emoteExt),
            new Emoticon(")", wikiAutoformat._REGEXES_EMOTICON.THUMBS_UP, "thumbs_up", "emotions_dlg.thumbs_up", "thumbs_up." + emoteExt),
            new Emoticon(")", wikiAutoformat._REGEXES_EMOTICON.THUMBS_DOWN, "thumbs_down", "emotions_dlg.thumbs_down", "thumbs_down." + emoteExt),
            new Emoticon(")", wikiAutoformat._REGEXES_EMOTICON.INFORMATION, "information", "emotions_dlg.information", "information." + emoteExt),
            new Emoticon(")", wikiAutoformat._REGEXES_EMOTICON.TICK, "check", "emotions_dlg.check", "check." + emoteExt),
            new Emoticon(")", wikiAutoformat._REGEXES_EMOTICON.CROSS, "error", "emotions_dlg.error", "error." + emoteExt),
            new Emoticon(")", wikiAutoformat._REGEXES_EMOTICON.WARNING, "warning", "emotions_dlg.warning", "warning." + emoteExt),
            new Emoticon(")", wikiAutoformat._REGEXES_EMOTICON.PLUS, "add", "emotions_dlg.add", "add." + emoteExt),
            new Emoticon(")", wikiAutoformat._REGEXES_EMOTICON.MINUS, "forbidden", "emotions_dlg.forbidden", "forbidden." + emoteExt),
            new Emoticon(")", wikiAutoformat._REGEXES_EMOTICON.QUESTION, "help_16", "emotions_dlg.help_16", "help_16." + emoteExt),
            new Emoticon(")", wikiAutoformat._REGEXES_EMOTICON.LIGHT_ON, "lightbulb_on", "emotions_dlg.lightbulb_on", "lightbulb_on." + emoteExt),
            new Emoticon(")", wikiAutoformat._REGEXES_EMOTICON.LIGHT_OFF, "lightbulb", "emotions_dlg.lightbulb", "lightbulb." + emoteExt),
            new Emoticon(")", wikiAutoformat._REGEXES_EMOTICON.YELLOW_STAR, "star_yellow", "emotions_dlg.star_yellow", "star_yellow." + emoteExt),
            new Emoticon(")", wikiAutoformat._REGEXES_EMOTICON.YELLOW_STAR_2, "star_yellow", "emotions_dlg.star_yellow", "star_yellow." + emoteExt),
            new Emoticon(")", wikiAutoformat._REGEXES_EMOTICON.RED_STAR, "star_red", "emotions_dlg.star_red", "star_red." + emoteExt),
            new Emoticon(")", wikiAutoformat._REGEXES_EMOTICON.GREEN_STAR, "star_green", "emotions_dlg.star_green", "star_green." + emoteExt),
            new Emoticon(")", wikiAutoformat._REGEXES_EMOTICON.BLUE_STAR, "star_blue", "emotions_dlg.star_blue", "star_blue." + emoteExt),
            new Emoticon(")", wikiAutoformat._REGEXES_EMOTICON.FLAG, "flag", "emotions_dlg.flag", "flag." + emoteExt),
            new Emoticon(")", wikiAutoformat._REGEXES_EMOTICON.FLAG_OFF, "flag_grey", "emotions_dlg.flag_grey", "flag_grey." + emoteExt)
        ];

        // Preload the emoticon images for faster auto-complete response
        var emoticonImages = new Array();
        for (var i = 0; i < emoticons.length; i++) {
            emoticonImages[i] = new Image();
            emoticonImages[i].src = emoticons[i].imagePath;
        }

        /* End Emoticons */
        function formatInlineText(inlineFormat, content) {
            var format = ed.formatter.get(inlineFormat)[0];
            var domNode = ed.dom.create(format.inline, {style: format.styles});

            domNode.appendChild(document.createTextNode(content + "{$caret}"));
            ed.execCommand('mceInsertContent', false, encodeDomNode(domNode), {skip_undo: true});
            ed.formatter.remove(inlineFormat); //disable the format so that typing after doesn't come out formatted
        }

        //add handlers for common inline wikimarkup styling
        var handlerFormatMap = {
            '*': 'bold',
            '_': 'italic',
            '~': 'subscript',
            '^': 'superscript',
            '+': 'underline',
            '-': 'strikethrough'
        };
        $.each(handlerFormatMap, function (handler, format) {
            var regex = new RegExp('(?:[\\s\\xA0\\u200b\\uFEFF]+|^)(\\' + handler + '(?=[^\\s' + handler + '])([^' + handler + ']*?[^\\s]))$'); //\\xA0 is &nbsp; which is used in ie to create cursor targets
            handlerManager.registerHandler(handler.charCodeAt(0), createHandler(regex, function (matchGroups) { // require a space before asterisk to handle people using asterisk to denote footnotes
                formatInlineText(format, matchGroups[2]);

                Analytics.sendEvent("editor.instance.autoformat.format");
            }, true));
        });

        //register the code element with tiny's text formatter engine (move this into tinymce.init config)
        ed.formatter.register('code', {inline: 'code'});
        handlerManager.registerHandler("}".charCodeAt(0), createHandler(/(?:[\s\xA0\u200b]+|^)({{(?=[^\s])([^}]*?[^\s])})$/, function (matchGroups) {
            formatInlineText('code', matchGroups[2]);

            Analytics.sendEvent("editor.instance.autoformat.inlinecode");
        }, true));

        ed.formatter.register('cite', {inline: 'cite'});
        handlerManager.registerHandler("?".charCodeAt(0), createHandler(/(?:[\s\xA0\u200b]+|^)(\?\?(?=[^\s])([^\?]*?[^\s])\?)$/, function (matchGroups) {
            formatInlineText('cite', matchGroups[2]);

            Analytics.sendEvent("editor.instance.autoformat.cite");
        }, true));

        //headers h1-h6
        for (var i = 1; i <= 6; i++) {
            (function (count) {
                handlerManager.registerHandler(" ".charCodeAt(0), createHandler(new RegExp("^\\u200b?(h" + count + "\\.)$"), function () {
                    ed.execCommand('formatBlock', false, 'h' + count, {skip_undo: true});

                    Analytics.sendEvent("editor.instance.autoformat.header");
                }, true));
            })(i);
        }

        handlerManager.registerHandler(" ".charCodeAt(0), createHandler(/^\u200b?(bq\.)$/, function () {
            ed.execCommand('formatBlock', false, 'blockquote', {skip_undo: true});

            Analytics.sendEvent("editor.instance.autoformat.bq");
        }, true));

        /**
         * Handle em and en dashes where there are surrounding spaces
         */
        handlerManager.registerHandler(" ".charCodeAt(0), createHandler(/[^-]*[\s](\-\-\-?)$/, function (matchGroups) {
            var dash = matchGroups[1].length === 2 ? "\u2013" : "\u2014";
            ed.execCommand('mceInsertContent', false, dash, {skip_undo: true});

            Analytics.sendEvent("editor.instance.autoformat.dash");
        }, false));

        /**
         * Handle em and en dashes with words on either side of the dashes. Delay the autoformat until a space or enter is
         * pressed after the last word has been typed. That is, for "foo--bar", only trigger when a space or enter is
         * pressed after "bar".
         */
        var dashHandler = createHandler(/(([^\s-]+)(\-\-\-?)([^\s-]+))$/, function (matchGroups) {
            var dash = matchGroups[3].length === 2 ? "\u2013" : "\u2014";
            ed.execCommand('mceInsertContent', false, matchGroups[2] + dash + matchGroups[4], {skip_undo: true});
        }, false);
        handlerManager.registerHandler(" ".charCodeAt(0), dashHandler);
        handlerManager.registerHandler(13, dashHandler);

        var hrHandler = createHandler(/^\u200b?(\-\-\-\-)$/, function () {
            ed.execCommand('mceInsertContent', false, '<hr />', {skip_undo: true});

            Analytics.sendEvent("editor.instance.autoformat.hr");
        }, true);
        handlerManager.registerHandler(" ".charCodeAt(0), hrHandler);
        handlerManager.registerHandler(13, hrHandler);

        // code handler
        handlerManager.registerHandler("}".charCodeAt(0), createHandler(/(\{code)$/, function () {
            ed.execCommand('mceInsertContent', false, '<div class="code panel" style="border-width: 1px;"><div class="codeContent panelContent"><pre class="code-java">{$caret} </pre></div></div>');

            Analytics.sendEvent("editor.instance.autoformat.code");
        }, true));

        // panel handler
        handlerManager.registerHandler("}".charCodeAt(0), createHandler(/(\{panel)$/, function () {
            ed.execCommand('mceInsertContent', false, '<div class="plain panel"><panel-title>' + AJS.I18n.getText('jira.wiki.editor.operation.panel.placeholder.title')+'</panel-title><p>{$caret} </p></div>');

            Analytics.sendEvent("editor.instance.autoformat.panel");
        }, true));

        // noformat handler
        handlerManager.registerHandler("}".charCodeAt(0), createHandler(/(\{noformat)$/, function () {
            ed.execCommand('mceInsertContent', false, '<div class="preformatted panel" style="border-width: 1px;"><div class="preformattedContent panelContent"><pre>{$caret} </pre></div></div>');

            Analytics.sendEvent("editor.instance.autoformat.noformat");
        }, true));

        // table header handler
        handlerManager.registerHandler(13, createHandler(/(^\u200b?\|\|\s*(?:[^|]*\s?\|\|\s?)+$)/, function (matchGroups) {
            var tableMarkup = "<table class='confluenceTable'><tr>";
            var tableRows = "";
            var cellsEmpty = true;
            var cellWords = $(matchGroups[1].slice(2,-2).split('||')).map(function(cellWord) {
                cellWord = $.trim(this);
                cellsEmpty = cellsEmpty && cellWord === "";
                return cellWord;
            });
            if (cellsEmpty) {
                cellWords[0] = AJS.I18n.getText("editor.autoformat.sampletext.firstcell");
            }

            for (var k = 0, l = cellWords.length; k < l; k++) {
                tableMarkup += "<th class='confluenceTh'>" + cellWords[k] + "</th>";
                tableRows += "<td class='confluenceTd'>&#x200b;</td>"; // zero width space
            }
            tableMarkup += "</tr><tr>" + tableRows + "</tr></table>";
            ed.execCommand('mceInsertContent', false, tableMarkup, {skip_undo: true});
            ed.selection.select($(ed.selection.getRng(true).commonAncestorContainer).parents('table').find(cellsEmpty ? 'th' : 'td')[0].childNodes[0]);
            $(ed.selection.getRng().startContainer).parent().closest('[contenteditable="true"]').focus();

            Analytics.sendEvent("editor.instance.table.autoformat.plain");
        }, true, 'table'));

        // table row handler
        handlerManager.registerHandler(13, createHandler(/(^\u200b?\|\s?(?:[^|]*\s?\|\s?)+$)/, function (matchGroups) {
            var tableMarkup = "<table class='confluenceTable'><tr>";
            var cellsEmpty = true;
            var cellWords = $(matchGroups[1].slice(1, -1).split('|')).map(function (cellWord) {
                cellWord = $.trim(this);
                cellsEmpty = cellsEmpty && cellWord === "";
                return cellWord;
            });
            if (cellsEmpty) {
                cellWords[0] = AJS.I18n.getText('editor.autoformat.sampletext.firstcell');
            }
            for (var k = 0, l = cellWords.length; k < l; k++) {
                tableMarkup += "<td class='confluenceTd'>" + cellWords[k] + "</td>";
            }
            tableMarkup += "</tr></table>";
            ed.execCommand('mceInsertContent', false, tableMarkup, {skip_undo: true});
            cellsEmpty && ed.selection.select($(ed.selection.getRng(true).commonAncestorContainer).parents('table').find('td')[0].childNodes[0]);
            $(ed.selection.getRng().startContainer).parent().closest('[contenteditable="true"]').focus();

            Analytics.sendEvent("editor.instance.table.autoformat");
        }, true, 'table'));

        // link markup autocomplete triggered with either
        $.each({
            "]": /(?:\s|^)((\[[^\[^\]]+))$/,
            "\r": /(?:\s|^)((\[[^\[^\]]+)(?:\]))$/,
            " ": /(?:\s|^)((\[[^\[^\]]+)(?:\]))$/
        }, function (trigger, regex) {
            handlerManager.registerHandler(trigger.charCodeAt(0), createHandler(regex, function (matchGroups, range) {
                var wiki = matchGroups[1] + (matchGroups[1].endsWith("]") ? "" : "]");

                htmlRenderer.renderMarkup(wiki, ed.settings.target.getRenderParams()).then(function(rendered) {
                    // unwrap first paragraph
                    rendered = $(new DOMParser().parseFromString(rendered, 'text/html').body).find('p').unwrap().html();
                    ed.execCommand('mceInsertContent', false, rendered, {skip_undo: true});

                    Analytics.sendEvent("editor.instance.autoformat.wikilink");
                });
            }, true));
        });

        var urlHandler = createHandler(/\b(((https?|ftp):\/\/|(www\.))[\w\.\$\-_\+!\*'\(\),/\?:@=&%#~;\[\]]+)$/, function (matchGroups) {
            //if a protocol is not provided, add http:// to the url.
            var url = matchGroups[3] ? matchGroups[1] : "http://" + matchGroups[1];
            var domNode = ed.dom.create('a', {href: url});

            domNode.appendChild(document.createTextNode(matchGroups[1]));
            ed.execCommand('mceInsertContent', false, encodeDomNode(domNode), {skip_undo: true});
            ed.getDoc().execCommand('unlink', false, {});

            Analytics.sendEvent("editor.instance.autoformat.autolink");
        }, false, 'a');
        handlerManager.registerHandler(" ".charCodeAt(0), urlHandler);
        handlerManager.registerHandler(13, urlHandler);

        // eslint-disable-next-line max-len
        var emailHandler = createHandler(/\b((([a-z]|\d|[!#\$%&'\*\+\-\/=\?\^_`{\|}~]|[\u00A0-\uD7FF\uF900-\uFDCF\uFDF0-\uFFEF])+(\.([a-z]|\d|[!#\$%&'\*\+\-\/=\?\^_`{\|}~]|[\u00A0-\uD7FF\uF900-\uFDCF\uFDF0-\uFFEF])+)*)@((([a-z]|\d|[\u00A0-\uD7FF\uF900-\uFDCF\uFDF0-\uFFEF])|(([a-z]|\d|[\u00A0-\uD7FF\uF900-\uFDCF\uFDF0-\uFFEF])([a-z]|\d|-|\.|_|~|[\u00A0-\uD7FF\uF900-\uFDCF\uFDF0-\uFFEF])*([a-z]|\d|[\u00A0-\uD7FF\uF900-\uFDCF\uFDF0-\uFFEF])))\.)+(([a-z]|[\u00A0-\uD7FF\uF900-\uFDCF\uFDF0-\uFFEF])|(([a-z]|[\u00A0-\uD7FF\uF900-\uFDCF\uFDF0-\uFFEF])([a-z]|\d|-|\.|_|~|[\u00A0-\uD7FF\uF900-\uFDCF\uFDF0-\uFFEF])*([a-z]|[\u00A0-\uD7FF\uF900-\uFDCF\uFDF0-\uFFEF])))\.?)$/i, function (matchGroups) {
            var domNode = ed.dom.create('a', {href: 'mailto:' + matchGroups[1]});
            domNode.appendChild(document.createTextNode(matchGroups[1]));
            ed.execCommand('mceInsertContent', false, encodeDomNode(domNode), {skip_undo: true});
            ed.getDoc().execCommand('unlink', false, {});
        }, false, 'a');
        handlerManager.registerHandler(" ".charCodeAt(0), emailHandler);
        handlerManager.registerHandler(13, emailHandler);

        var addQuickCorrect = function (regexLiteral, replacement) {
            var handler = createHandler(regexLiteral, function () {
                ed.execCommand('mceInsertContent', false, replacement, {skip_undo: true});

                Analytics.sendEvent("editor.instance.autoformat.quickcorrect");
            }, false);
            handlerManager.registerHandler(" ".charCodeAt(0), handler);
            handlerManager.registerHandler(13, handler);
        };

        addQuickCorrect(/(?:\b|^)([Jj]ira)$/, "JIRA");
        addQuickCorrect(/(?:\b|^)(bitbucket|[Bb]itBucket)$/, "Bitbucket");
        addQuickCorrect(/(?:\b|^)(atlassian)$/, "Atlassian");
        addQuickCorrect(/(?:\b|^)([Hh]ipchat)$/, "HipChat");

        ed.on("keydown", function (nativeEvent) {
            return handlerManager.executeHandlers(getCharCode(nativeEvent), ed, nativeEvent);
        }, true);

        ed.on("keypress", function (nativeEvent) {
            return handlerManager.executeHandlers(getCharCode(nativeEvent), ed, nativeEvent);
        }, true);

    }

    tinymce.create('tinymce.plugins.jira.WikiAutoFormat', {
        init: function (ed) {
            ed.on("init", function() {
                initAutoFormat(ed);
            });
        }
    });

    tinymce.PluginManager.add('jira.wikiautoformat', tinymce.plugins.jira.WikiAutoFormat);
    tinymce.PluginManager.urls["jira.wikiautoformat"] = true;

    return {
        configure: function (instance, settings) {
            settings.plugins.push('jira.wikiautoformat');
        }
    };
});
