define("jira/editor/plugins/paste", [
    "jira/editor/tinymce",
    "jira/editor/converter/util/strings",
    "jira/editor/schema",
    "jira/editor/analytics",
    "jquery",
    "wrm/context-path",
    "dndattachment/ctrlv/utility",
    "dndattachment/ctrlv/base64decode",
    "jira/editor/wrm"
], function(
    tinymce,
    Strings,
    EditorSchema,
    Analytics,
    $,
    wrmContextPath,
    pasteUtility,
    base64decode,
    wrmResources
) {
    "use strict";

    var isAttachmentsAllowed = wrmResources.isAttachmentsAllowed();

    tinymce.create('tinymce.plugins.jira.Paste', {});

    tinymce.PluginManager.add('jira.paste', tinymce.plugins.jira.Paste);
    tinymce.PluginManager.urls["jira.paste"] = true;

    return {
        configure: function(instance, settings) {

            function addImage(data, fileName) {
                if (!isAttachmentsAllowed) {
                    return false;
                }
                if (data.match(/^(data|blob):http/)) {
                    getAndUploadImage(data, fileName);
                } else if (data.match(/^(data|blob):image/)) {
                    uploadImage(new Blob([base64decode.decodeBase64DataUri(data)]), fileName);
                }
                return true;
            }

            function uploadImage(data, fileName) {
                var file = pasteUtility.convertBlobToImage(data, fileName);
                if (!pasteUtility.dropFileToElement(file, instance.element.$wikiTextarea)) {
                    console.warn("Uploading attachment failed. Check 'allow attachment' setting.");
                }
            }

            function getAndUploadImage(src, fileName) {
                var xhr = new XMLHttpRequest();
                xhr.open("GET", src, true);
                xhr.responseType = "blob";
                xhr.onload = function (e) {
                    uploadImage(this.response, fileName);
                };
                xhr.send();
            }

            settings.plugins.push('paste');
            settings.paste_data_images = true;
            settings.paste_preprocess = function(plugin, args) {
                var schema = instance.contextManager.getPasteSchemaForContext();
                args.content = EditorSchema.sanitizeHtml(args.content, instance.editor, schema);
            };
            settings.paste_postprocess = function (plugin, args) {
                Analytics.sendEvent("editor.instance.paste");
                var $image = $(args.node).find("img:first-child:last-child");
                if ($image.length === 1 && $image.is("img")) {
                    Analytics.sendEvent("editor.instance.paste.image");
                    var src = $image.attr('src');

                    if ($image.hasClass('.emoticon') && Strings.startsWith(src, document.location.origin + wrmContextPath() + "/images/icons/emoticons/")) {
                        return;
                    }

                    var filename = src;
                    if (Strings.isAttachmentPath(src) || Strings.isThumbnailPath(src)) {
                        filename = Strings.getFilenameFromHref(src);
                    }
                    var param = {
                        source: src,
                        filename: filename,
                        width: $image.attr('width'),
                        height: $image.attr('height'),
                        id: $image.attr('id'),
                        vspace: $image.attr('vspace')
                    };
                    if (src.match(/^((data|blob):http)|(data:image)|(webkit-fake-url:\/\/)/)) {
                        filename = pasteUtility.generateFileName() + ".png";
                        if (addImage(src, filename)) {
                            param.filename = filename;
                            var result = JIRA.Editor.Tags.Templates.image(param);
                            $image.replaceWith($(result));
                        } else {
                            $image.remove();
                        }
                    } else {
                        var result = JIRA.Editor.Tags.Templates.image(param);
                        $image.replaceWith($(result));
                    }
                } else {
                    Analytics.sendEvent("editor.instance.paste.other");
                }
            };

            return settings;
        }
    };
});