define('jira/editor/plugins/mentions/mention-rte', [
    'jira/mention/mention-matcher',
    'jquery',
    "jira/mention/mention",
    "underscore",
    "jira/editor/plugins/mentions/mentions-util"
], function (
    MentionMatcher,
    jQuery,
    MentionJira,
    _,
    mentionsUtil
) {
    "use strict";

    var MentionRte = MentionJira.extend({

        init: function (issueKey, editor, skipProcessing) {
            MentionJira.prototype.init.call(this, issueKey);
            this.editor = editor;
            this.skipProcessing = skipProcessing;
        },

        /**
         * Replaces the text in text node that the caret is currently in with rendered mention.
         * The text node is replaced by 3 nodes (before text node, mention node, after text node).
         *
         * @param selectedUserName The username of the selected user.
         * @param selectedDisplayName The username of the selected user.
         */
        _replaceCurrentUserName: function (selectedUserName, selectedDisplayName) {
            if (this._selectMentionText()) {
                this.editor.execCommand('mceInsertContent', false, this._renderMentionLink(selectedUserName, selectedDisplayName));
            }
        },

        _selectMentionText: function () {
            var doc = this._getDocument();
            var selection = this._getSelection();
            if (selection && selection.isCollapsed) {
                var obj = this._getNodeForUsernameRetrieval();
                var nodes = this._getPreviousSiblingsTextNodes(obj.node);

                if (nodes.length > 0) {
                    var startNode = nodes[0];
                    var endNode = nodes[nodes.length - 1];

                    var offset = obj.offset;
                    var beforeCaret = this._getStringFormPreviousSiblingsTextNodes(nodes, offset);
                    var wordStartIndex = MentionMatcher.getLastWordBoundaryIndex(beforeCaret, true);

                    var range = doc.createRange();
                    range.setStart(startNode, wordStartIndex + 1);
                    range.setEnd(endNode, offset);
                    this.editor.selection.setRng(range);
                    return true;
                }
            } else {
                return false;
            }
        },

        /**
         * Renders mention as string.
         * Currently renders as A tag with contenteditable attribute set to false to prevent editing inside of mention.
         * @param username
         * @param displayName
         * @returns {string}
         * @private
         */
        _renderMentionLink: function (username, displayName) {
            return JIRA.Editor.Tags.Templates.mentionlink({username: username, displayname: displayName});
        },

        /**
         * Gets the username which the caret is currently next to from the text node that the caret is currently in.
         * Selection has to be collapsed.
         */
        _getUserNameFromInput: function () {
            if (_.isFunction(this.skipProcessing) && this.skipProcessing()) {
                return null;
            }

            var selection = this._getSelection();
            if (selection && selection.isCollapsed) {
                var obj = this._getNodeForUsernameRetrieval();
                var nodes = this._getPreviousSiblingsTextNodes(obj.node);
                var text = this._getStringFormPreviousSiblingsTextNodes(nodes, obj.offset);
                this.currentUserName = MentionMatcher.getUserNameFromCurrentWord(text, text.length);
                return this.currentUserName;
            } else {
                return null;
            }
        },

        _getStringFormPreviousSiblingsTextNodes: function (nodes, lastNodeOffset) {
            var str = "";
            for (var i = 0; i < nodes.length; i = i + 1) {
                var node = nodes[i];
                if (i === nodes.length - 1) {
                    str = str + node.nodeValue.substring(0, lastNodeOffset);
                } else {
                    str = str + node.nodeValue;
                }
            }
            return str;
        },

        _getPreviousSiblingsTextNodes: function (startNode) {
            var arr = [];

            if (startNode && startNode.nodeType === Node.TEXT_NODE) {
                for (var node = startNode; node; node = node.previousSibling) {
                    if (node.nodeType === Node.TEXT_NODE) {
                        arr.unshift(node);
                        if (node.nodeValue.indexOf("@") >= 0 || node.nodeValue.indexOf("[~") >= 0) {
                            return arr;
                        }
                    } else {
                        break;
                    }
                }
            }
            return [];
        },

        _getNodeForUsernameRetrieval: function () {
            return mentionsUtil.getPreviousNode(this._getDocument());
        },

        _getSelection: function () {
            if (this.$textarea) {
                var nativeElement = this.$textarea.get(0);
                if (nativeElement && nativeElement.ownerDocument) {
                    return nativeElement.ownerDocument.getSelection();
                }
            }
            return null;
        },

        _getDocument: function () {
            return this.$textarea.get(0).ownerDocument;
        },

        _getOffsetTarget: function () {
            return jQuery(this.editor.getContainer());
        }
    });

    return MentionRte;
});