define("jira/editor/plugins/mentions/jira-mentions", [
    "jira/editor/tinymce",
    "jquery",
    "jira/editor/plugins/mentions/mention-rte",
    "jira/editor/plugins/mentions/mentions-util"
], function (
    tinymce,
    $,
    Mention,
    mentionsUtil
) {
    "use strict";

    var BACKSPACE = 8;

    var FORWARD_KEYS = {
        9: "Tab",
        13: "Return",
        27: "Esc",
        38: "Up",
        40: "Down"
    };

    var triggerAuiEvent = function (evt) {
        var $event = new $.Event("aui:keydown");
        $event.key = FORWARD_KEYS[evt.which];
        $event.shiftKey = evt.shiftKey;
        $event.ctrlKey  = evt.ctrlKey;
        $event.altKey   = evt.altKey;

        $event.target = evt.target;
        arguments[0] = $event;
        $.event.trigger($event, arguments, document, true);
    };

    var overrideEditorEvents = function (editor, controller) {
        var doc = controller.$textarea.get(0).ownerDocument;

        controller.mentioningInProgress = false;
        controller.layerController.bind("showLayer", function () {
            controller.mentioningInProgress = true;
        }).bind("hideLayer", function () {
            controller.mentioningInProgress = false;
        });

        editor.on('keydown',
            function mentionsKeydownOverride(evt) {
                var forwardToParentDoc = evt.which in FORWARD_KEYS && controller.mentioningInProgress === true;

                if( forwardToParentDoc ) {
                    evt.preventDefault();
                    evt.stopPropagation();
                    triggerAuiEvent(evt);
                } else if (evt.keyCode === BACKSPACE) {
                    if( mentionsUtil.isCaretAfterMention(doc) ) {
                        var mentionNode = mentionsUtil.getMentionBeforeCaret(doc);
                        mentionsUtil.selectMention(mentionNode, evt, this);
                    }
                }
            },
            true);
    };

    tinymce.create('tinymce.plugins.jira.JiraMentions', {
        init: function (ed) {
            var settings = ed.settings;

            if (settings.jiramentions.mentionable) {
                ed.on('init', function () {
                    var body = ed.getBody();

                    var skipProcessing = function () {
                        var $anchor = $(ed.selection.getSel().anchorNode);
                        if ($anchor.parents('pre').length > 0 || $anchor.parents('div.panelHeader,panel-title').length > 0) {
                            return true;
                        }
                    };

                    var issueKey = settings.jiramentions.issuekey;
                    var projectKey = settings.jiramentions.projectkey;
                    $(body).attr("data-issuekey", issueKey);
                    $(body).attr("data-projectkey", projectKey);
                    if (!body._controller) {
                        body._controller = new Mention(issueKey, ed, skipProcessing);
                    }

                    body._controller.textarea(body);

                    overrideEditorEvents(ed, body._controller);
                });
            }
        }
    });

    tinymce.PluginManager.add('jira.jiramentions', tinymce.plugins.jira.JiraMentions);
    tinymce.PluginManager.urls["jira.jiramentions"] = true;

    return {
        configure: function (instance, settings) {
            settings.plugins.push('jira.jiramentions');

            var $wikiTextarea = instance.element.$wikiTextarea;

            var data = $wikiTextarea.data();
            settings.jiramentions = {
                issuekey: data.issuekey,
                projectkey: data.projectkey,
                mentionable: $wikiTextarea.hasClass("mentionable")
            };
        }
    };
});