define("jira/editor/schema", [
    "jira/editor/schema-builder",
    "jira/richeditor/editor-supported-operations",
    "jira/editor/tinymce",
    "jquery"
], function(
    SchemaBuilder,
    supportedOperations,
    tinymce,
    $
) {
    "use strict";

    var unbreakableNodes = ["strong", "b", "em", "i", "tt", "code", "sub", "sup", "del", "cite", "panel-title"];
    var unsupportedNesting = [["td", "table"], ["th", "table"], ["td", "p"], ["th", "p"]];
    var unwantedAttributes = [];
    var extendedValidElements = ["strong/b"];
    unwantedAttributes.push("xml:lang");
    unwantedAttributes.push("xml:space");

    var EditorSchema = {};
    EditorSchema.getSchemaSpecBuilder = function (builder) {
        if (undefined === builder) {
            builder = new SchemaBuilder();
        }
        return builder
            .withValidElements(supportedOperations.getRequiredHtmlTags())
            .withNoBreak(unbreakableNodes)
            .withNoSelfNest(unbreakableNodes)
            .withNoNest(unsupportedNesting)
            .withUnwantedAttributes(unwantedAttributes)
            .withExtendedValidElements(extendedValidElements)
            .withCustomElement("panel-title", ["span", "#text"], ["style"])
            .setValidChildren("pre", "br|span|#text|pre|p|panel-title")
    };
    EditorSchema.getPasteSchemaSpecBuilder = function (builder) {
        if (undefined === builder) {
            builder = EditorSchema.getSchemaSpecBuilder();
        }
        return builder
            .withUnwantedAttributes(["contenteditable"])
    };
    EditorSchema.getPasteInsidePreSchemaSpecBuilder = function (builder) {
        if (undefined === builder) {
            builder = new SchemaBuilder();
        }
        return builder
            .withValidElements(["br", "span", "p", "pre"])
            .withUnwantedAttributes(["style"])
    };

    EditorSchema.getSchemaFromSpec = function (schemaSpec) {
        var schema = new tinymce.html.Schema({
            schema: schemaSpec.getType(),
            custom_elements: schemaSpec.getCustomElements(),
            valid_children: schemaSpec.getValidChildren(),
            valid_elements: schemaSpec.getValidElements(),
            extended_valid_elements: schemaSpec.getExtendedValidElements()
        });
        schemaSpec.fixChildren(schema);
        return schema;
    };

    function sanitizeMarkup(html) {
        var node = $('<div>').addClass('richeditor-sanitize').append(html)[0];
        try {
            $(node).appendTo(document.body);
            return AJS.escapeHTML(node.innerText || node.textContent);
        } finally {
            $(node).remove();
        }
    }

    EditorSchema.sanitizeHtml = function sanitizePastedHtml(html, editor, schemaSpec) {
        var $selection = $(editor.selection.getStart()).add(editor.selection.getEnd()).add(editor.selection.getNode());

        if ($selection.is('pre, pre > *')) {
            html = '<pre class="'+$selection.attr("class")+'">' + sanitizeMarkup(html) + '</pre>';
        }

        if ($selection.is('panel-title, panel-title > *')) {
            html = sanitizeMarkup(html);
        }

        var schema = EditorSchema.getSchemaFromSpec(schemaSpec);
        var htmlDomParser = new tinymce.html.DomParser(editor.settings, schema);
        var htmlSerializer = new tinymce.html.Serializer(editor.settings, schema);

        var dom = htmlDomParser.parse(html, {
            forced_root_block: undefined
        });
        return htmlSerializer.serialize(dom);
    };

    return EditorSchema;

    /**
     * Uses tinymce schema but limit it to given set of elements.
     * Thus tinymce convention is used regarding valid elements, children and so on, please see the doc:
     * {@link https://www.tinymce.com/docs/configure/content-filtering/#valid_elements}
     *
     * @param {Array} elements html tags which are allowed to use
     * @param {Array} withoutAttributes attributes which will we removed from schema
     * @param {String} baseSchema any supported value by tinymce such as "html5", "html5-strict", "html4"
     * @returns {Array} valid_elements for tinymce
     */
    function decorateElements(elements, withoutAttributes, baseSchema) {
        var srcSchema = new tinymce.html.Schema({
            schema: baseSchema
        });
        var srcElements = srcSchema.elements;

        // add constrains
        srcElements.img.attributesRequired = ["src"];

        return elements.map(function (el) {
            var srcSpec = srcElements[el];
            if (!srcSpec) {
                // remove element if it does not exist in schema
                return false;
            }

            // see @link: "!" is "Enables removal of elements with no attributes. They can still have content though."
            var postfix = (srcSpec.removeEmptyAttrs) ? "!" : "";
            var prefix = "";
            if (srcSpec.paddEmpty) {
                // see @link: "#" is "Enables padding of empty elements. This will pad with &nbsp; if they are empty."
                prefix = "#";
            } else if (srcSpec.removeEmpty) {
                // see @link: "-" is "Enables removal of empty elements such as <strong />"
                prefix = "-";
            }

            var attributes = srcSpec.attributesOrder.map(function (attr) {
                var prefix = "";
                if ((srcSpec.attributesRequired || []).indexOf(attr) !== -1) {
                    // see @link: "!" is "Makes attributes required. If none of the required attributes are set,
                    //  the element will be removed."
                    prefix = "!";
                }
                return prefix + attr;
            }).filter(function (attr) {
                // remove unwanted attributes such as "contenteditable"
                return withoutAttributes.indexOf(attr) === -1;
            });

            // required format by tinymce, example: #a[!href|title]
            return prefix + el + postfix + "[" + attributes.join("|") + "]";
        }).filter(function (el) {
            return !!el;
        });
    }
});