AJS.test.require(["com.atlassian.jira.plugins.jira-editor-plugin:schema"], function () {
    "use strict";

    QUnit.config.testTimeout = 5000;

    var _ = require("underscore");
    var EditorSchema = require("jira/editor/schema");

    var someForbiddenEventHandlers = ["onload", "onerror", "onmouseover", "onmouseout", "onfocus", "onclick", "ondblclick", "ondrag",
        "onmousedown", "onmousemove", "onmouseup", "onmousewheel", "oncopy", "onpaste", "oncut"];

    module("EditorSchema - default schema", {
        setup: function () {
            this.schema = EditorSchema.getSchemaFromSpec(EditorSchema.getSchemaSpecBuilder().build());
        }
    });

    commonTests();

    test("Should allow contenteditable attribute", function (assert) {
        assert.ok(isStrictValid("div", ["contenteditable"], this.schema));
    });


    module("EditorSchema - paste schema", {
        setup: function () {
            this.schema = EditorSchema.getSchemaFromSpec(EditorSchema.getPasteSchemaSpecBuilder().build());
        }
    });

    commonTests();

    test("Should not allow contenteditable attribute", function (assert) {
        assert.ok(!isStrictValid("div", ["contenteditable"], this.schema));
    });


    module("EditorSchema - schema for pre", {
        setup: function () {
            this.schema = EditorSchema.getSchemaFromSpec(EditorSchema.getPasteInsidePreSchemaSpecBuilder().build());
        }
    });

    test("Should allow some tags but without 'style' attribute", function () {
        var supportedTags = ["br", "span", "p"];

        supportedTags.forEach(function (tagName) {
            ok(this.schema.isValid(tagName), "Should allow '" + tagName + "' tag");
            ok(!isStrictValid(tagName, ["style"], this.schema), "Should not allow 'style' attribute for " + tagName);
        }.bind(this));

        ok(this.schema.isValidChild("span", "#text"), "'#text' node should be allowed inside <span>")
        ok(this.schema.isValidChild("p", "#text"), "'#text' node should be allowed inside <p>")
    });

    test("Should not allow unsupported tags", function (assert) {
        var someUnsupportedTags = ["div", "panel-title", "table", "b", "strong", "sup", "sub", "img",
            "tt", "code", "del", "cite", "h1", "applet", "basefont", "audio", "canvas",
            "command", "dialog", "embed", "iframe", "noframes", "script", "noscript", "object",
            "picture", "summary", "time", "video", "wbr"];

        someUnsupportedTags.forEach(function (tagName) {
            assert.ok(!this.schema.isValid(tagName), "Should not allow '" + tagName + "' tag");
        }.bind(this));
    });


    function commonTests() {
        test("Schema should be created", function (assert) {
            assert.ok(this.schema);
        });

        test("Should not allow explicit <body>", function (assert) {
            assert.ok(!this.schema.isValid("body"), "body element");
            assert.ok(!isStrictValid("body", [], this.schema), "body without attributes");
            assert.ok(!isStrictValid("body", ["onload"], this.schema), "body with onload attribute");
        });

        test("Should not allow <a> with event handlers", function (assert) {
            someForbiddenEventHandlers.forEach(function (eventName) {
                assert.ok(!this.schema.isValid("a", eventName), eventName);
            }.bind(this));
        });

        test("Should allow <a> with attributes", function (assert) {
            assert.ok(isStrictValid("a", ["href", "title", "target", "rel", "name"], this.schema));
        });

        test("Should not allow <img> with event handlers", function (assert) {
            someForbiddenEventHandlers.forEach(function (eventName) {
                assert.ok(!this.schema.isValid("img", eventName), eventName);
            }.bind(this));
        });

        test("Should not allow <img> without attributes", function (assert) {
            assert.ok(!isStrictValid("img", [], this.schema));
        });

        test("Should allow <img> with src specified", function (assert) {
            assert.ok(isStrictValid("img", ["src"], this.schema));
        });

        test("Should allow <img> with attributes", function (assert) {
            assert.ok(isStrictValid("img", ["src", "class", "height", "width", "align", "alt", "border"], this.schema));
        });

        test("Should allow <img> inside <span>", function (assert) {
            assert.ok(this.schema.isValidChild("span", "img"))
        });

        test("Should allow headings", function (assert) {
            var headings = ["h1", "h2", "h3", "h4", "h5", "h6"];

            headings.forEach(function (tagName) {
                assert.ok(this.schema.isValid(tagName), "Should allow '" + tagName + "' tag");
            }.bind(this));
        });

        test("Should not allow unsupported tags", function (assert) {
            var someUnsupportedTags = ["applet", "basefont", "audio", "canvas",
                "command", "dialog", "embed", "iframe", "noframes", "script", "noscript", "object",
                "picture", "summary", "time", "video", "wbr"];

            someUnsupportedTags.forEach(function (tagName) {
                assert.ok(!this.schema.isValid(tagName), "Should not allow '" + tagName + "' tag");
            }.bind(this));
        });

        test("Should allow tables", function (assert) {
            assert.ok(isStrictValid("table", ["class"], this.schema));
            assert.ok(isStrictValid("td", ["class"], this.schema));
        });

        test("Should not allow nested tables", function (assert) {
            assert.ok(!this.schema.isValidChild("td", "table"));
            assert.ok(!this.schema.isValidChild("th", "table"));
        });

        test("Should not allow paragraph inside table cells", function (assert) {
            assert.ok(!this.schema.isValidChild("td", "p"));
            assert.ok(!this.schema.isValidChild("th", "p"));
        });

        test("Should allow panels", function (assert) {
            assert.ok(isStrictValid("div", ["class", "style"], this.schema));
            assert.ok(isStrictValid("div", ["class"], this.schema));
            assert.ok(isStrictValid("div", [], this.schema));
        });

        test("Should not allow forms", function (assert) {
            ["form", "input", "select"].forEach(function (tagName) {
                assert.ok(!this.schema.isValid(tagName), tagName);
            }.bind(this));
        });

        test("Should support custom element", function (assert) {
            var elName = "panel-title";
            ok(this.schema.isValid(elName), elName);
            ok(this.schema.isValidChild(elName, "#text"));
            ok(this.schema.isValidChild(elName, "span"));

            ["h2", "b", "strong", "p", "div", "u", "em", "table", "img", "br"].forEach(function(tagName) {
                assert.ok(!this.schema.isValidChild(elName, tagName), "Should not allow '" + tagName + "' inside <" + elName + ">");
            }.bind(this));
        });

        // known issues:
        // 1. <span> has to be allowed, because:
        //  - code macro uses <span> to colorize text
        //  - TinyMCE sometimes eat whole <pre> when <span> is forbidden
        // However it leads to allowing users to colorize text and inside do whatever they want.
        test("Should allow only new lines, span and #text in <pre>", function (assert) {
            ok(this.schema.isValidChild("pre", "br"), "Should allow <br> inside <pre>");
            ok(this.schema.isValidChild("pre", "#text"), "Should allow #text inside <pre>");
            ok(this.schema.isValidChild("pre", "span"), "Should allow <span> inside <pre>");
            ok(this.schema.isValidChild("pre", "pre"), "Should allow <pre> inside <pre>");
            ok(this.schema.isValidChild("pre", "panel-title"), "Should allow <panel-title> inside <pre>");

            ["h2", "b", "strong", "div", "ins", "u", "em", "i", "table", "img"].forEach(function(tagName) {
                assert.ok(!this.schema.isValidChild("pre", tagName), "Should not allow '" + tagName + "' inside <pre>");
            }.bind(this));
        });
    }

    /**
     * Checks if given tag with attributes is valid
     *  according to the schema.
     *
     * @param {string} tagName
     * @param {Array} attributes
     * @param {tinymce.html.Schema} schema
     * @returns Boolean true if tagName is valid and
     *   attributes are valid, and required attributes
     *   are present; false otherwise
     */
    function isStrictValid(tagName, attributes, schema) {
        if (!schema.isValid(tagName)) {
            return false;
        }

        var rule = schema.getElementRule(tagName);
        return (0 === _.difference(attributes, Object.keys(rule.attributes)).length
            && 0 === _.difference(rule.attributesRequired, attributes).length);
    }
});