define("jira/editor/instance", [
    'jira/editor/analytics',
    'jira/editor/analytics-shortcuts',
    'jira/util/navigator',
    'jira/editor/tinymce',
    'jira/editor/context-manager',
    'jira/editor/context-detector',
    "jira/editor/schema",
    "jira/editor/selection",
    'jquery',
    'backbone',
    'underscore'
], function(
    Analytics,
    AnalyticsShortcuts,
    Navigator,
    tinymce,
    ContextManager,
    ContextDetector,
    EditorSchema,
    EditorSelection,
    $,
    Backbone,
    _
) {
    var EditorInstance = function(element, options) {
        this.element = element;
        this.options = _.extend({}, options);
    };

    _.extend(EditorInstance.prototype, Backbone.Events);

    EditorInstance.prototype.init = function(editor) {
        this.editor = editor;

        this.operationSelectedListeners = {};
        this.operationSelectedState = {};

        this.operationOverride = {};

        this.analyticsShortcuts = new AnalyticsShortcuts(this.editor);
        this.selection = new EditorSelection(this.editor);

        this.editor.on('NodeChange', function (e) {
            var nodeChangePerf = Analytics.startMeasure();
            Object.keys(this.operationSelectedListeners).forEach(function (key) {
                var previousState = this.operationSelectedState[key];
                var currentState = this._getOperationState(key);
                if (currentState !== previousState) {
                    this.operationSelectedListeners[key].forEach(function (listener) {
                        listener(currentState);
                    });
                }
            }.bind(this));

            if (!e) {
                return;
            }

            if (e.element.nodeName.toLowerCase() === 'img') {
                this.editor.fire('content');
            }

            var contextDetectionPerf = Analytics.startMeasure();
            var node = $(e.element);
            this.trigger("selection:update", {
                insidePreformatted: ContextDetector.detectPre(node),
                preformattedSelected: ContextDetector.detectPreWithinSelection(this.editor.selection.getContent()),
                insideTable: ContextDetector.detectTable(node),
                insideA: ContextDetector.detectA(node)
            });
            contextDetectionPerf.measure('nodechange.context-detection');
            nodeChangePerf.measure('nodechange');
        }.bind(this));

        this.editor.on("change SetContent blur", this._onChange.bind(this));
        this.editor.on("keyup", _.debounce(this._onChange.bind(this), 1000));

        this.editor.on('init', function (e) {
            var editor = e.target;
            ['tt', 'del', 'sup', 'sub', 'cite'].forEach(function (tag) {
                editor.formatter.register(tag, {block: tag, remove: 'all'});
            } );

            //unbind default shortcuts
            var defaultShortcutsToDelete = ['meta+b', 'meta+i', 'meta+u'];
            for (var i = 1; i <= 9; i++) {
                defaultShortcutsToDelete.push('access+' + i);
            }
            var noop = function () {
            };
            defaultShortcutsToDelete.forEach(function (shortcut) {
                editor.addShortcut(shortcut, 'deleted_shortcut', noop);
            });
        });

        editor.on('keydown', function(e) {
            if (e.isDefaultPrevented()) {
                return;
            }

            // we need this to preserve correct table markup, which does not support multiple
            // paragraphs inside table cells
            var $selectionStart = $(editor.selection.getStart());

            if (e.keyCode === tinymce.util.VK.ENTER && !e.shiftKey
                && $selectionStart.is('td > p, th > p, th > br, td > br, td, th')) {
                e.preventDefault();

                editor.execCommand("InsertLineBreak", false, e);
            }

            if (e.keyCode === tinymce.util.VK.ENTER && ($selectionStart.is('panel-title, panel-title *') || $selectionStart.parent().hasClass('panelHeader'))) {
                e.preventDefault();
            }
        });

        var getBrowserAnalyticsName = function() {
            if ((/(Edge)\/(\d+)\.(\d+)/).test(Navigator._getUserAgent())) {
                return 'edge';
            } else if (Navigator.isIE()) {
                return 'ie';
            } else if (Navigator.isChrome()) {
                return 'chrome';
            } else if (Navigator.isMozilla()) {
                return 'firefox';
            } else if (Navigator.isSafari()) {
                return 'safari';
            }
            return '';
        };

        Analytics.sendEvent("editor.instance.init");
        Analytics.sendEvent("bundled.editor.instance.init");
        var browserName = getBrowserAnalyticsName();
        if (browserName) {
            Analytics.sendEvent('editor.instance.init.' + browserName);
        }

        this.contextManager = new ContextManager(this);
        editor.contextManager = this.contextManager;
    };

    EditorInstance.prototype.getId = function () {
        return this.editor.id;
    };

    /**
     * Relay event handler to TinyMCE instance
     * @param name
     * @param fn
     * @param prepend
     */
    EditorInstance.prototype.relayEvent = function(name, fn, prepend) {
        this.editor.on(name, function (e) {
            fn(e);
        }, prepend);
    };

    //TODO define contract on generated HTML content
    EditorInstance.prototype.getAllowedOperations = function () {
        return [
            "paragraph", "h1", "h2", "h3", "h4", "h5", "h6", "monospace", "paragraph-quote", "block-quote", "delete", "superscript", "subscript",
            "cite",
            {
                name: "icon",
                values: [
                    ":)", ":(", ":P", ":D", ";)", "(y)", "(n)", "(i)", "(/)", "(x)", "(!)", "(+)", "(-)", "(?)", "(on)", "(off)", "(*)", "(*r)", "(*g)", "(*b)", "(*y)"
                ]
            }, "bold", "italic", "underline", "color",
            "bullet-list", "numbered-list", "mention", "table", "code", "noformat", "panel",
            "hr", "speech", "link", "link-mail", "link-anchor", "link-attachment", "image", "image-attachment", "attachment",
            "editorInsertContent", "editorInsertContentInNewLine", "editorReplaceContent", "editorReplaceContentInNewLine"
        ];
    };

    EditorInstance.prototype._isOperationSupported = function (name) {
        var allowed = this.getAllowedOperations().filter(function (el) {
            if (el instanceof Object) {
                return el.name === name;
            } else {
                return el === name;
            }
        });
        return allowed.length > 0;
    };

    EditorInstance.prototype._assertOperationIsSupported = function (operationName) {
        if (!this._isOperationSupported(operationName)) {
            console.error("Operation not supported:", operationName);
        }
    };

    EditorInstance.prototype._selectedTextSanitized = function () {
        return EditorSchema.sanitizeHtml(this.editor.selection.getContent(), this.editor, this.pasteInsidePreSchemaSpec);
    };

    EditorInstance.prototype.executeOperation = function (operationName, args) { // eslint-disable-line complexity
        this._assertOperationIsSupported(operationName);
        var tinymceMapped = this._mapOperationNameToTinymce(operationName);

        var funOverride = this.operationOverride[operationName];
        if (funOverride) {
            Analytics.sendEvent("editor.instance.operation." + operationName);
            funOverride(args);
            return true;
        }

        if ('editorReplaceContentInNewLine' === operationName) {
            this.editor.execCommand('mceReplaceContent', false, '<br />' + args.content);
        } else if ('editorReplaceContent' === operationName) {
            this.editor.execCommand('mceReplaceContent', false, args.content);
        } else if ('editorInsertContentInNewLine' === operationName) {
            this.editor.insertContent('<br />' + args.content);
        } else if ('editorInsertContent' === operationName) {
            this.editor.insertContent(args.content);
        } else if ('hr' === operationName) {
            this.editor.insertContent('<hr />');
        } else if ('color' === operationName) {
            this.editor.execCommand('ForeColor', true, args.color);
        } else if (['h1','h2','h3','h4','h5','h6','paragraph','paragraph-quote','block-quote','monospace','cite'].indexOf(operationName) > -1) {
            this.editor.execCommand('mceToggleFormat', true, tinymceMapped);
        } else if (['bold', 'italic', 'underline', 'delete', 'superscript', 'subscript'].indexOf(operationName) > -1) {
            if (this.selection.trimSelection()) {
                if (!this.selection.hasSelection()) {
                    Analytics.sendEvent("editor.instance.selection.collapsed", {op: operationName});
                    return false;
                } else {
                    Analytics.sendEvent("editor.instance.selection.trimmed", {op: operationName});
                }
            }

            this.editor.execCommand(tinymceMapped, true);
        } else if (['bullet-list', 'numbered-list'].indexOf(operationName) > -1) {
            this.editor.execCommand(tinymceMapped, true);
        } else if ("icon" === operationName) {
            var imageContextPath;
            if (AJS && AJS.contextPath) {
                imageContextPath = AJS.contextPath();
            } else {
                imageContextPath = '';
            }
            var sourceImageUrl = imageContextPath + '/images/icons/emoticons/' + this._emoticonSourceMap(args.icon);
            this.editor.insertContent('<img class="emoticon" src="' + sourceImageUrl + '" height="16" width="16" align="absmiddle" alt="" border="0">');
        } else if ("attachment" === operationName && args.attachment) {
            this.editor.insertContent(args.attachment);
        } else if ("code" === operationName) {
            var content = (this._selectedTextSanitized() || AJS.I18n.getText('jira.editor.macro.code.placeholder') + '\n');
            this.editor.selection.setContent('<pre class="code panel" data-language="code-java">'+content+'</pre>');
        } else if ("panel" === operationName) {
            var content = (this.editor.selection.getContent() || AJS.I18n.getText('jira.wiki.editor.operation.panel.placeholder'));
            this.editor.selection.setContent('<div class="plain panel" style="border-width: 1px;"><panel-title>' +  AJS.I18n.getText('jira.wiki.editor.operation.panel.placeholder.title')+'</panel-title>'+
                '<p>'+content+'</p></div>');
        } else if ("noformat" === operationName) {
            var content = (this._selectedTextSanitized() || AJS.I18n.getText('jira.wiki.editor.operation.noFormat.placeholder'));
            this.editor.selection.setContent('<pre class="noformat panel">'+content+'</pre>');
        } else if ("table" === operationName) {
            var content = (this.editor.selection.getContent() || (AJS.I18n.getText('jira.wiki.editor.operation.table.placeholder.column') + ' A1'));
            this.editor.selection.setContent('<div class="table-wrap"><table class="confluenceTable mce-item-table" data-mce-selected="1"><tbody><tr><th class="confluenceTh">' +
                AJS.I18n.getText('jira.wiki.editor.operation.table.placeholder.heading') + ' 1</th><th class="confluenceTh">' +
                AJS.I18n.getText('jira.wiki.editor.operation.table.placeholder.heading') + ' 2</th></tr><tr><td class="confluenceTd">' +
                content + '</td><td class="confluenceTd">Col A2</td></tr></tbody></table></div>');
            Analytics.sendEvent("editor.instance.table.toolbar");
        } else if (["mention", "speech", "link", "link-mail", "link-anchor", "link-attachment", "image", "image-attachment"].indexOf(operationName) > -1) {
            console.warn('Not supported yet ' + operationName);
            return false;
        } else {
            console.warn('Unsupported operation ' + operationName);
            return false;
        }

        Analytics.sendEvent("editor.instance.operation." + operationName);
        this.trigger("content");
        return true;
    };

    /**
     *
     * @param operationName
     * @param listener listener function with 1 parameter indicating if button should be selected/not selected
     */
    EditorInstance.prototype.bindOperationSelectedListener = function (operationName, listener) {
        var ar = this.operationSelectedListeners[operationName] || [];
        ar.push(listener);
        this.operationSelectedListeners[operationName] = ar;
    };

    EditorInstance.prototype.unbindOperationSelectedListener = function (operationName, listener) {
        var ar = this.operationSelectedListeners[operationName] || [];
        var index = ar.indexOf(listener);
        if (index > -1) {
            ar.splice(index, 1);
        }
    };

    EditorInstance.prototype._getOperationState = function (operationName) {
        return this.editor.queryCommandState(this._mapOperationNameToTinymce(operationName));
    };

    EditorInstance.prototype._mapOperationNameToTinymce = function (operationName) {
        var map = {
            'bold': 'Bold',
            'italic': 'Italic',
            'underline': 'Underline',
            'bullet-list': 'InsertUnorderedList',
            'numbered-list': 'InsertOrderedList',
            'blockquote': 'mceBlockQuote',
            'paragraph': 'p',
            'paragraph-quote': 'blockquote',
            'block-quote': 'blockquote',
            'monospace': 'tt',
            'delete': 'strikethrough',
            'superscript': 'superscript',
            'subscript': 'subscript'
        };
        if (operationName in map) {
            return map[operationName];
        }
        return operationName;
    };

    EditorInstance.prototype._emoticonSourceMap = function(emoticonName) {
        var emoticonSourceMap = {
            ':)': 'smile.png', // We can't distinguish between ':-)' and ':)' , they render the same
            ':(': 'sad.png',
            ':P': 'tongue.png',
            ':D': 'biggrin.png',
            ';)': 'wink.png',
            '(y)': 'thumbs_up.png',
            '(n)': 'thumbs_down.png',
            '(i)': 'information.png',
            '(/)': 'check.png',
            '(x)': 'error.png',
            '(!)': 'warning.png',

            '(+)': 'add.png',
            '(-)': 'forbidden.png',
            '(?)': 'help_16.png',
            '(on)': 'lightbulb_on.png',
            '(off)': 'lightbulb.png',
            '(*)': 'star_yellow.png', // We can't distinguish between '(*y)' and '(*)', they render the same
            '(*r)': 'star_red.png',
            '(*g)': 'star_green.png',
            '(*b)': 'star_blue.png',
            '(*y)': 'star_yellow.png',

            '(flag)' : 'flag.png',
            '(flagoff)' : 'flag_grey.png'
        };
        if (emoticonName in emoticonSourceMap) {
            return emoticonSourceMap[emoticonName];
        }
        return emoticonName;
    };

    /**
     * focuses the editor (places dom focus in the editor).
     */
    EditorInstance.prototype.focus = function () {
        this.editor.focus();
    };

    /**
     * destroys instance and removes used resources
     */
    EditorInstance.prototype.destroy = function () {
        this.editor.contextManager = null;
        this.editor.remove();

        Analytics.sendEvent("editor.instance.destroy");
        Analytics.sendEvent("bundled.editor.instance.destroy");
    };

    /**
     * Sets the content into editor
     * @param content Content to set
     * @param raw
     * @param silent Do not trigger on change listeners
     */
    EditorInstance.prototype.setContent = function (content, raw, silent) {
        var options = {};
        if (raw) {
            options = {format: 'raw'};
        }
        options.no_events = silent;

        this.editor.setContent(content, options);

        this.lastContent = this.getContent();

        if (!silent) {
            this.trigger("content");
        }
    };

    EditorInstance.prototype.replaceSelection = function (content) {
        this.editor.selection.setContent(content);

        this.trigger("content");
    };

    EditorInstance.prototype.selectAll = function () {
        this.editor.selection.select(this.editor.getBody(), true);
    };

    /**
     * Gets the content from editor
     * @param raw
     */
    EditorInstance.prototype.getContent = function (raw) {
        var options = {};
        if (raw) {
            options = {format: 'raw'};
        }
        return this.editor.getContent(options);
    };

    EditorInstance.prototype._onChange = function() {
        if (this.editor.destroyed) {
            return;
        }

        var newContent = this.getContent();
        var changed = newContent !== this.lastContent;
        this.lastContent = newContent;

        if (!this.hidden && changed) {
            this.trigger("content", this.getContent());
        }
    };

    EditorInstance.prototype.getSelectedContent = function (plainText) {
        if (plainText) {
            return this.editor.selection.getContent({format: 'text'});
        } else {
            return this.editor.selection.getContent();
        }
    };

    EditorInstance.prototype.addShortcut = function (shortcut, callback) {
        this.editor.addShortcut(shortcut, '', callback);
    };

    EditorInstance.prototype.removeShortcut = function (shortcut) {
        this.editor.shortcuts.remove(shortcut);
    };

    EditorInstance.prototype.addOperationOverride = function (operationName, fun) {
        this._assertOperationIsSupported(operationName);
        this.operationOverride[operationName] = fun;
    };

    EditorInstance.prototype.removeOperationOverride = function (operationName) {
        this._assertOperationIsSupported(operationName);
        delete this.operationOverrideoperationOverride[operationName];
    };

    EditorInstance.prototype.hide = function() {
        this.hidden = true;
        this.editor.hide();
    };

    EditorInstance.prototype.show = function() {
        this.editor.show();
        delete this.hidden;
    };

    EditorInstance.prototype.isVisible = function() {
        return !this.editor.isHidden();
    };

    EditorInstance.prototype.switchMode = function (showSource) {
        this.trigger("switchMode", showSource);
    };

    EditorInstance.prototype.enable = function () {
        this.editor.setProgressState(false);
    };

    EditorInstance.prototype.disable = function () {
        this.editor.setProgressState(true);
    };

    return EditorInstance;
});