define("jira/editor/converter/handlers/text-handler", [
    "jira/editor/converter/util/strings",
    "jira/editor/converter/util/rte-strings",
    "jira/editor/converter/handlers/node-handler",
    "jira/editor/context-detector"
], function (
    Strings,
    RteStrings,
    NodeHandler,
    ContextDetector
) {
    "use strict";

    var TextHandler = NodeHandler.extend({
        init: function TextHandler(node) {
            this.node = node;
            this.preformatted = this.node.preformatted;
        },

        processText: function () {
            var text = this.node.textContent || '';
            if (this.node.preformatted) {
                // '\n' text and contents of the preformatted tags
                return text;
            }
            if (!RteStrings.isWhitespace(text)) {
                // replace all \n before and after text to ''
                text = this.trimNewlines(text);
            }

            // RTE-6, RTE-117: replace text entries of open curly brace '{' with encoded '\{'
            var resultText = Strings.replace(text, '{', '\\{');

            // Escape `|` inside tables
            if (ContextDetector.detectTable(this.node)) {
                resultText = Strings.replace(resultText, '|', '\\|');
            }

            return resultText;
        },

        children: function () {
            return [];
        },

        trimNewlines: function(str) {
            var i;
            for (i = 0; i < str.length; i++) {
                if (!RteStrings.isWhitespace(str[i])) {
                    break;
                }
            }
            var j;
            for (j = str.length - 1; j >= 0; j--) {
                if (!RteStrings.isWhitespace(str[j])) {
                    break;
                }
            }
            var beforeText = str.substring(0, i);
            var text = str.substring(i, j + 1);
            var afterText = str.substring(j + 1);
            // plain trim can't be used, as it cuts the space ' ' indentation we use for list items
            return Strings.replace(beforeText, '\n', '') + text + Strings.replace(afterText, '\n', '');
        }
    });

    /**
     * Factory method to create a text handler from a plain text
     * @param {String | TextHandler}
     * @returns {TextHandler}
     */
    TextHandler.createTextHandler = function (input) {
        var text = input;

        if (input instanceof Node) {
            text = input.textContent;
        }

        return new TextHandler({
            textContent: text || ''
        });
    };

    /**
     * Factory method to create a text handler from a plain pre-formatted text
     * IMPORTANT! This method will not encode special characters, i.e. '{' will remain '{'
     * Use case - to build markup tags to ensure they are not encoded
     *
     * @param {String | TextHandler}
     * @returns {TextHandler}
     */
    TextHandler.createPreformattedTextHandler = function (input) {
        var text = input;

        if (input instanceof Node) {
            text = input.textContent;
        }

        return new TextHandler({
            textContent: text || '',
            preformatted: true
        });
    };

    TextHandler.newlineHandler = function() {
        return new TextHandler({
            textContent: '\n',
            preformatted: true
        });
    };

    return TextHandler;
});