AJS.test.require(['com.atlassian.jira.plugins.jira-editor-plugin:converter'], function () {
    "use strict";

    var htmlConverter = require('jira/editor/converter');

    module('Text effect handlers', {
        setup: function () {
            this.sandbox = sinon.sandbox.create();

            this.supportedTags = [
                ["b", "*"],
                ["strong", "*"],
                ["em", "_"],
                ["i", "_"],
                ["del", "-"],
                ["sub", "~"],
                ["sup", "^"],
                ["ins", "+"],
                ["u", "+"]
            ];
        },

        teardown: function () {
            this.sandbox.restore();
        }
    });

    test('Should extract left spaces', function () {
        this.supportedTags.forEach(function(entry) {
            var tag = entry[0];
            var wikiLeft = (typeof entry[1] === "string") ? entry[1] : entry[1].left;
            var wikiRight = (typeof entry[1] === "string") ? entry[1] : entry[1].right;
            assertConversion(
                'lorem<' + tag + '> Some text</' + tag + '>',
                'lorem ' + wikiLeft + 'Some text' + wikiRight,
                'text and <' + tag + '> with one space'
            );
            assertConversion(
                '<' + tag + '>\u00A0Some text</' + tag + '>',
                '\u00A0' + wikiLeft + 'Some text' + wikiRight,
                '<' + tag + '> one space'
            );
            assertConversion(
                '<' + tag + '>&nbsp; Some text</' + tag + '>',
                '\u00A0 ' + wikiLeft + 'Some text' + wikiRight,
                '<' + tag + '> two spaces'
            );
            assertConversion(
                '<' + tag + '>&nbsp; &nbsp;Some text</' + tag + '>',
                '\u00A0 \u00A0' + wikiLeft + 'Some text' + wikiRight,
                '<' + tag + '> three spaces'
            );
        });
    });

    test('Should extract left spaces - nested elements', function () {
        this.supportedTags.forEach(function(entry) {
            var tag = entry[0];
            var wikiLeft = (typeof entry[1] === "string") ? entry[1] : entry[1].left;
            var wikiRight = (typeof entry[1] === "string") ? entry[1] : entry[1].right;
            assertConversion(
                '<' + tag + '><span>\u00A0Some text</span></' + tag + '>',
                '\u00A0' + wikiLeft + 'Some text' + wikiRight,
                '<' + tag + '> a nested span'
            );
            assertConversion(
                '<' + tag + '><span><span>\u00A0Some text</span></span></' + tag + '>',
                '\u00A0' + wikiLeft + 'Some text' + wikiRight,
                '<' + tag + '> two nested spans'
            );

            assertConversion(
                '<' + tag + '><b>\u00A0Some text</b></' + tag + '>',
                '\u00A0' + wikiLeft + '*Some text*' + wikiRight,
                '<' + tag + '> with nested <b>'
            );
            assertConversion(
                '<' + tag + '><b><em>\u00A0Some text</em></b></' + tag + '>',
                '\u00A0' + wikiLeft + '*_Some text_*' + wikiRight,
                '<' + tag + '> with nested <b> with nested <em>'
            );

            assertConversion(
                '<' + tag + '><b><em>&nbsp; &nbsp;Some text</em></b></' + tag + '>',
                '\u00A0 \u00A0' + wikiLeft + '*_Some text_*' + wikiRight,
                '<' + tag + '> three spaces with nested <b> with nested <em>'
            );
        });
    });

    // desired behaviour is commented because it's not supported now, fixing those is a good starting point for improving this feature
    // Those tests show side effects of the actual behaviour, ie. extra space might be created.
    test('Should extract left spaces - mischievously nested elements', function () {
        this.supportedTags.forEach(function(entry) {
            var tag = entry[0];
            var wikiLeft = (typeof entry[1] === "string") ? entry[1] : entry[1].left;
            var wikiRight = (typeof entry[1] === "string") ? entry[1] : entry[1].right;
            assertConversion(
                '<' + tag + '><span><span>\u00A0<span> Some text</span></span></span></' + tag + '>',
                '\u00A0 ' + wikiLeft + ' Some text' + wikiRight,
                '<' + tag + '> nested spans with extra space'
            );
            // ideally should be:
            // assertConversion(
            //     '<' + tag + '><span><span>\u00A0<span> Some text</span></span></span></' + tag + '>',
            //     '\u00A0 ' + wikiLeft + 'Some text' + wikiRight,
            //     '<' + tag + '> nested spans with extra space'
            // );

            assertConversion(
                '<' + tag + '><b><em>\u00A0<ins> Some</ins> text</em></b></' + tag + '>',
                '\u00A0 ' + wikiLeft + '*_ +Some+ text_*' + wikiRight,
                '<' + tag + '> with nested <b>, <em> and <ins>'
            );
            // ideally should be:
            // assertConversion(
            //     '<' + tag + '><b><em>\u00A0<ins> Some</ins> text</em></b></' + tag + '>',
            //     '\u00A0 ' + wikiLeft + '*_+Some+ text_*' + wikiRight,
            //     '<' + tag + '> with nested <b>, <em> and <ins>'
            // );
        });
    });

    test('Should extract right spaces', function () {
        this.supportedTags.forEach(function(entry) {
            var tag = entry[0];
            var wikiLeft = (typeof entry[1] === "string") ? entry[1] : entry[1].left;
            var wikiRight = (typeof entry[1] === "string") ? entry[1] : entry[1].right;
            assertConversion(
                '<' + tag + '>Some text </' + tag + '>lorem',
                wikiLeft + 'Some text' + wikiRight + ' lorem',
                'text and <' + tag + '> with one space'
            );
            assertConversion(
                '<' + tag + '>Some text\u00A0</' + tag + '>',
                wikiLeft + 'Some text' + wikiRight + '\u00A0',
                '<' + tag + '> one space'
            );
            assertConversion(
                '<' + tag + '>Some text &nbsp;</' + tag + '>',
                wikiLeft + 'Some text' + wikiRight + ' \u00A0',
                '<' + tag + '> two spaces'
            );
            assertConversion(
                '<' + tag + '>Some text&nbsp; &nbsp;</' + tag + '>',
                wikiLeft + 'Some text' + wikiRight + '\u00A0 \u00A0',
                '<' + tag + '> three spaces'
            );
        });
    });

    test('Should extract right spaces - nested elements', function () {
        this.supportedTags.forEach(function(entry) {
            var tag = entry[0];
            var wikiLeft = (typeof entry[1] === "string") ? entry[1] : entry[1].left;
            var wikiRight = (typeof entry[1] === "string") ? entry[1] : entry[1].right;
            assertConversion(
                '<' + tag + '><span>Some text\u00A0</span></' + tag + '>',
                wikiLeft + 'Some text' + wikiRight + '\u00A0',
                '<' + tag + '> a nested span'
            );
            assertConversion(
                '<' + tag + '><span><span>Some text\u00A0</span></span></' + tag + '>',
                wikiLeft + 'Some text' + wikiRight + '\u00A0',
                '<' + tag + '> two nested spans'
            );

            assertConversion(
                '<' + tag + '><b>Some text\u00A0</b></' + tag + '>',
                wikiLeft + '*Some text*' + wikiRight + '\u00A0',
                '<' + tag + '> with nested <b>'
            );
            assertConversion(
                '<' + tag + '><b><em>Some text\u00A0</em></b></' + tag + '>',
                wikiLeft + '*_Some text_*' + wikiRight + '\u00A0',
                '<' + tag + '> with nested <b> with nested <em>'
            );

            assertConversion(
                '<' + tag + '><b><em>Some text&nbsp; &nbsp;</em></b></' + tag + '>',
                wikiLeft + '*_Some text_*' + wikiRight + '\u00A0 \u00A0',
                '<' + tag + '> three spaces with nested <b> with nested <em>'
            );
        });
    });

    test('Should extract left and right spaces', function () {
        this.supportedTags.forEach(function(entry) {
            var tag = entry[0];
            var wikiLeft = (typeof entry[1] === "string") ? entry[1] : entry[1].left;
            var wikiRight = (typeof entry[1] === "string") ? entry[1] : entry[1].right;
            assertConversion(
                'lor<' + tag + '> Some text </' + tag + '>em',
                'lor ' + wikiLeft + 'Some text' + wikiRight + ' em',
                'text and <' + tag + '> with one space each side'
            );
            assertConversion(
                '<' + tag + '>\u00A0Some text\u00A0</' + tag + '>',
                '\u00A0' + wikiLeft + 'Some text' + wikiRight + '\u00A0',
                '<' + tag + '> one space each side'
            );
            assertConversion(
                '<' + tag + '>&nbsp; Some text &nbsp;</' + tag + '>',
                '\u00A0 ' + wikiLeft + 'Some text' + wikiRight + ' \u00A0',
                '<' + tag + '> two spaces each side'
            );
            assertConversion(
                '<' + tag + '>&nbsp; &nbsp;Some text&nbsp; &nbsp;</' + tag + '>',
                '\u00A0 \u00A0' + wikiLeft + 'Some text' + wikiRight + '\u00A0 \u00A0',
                '<' + tag + '> three spaces each side'
            );

            assertConversion(
                '<' + tag + '>&nbsp; Some text &nbsp; &nbsp;</' + tag + '>',
                '\u00A0 ' + wikiLeft + 'Some text' + wikiRight + ' \u00A0 \u00A0',
                '<' + tag + '> two left spaces and four right on the right'
            );
        });
    });

    test('Should extract left and right spaces - nested elements', function () {
        this.supportedTags.forEach(function(entry) {
            var tag = entry[0];
            var wikiLeft = (typeof entry[1] === "string") ? entry[1] : entry[1].left;
            var wikiRight = (typeof entry[1] === "string") ? entry[1] : entry[1].right;

            assertConversion(
                '<' + tag + '><b><em>&nbsp; Some text&nbsp; &nbsp;</em></b></' + tag + '>',
                '\u00A0 ' + wikiLeft + '*_Some text_*' + wikiRight + '\u00A0 \u00A0',
                '<' + tag + '> three spaces with nested <b> with nested <em>'
            );
            assertConversion(
                '<' + tag + '>&nbsp; Some<em> text&nbsp;</em> &nbsp;</' + tag + '>',
                '\u00A0 ' + wikiLeft + 'Some _text_\u00A0' + wikiRight + ' \u00A0',
                '<' + tag + '> nested <em> with spaces'
            );
        });
    });

    // desired behaviour is commented because it's not supported now, fixing those is a good starting point for improving this feature
    // Those tests show side effects of the actual behaviour, ie. extra space might be created.
    test('Should extract left and right spaces - mischievously nested elements', function () {
        this.supportedTags.forEach(function(entry) {
            var tag = entry[0];
            var wikiLeft = (typeof entry[1] === "string") ? entry[1] : entry[1].left;
            var wikiRight = (typeof entry[1] === "string") ? entry[1] : entry[1].right;
            assertConversion(
                '<' + tag + '><b>\u00A0 <em>\u00A0</em><ins> Some</ins>\u00A0text </b>\u00A0</' + tag + '>',
                '\u00A0 \u00A0 ' + wikiLeft + '*\u00A0__\u00A0 +Some+\u00A0text* ' + wikiRight + '\u00A0',
                '<' + tag + '> with nested <b>, <em> and <ins>'
            );
            // ideally should be:
            // assertConversion(
            //     '<' + tag + '><b>\u00A0 <em>\u00A0</em><ins> Some</ins>\u00A0text </b>\u00A0</' + tag + '>',
            //     '\u00A0 \u00A0 ' + wikiLeft + '*+Some+\u00A0text*' + wikiRight + ' \u00A0',
            //     '<' + tag + '> with nested <b>, <em> and <ins>'
            // );

            assertConversion(
                '<' + tag + '><b>\u00A0 <em>\u00A0</em>lorem<ins> Some</ins>\u00A0text </b>\u00A0</' + tag + '>',
                '\u00A0 \u00A0' + wikiLeft + '*\u00A0__\u00A0lorem +Some+\u00A0text* ' + wikiRight + '\u00A0',
                '<' + tag + '> with nested <b>, <em> and <ins>'
            );
            // ideally should be:
            // assertConversion(
            //     '<' + tag + '><b>\u00A0 <em>\u00A0</em>lorem<ins> Some</ins>\u00A0text </b>\u00A0</' + tag + '>',
            //     '\u00A0 \u00A0' + wikiLeft + '*lorem +Some+\u00A0text*' + wikiRight + ' \u00A0',
            //     '<' + tag + '> with nested <b>, <em> and <ins>'
            // );
        });
    });

    var assertConversion = function (html, markup, testName) {
        htmlConverter.convert(html).then(function (result) {
            equal(result, markup, testName);
        }).fail(function (e) {
            throw e;
        });
    };
});