define('jira/editor/registry/entry', [
    'underscore',
    'jquery',
    'jira/editor/constants'
], function (
    _,
    $,
    Constants
) {

    var Modes = Constants.Modes;

    /**
     * Editor Registry entry that keeps current mode value and references to textarea, toolbar, rich editor instance.
     *
     * @param id editor's id used to keep it in the registry
     * @param textArea text mode textarea element
     * @param editorRegistry Registry keeping this entry
     * @class
     */
    function Entry(id, textArea, editorRegistry) {
        var visualEditorDeferred = $.Deferred();
        this._rteInstance = visualEditorDeferred;
        this._editorRegistry = editorRegistry;

        /** @member {string} id Editor's id */
        this.id = id;
        /** @member {string} currentMode Editor's current mode value (@see 'jira/editor/constants') */
        this.currentMode = undefined;
        /** @member {object} textArea Editor's textarea element used in Text mode */
        this.textArea = textArea;
        /** @member {object} toolbar Editor's toolbar element */
        this.toolbar = this._findToolbar(textArea);
        /** @member {object} rteInstance jQuery Promise backed by rich text editor instance */
        this.rteInstance = visualEditorDeferred.promise();
    }

    _.extend(Entry.prototype, {
        /**
         * Checks if editor described by the current entry is in the Visual mode
         * @returns {boolean}
         */
        isVisualMode: function () {
            return this.currentMode === Modes.VISUAL;
        },

        /**
         * Checks if editor described by the current entry is in the Text mode
         * @returns {boolean}
         */
        isTextMode: function () {
            return this.currentMode === Modes.TEXT;
        },

        /**
         * Executes callback on the current entry only if the editor is in the Text mode
         * @returns {object} this
         */
        applyIfTextMode: function (callback) {
            return this._applyIfMode(Modes.TEXT, callback);
        },

        /**
         * Executes callback on the current entry only if the editor is in the Text mode
         * @returns {object} this
         */
        applyIfVisualMode: function (callback) {
            return this._applyIfMode(Modes.VISUAL, callback);
        },

        /**
         * Binds callback to the `switch` event related to the editor described by this entry
         * @param {Function} callback
         * @returns {Entry}
         */
        onSwitch: function (callback) {
            this._editorRegistry.on('switch::' + this.id, callback);
            return this;
        },

        /**
         * Binds callback to the `unregister` event related only to the editor described by this entry
         * @param {Function} callback
         * @returns {Entry}
         */
        onUnregister: function (callback) {
            this._editorRegistry.once('unregister::' + this.id, callback);
            return this;
        },

        /**
         * Binds callback to the `switch` (to Text mode) event related only to the editor described by this entry
         * @param {Function} callback
         * @returns {Entry}
         */
        onSwitchToText: function (callback) {
            this._editorRegistry.on('switch:' + Modes.TEXT + ':' + this.id, callback);
            return this;
        },

        /**
         * Binds callback to the `switch` (to Visual mode) event related only to the editor described by this entry
         * @param {Function} callback
         * @returns {Entry}
         */
        onSwitchToVisual: function (callback) {
            this._editorRegistry.on('switch:' + Modes.VISUAL + ':' + this.id, callback);
            return this;
        },

        _applyIfMode: function (mode, callback) {
            if (this.currentMode === mode) {
                callback(this);
            }
            return this;
        },

        _findToolbar: function (textArea) {
            var wikiEditContent = textArea.parentNode;
            var wikiEdit = wikiEditContent.parentNode;
            return wikiEdit.querySelector('div.wiki-edit-toolbar');
        }
    });

    return Entry;
});