define('jira/auditing/records', [
    'jira/ajs/ajax/smart-ajax',
    'jira/auditing/record',
    'backbone',
    'jquery',
    'underscore'
], function(
    SmartAjax,
    Record,
    Backbone,
    $,
    _
) {
    var contextPath = AJS.contextPath;
    var Paginator = Backbone.Paginator;
    return Paginator.requestPager.extend({
        model: Record,

        // going to remember the first record we saw when loading the page so pagination doesn't change when new events are added
        // later we're going to add a pop up letting the user know it's time to refresh the page because new records were added
        topId: undefined,
        filter: "",
        //this goes as POST params, this is a workaround to make them empty instead of "null" or "undefined"
        timeUnit: "",
        timeUnitValue: "",
        fromDate: "",
        toDate: "",
        refreshTopId: undefined,

        // used to block turning pending state on in case of initial request (the one that is made for sync collection
        // if url contains hash)
        isInitialRequest: false,

        // used to not cause inability to navigate back after user visited non-existing page and collection was updated
        // to the latest page available; see this.ensurePageAvailable()
        _ensuringPageAvailable: false,

        _lastRefreshRange: undefined,
        _pullingInterval: undefined,

        // these two params are used to determine if there should be created new history item or current url should be replaced;
        // this fixes problem with inability to navigate back after empty result set has been refreshed
        isRefreshing: false,
        topIdBeforeRefresh: undefined,

        initializePulling: function(){
            this._enablePulling();

            // don't check for newer records if tab is inactive
            $(document).on("hide", _.bind(this._disablePulling, this));
            $(document).on("show", _.bind(this._enablePulling, this));

            this.on("sync", this._enablePulling, this);
        },

        // asks server about highest id of records for current criteria, returns promise
        _checkForNewerRecords: function(){
            var deferred = $.Deferred();
            var params = _.extend(_.omit(this.getCurrentParams(), ["currentPage", "topId"]), {count: 1, page: 1});

            SmartAjax.makeRequest({
                cache: false,
                data: $.param(params),
                dataType: "json",
                type: "GET",
                url: contextPath() + "/rest/jira-auditing-plugin/1/view",
                success: _.bind(function(response){
                    this.setRefreshTopId(response.refreshMaxId);
                    deferred.resolve();
                }, this),
                error: function(){
                    deferred.reject();
                }
            });

            return deferred.promise();
        },

        // ask server if there are newer records, in such case stop pulling and trigger event
        _pullingFunc: function(){
            this._checkForNewerRecords().done(_.bind(function(){
                if (this.isRefreshable()){
                    this._disablePulling();
                    this.trigger("auditing:update-records-possible");
                }
            }, this));
        },

        // helper function, enables pulling if disabled
        _enablePulling: function(){
            if (!this._pullingInterval && !this.isRefreshable()){
                this._pullingInterval = setInterval(_.bind(this._pullingFunc, this), 30000);
                this._pullingFunc();
            }
        },

        _disablePulling: function(){
            clearInterval(this._pullingInterval);
            this._pullingInterval = undefined;
        },

        getCurrentParams: function(){
            return {
                filter: this.filter,
                fromDate: this.fromDate,
                currentPage: this.currentPage,
                timeUnit: this.timeUnit,
                timeUnitValue: this.timeUnitValue,
                toDate: this.toDate,
                topId: this.topId
            };
        },

        // encoded params are used to save browser history position
        // and as data sent to sync collection
        getCurrentParamsEncoded: function(){
            var params = this.getCurrentParams();

            $.each(params, function(key, value){
                if (typeof value === "string"){
                    params[key] = encodeURIComponent(value);
                }
            });

            return params;
        },

        setCurrentParams: function(params){
            for (var i in params){
                this[i] = params[i];
            }

            this.currentPage = +this.currentPage;
        },

        hasExactParams: function(params){
            for (var i in params){
                if (this[i] != params[i]) return false;
            }

            return true;
        },

        isRefreshable: function(){
            return !!(this.refreshTopId && ((this.topId && +this.refreshTopId > +this.topId) || typeof this.topId === "undefined"));
        },

        setRefreshTopId: function(refreshTopId){
            this.refreshTopId = refreshTopId;
        },

        getLastRefreshRange: function(){
            return this._lastRefreshRange;
        },

        ensurePageAvailable: function() {
            if(this.size() === 0 && this.totalRecords > 0) {
                this._ensuringPageAvailable = true;
                this.goTo(this.info().totalPages);
            }
        },

        comparator: function descendingIdComparator (a, b){
            return parseInt(b.get("id")) - parseInt(a.get("id"))
        },

        paginator_core: {
            type: 'GET',
            dataType: 'json',
            url: contextPath() + "/rest/jira-auditing-plugin/1/view"
        },

        paginator_ui: {
            firstPage: 1,
            perPage: null,
            pagesInRange: 2
        },

        server_api: {
            'count': function() { return this.perPage },
            'maxId': function() {
                if (this.topId !== undefined) {
                    return this.topId;
                } else {
                    return "";
                }
            },
            'page': function() { return this.currentPage },
            'filter': function() { return this.filter },
            'timeUnit' : function() { return this.timeUnit },
            'timeUnitValue' : function() { return this.timeUnitValue },
            'fromDate' : function() { return this.fromDate },
            'toDate' : function() { return this.toDate }
        },


        search : function() {
            this.topId = undefined;
            this.goTo(this.information.firstPage);
        },

        refresh: function(){
            this.isRefreshing = true;
            this.topIdBeforeRefresh = this.topId;

            this.topId = undefined;
            this.goTo(this.information.firstPage);
        },

        parse : function(response) {
            this.totalRecords = response.count;
            this.empty = response.empty;
            this.setRefreshTopId(response.refreshMaxId);

            if (this.topId === undefined && response.records.length > 0) {
                this.topId = response.records[0].id;
            }

            this._lastRefreshRange = this.isRefreshing && this.topId ? {
                from: this.topIdBeforeRefresh ? +this.topIdBeforeRefresh + 1: 0,
                to: +this.topId
            } : undefined;

            return response.records;
        },

        xhr : null,
        requestCount: 0,

        sync: function(method, model, options) {
            var requestNum = ++this.requestCount;
            if(this.xhr) {
                //since we are interested in results of only the latest query we can abort pending query
                this.xhr.abort();
            }
            options.success = this.getCallback(options.success, requestNum);
            options.error = this.getCallback(options.error, requestNum);
            options.data = this.getCurrentParamsEncoded();

            this.xhr = Paginator.requestPager.prototype.sync.call(this, method, model, options);
        },

        //it wraps callback in if statement that checks whether it is response to latest query if not we can
        //discard this result
        getCallback : function(callback, requestNum) {
            var self = this;
            return function(model, resp, options) {
                if(self.requestCount === requestNum) {
                    self.xhr = null;
                    callback(model, resp, options)
                }
            };
        }

    });
});

AJS.namespace('JIRA.Auditing.Records', null, require('jira/auditing/records'));

