define('jira/auditing/records-view', [
    'backbone',
    'jquery',
    'underscore',
    'jira/analytics',
    'jira/util/strings'
], function(
    Backbone,
    $,
    _,
    Analytics,
    Strings
) {
    return Backbone.View.extend({
        el: ".auditing-table",
        renderEmptyTable: false,

        events: {
            'click .record-row' : 'showDetails'
        },

        templates: {
            records: JIRA.Templates.Auditing.records,
            details: JIRA.Templates.Auditing.recordDetails,
            associatedItems: JIRA.Templates.Auditing.associatedItems
        },

        initialize: function(options) {
            this.renderEmptyTable = options.emptyTable;
            this.render();
        },

        render: function(){
            var records = this.renderEmptyTable ? {} : this.collection.toJSON();
            var range = this.collection.getLastRefreshRange();

            this.$el.html(this.templates.records({records: records}));

            // mark the records added by refresh
            if (!range) return;

            var $rows = this.$(".record-row");
            $rows.filter(function(){
                var id = +$(this).data("id");

                return id >= range.from && id <= range.to;
            })
                    .addClass("auditing-record-added auditing-record-added-animate")
                    .delay(500)
                    .animate(
                    {backgroundColor: "#ffffff"},
                    {duration: 1500, complete: function(){
                        $rows
                                .removeClass("auditing-record-added-animate")
                                .removeAttr("style");
                    }}
            );
        },

        showDetails: function(event) {
            if ($(event.target).prop("tagName") === "A"){
                return;
            }

            var target = $(event.currentTarget);
            var detailsWrapper = target.next();
            var actionLink = target.find(".show-details");

            var recordId = parseInt(target.attr("data-id"), 10);
            var record = this.collection.get(recordId).toJSON();

            var associatedItems,
                    detailContainer;

            if (detailsWrapper.hasClass("hidden")) {
                detailContainer = detailsWrapper.find(".auditing-record-details");
                detailContainer.html(this.templates.details({ record: record }));

                //we need to get associated-items after they appear in DOM
                associatedItems = detailContainer.find(".associated-items");
                associatedItems.html(this.templates.associatedItems({ mappedItems: this._mapAssociatedItems(record) }));

                target.addClass('active');
                detailsWrapper.removeClass("hidden");
                actionLink.text(AJS.I18n.getText("jap.view.see.less"));

                this._sendToggleAnalytics(record, true);
            } else {
                target.removeClass('active');
                detailsWrapper.addClass("hidden");
                actionLink.text(AJS.I18n.getText("jap.view.see.more"));

                this._sendToggleAnalytics(record, false);
            }
        },

        _sendToggleAnalytics: function(record, show) {
            Analytics.send({
                name: "administration.auditing.details.toggle",
                properties: {
                    categoryId: record.categoryId,
                    categoryIdHash: Strings.hashCode(record.categoryId),
                    id: record.id,
                    show: show
                }
            });
        },

        _mapAssociatedItems: function(record) {

            var items = {
                user: [],
                group: [],
                scheme: [],
                project: [],

                showDefaultMessage: true
            };

            _.each(record.associatedItems, function(item) {
                if (item.objectType && items[item.objectType.toLowerCase()]) {
                    items[item.objectType.toLowerCase()].push(item);
                    items.showDefaultMessage = false;
                }
            });

            return items;
        }
    });
});

AJS.namespace('JIRA.Auditing.RecordsView', null, require('jira/auditing/records-view'));
