define('jira/auditing/records-controller', [
    'jira/auditing/records-errors-view',
    'jira/auditing/records-pages-view',
    'jira/auditing/search-view',
    'jira/auditing/time-filter-controller',
    'jira/auditing/router',
    'jira/auditing/router-helper',
    'jira/auditing/records-analytics',
    'jira/auditing/records',
    'jira/auditing/settings-dialog-factory',
    'jira/dialog/form-dialog',
    'backbone',
    'jquery'
], function(
    RecordsErrorsView,
    RecordsPagesView,
    SearchView,
    TimeFilterController,
    Router,
    RouterHelper,
    RecordsAnalytics,
    Records,
    SettingsDialogFactory,
    FormDialog,
    Backbone,
    $
) {
    return function() {
        var triggerEvent = AJS.trigger;
        var contextPath = AJS.contextPath;

        return {
            analytics: undefined,
            collection: null,
            rootPath: contextPath() + "/auditing/view",
            settingsDialog: null,
            timeFilter: null,

            init: function() {
                if (!this.ensureCorrectPath()) return;

                new FormDialog({
                    id: "export-data-dialog",
                    trigger: "#auditing-export-data"
                });

                $("#auditing-export-data").on('click', function () {
                    triggerEvent('analyticsEvent', {name: 'administration.auditing.view.export.clicked', data: {}});
                });

                var $auditing = $(".auditing-view-records");
                this.collection = new Records();
                this.collection.on("sync", this.collection.ensurePageAvailable, this.collection);
                this.collection.paginator_ui.perPage = $auditing.data("perpage");
                this.collection.reset(this.collection.parse($auditing.data("records")));
                this.collection.bootstrap();

                this.collection.setCurrentParams($auditing.data("current-params"));
                this.collection.initializePulling();

                new RecordsErrorsView({ collection : this.collection });
                var mainView = new RecordsPagesView({ collection : this.collection });
                new SearchView({ collection : this.collection });

                this.timeFilter = TimeFilterController();
                this.timeFilter.init({ collection : this.collection });

                //initialize listeners
                this.collection.on("auditing:refresh-records", this.refreshRecords, this);
                this.collection.on("auditing:update-records", this.updateRecords, this);
                this.collection.on("sync", this.saveHistoryState, this);

                //do not block leaving auditing page
                window.onbeforeunload = function() {};

                this.analytics = new RecordsAnalytics();
                this.analytics.init(this.collection, this.timeFilter.model);

                this.settingsDialog = SettingsDialogFactory(this.collection);

                this.router = new Router({
                    collection: this.collection,
                    mainView: mainView
                });
                Backbone.history.start({
                    pushState: true,
                    root: this.rootPath,
                    silent: false
                });
            },

            // used to refresh records once filters changed
            refreshRecords: function (searchParams) {
                this.collection.filter = searchParams.filter || this.collection.filter;
                this.collection.timeUnit = searchParams.timeUnit || this.timeFilter.model.unit;
                this.collection.timeUnitValue = searchParams.timeUnitValue || this.timeFilter.model.unitValue;
                this.collection.fromDate = searchParams.fromDate || this.timeFilter.model.fromDate;
                this.collection.toDate = searchParams.toDate || this.timeFilter.model.toDate;
                this.analytics.searchStarted();
                this.collection.search();
            },

            // used to update collection with newer records that exists for current parameters - this is what user
            // literally see as a refresh
            updateRecords: function(){
                this.collection.refresh();
            },

            saveHistoryState: function(){
                this.router.saveHistoryState();
            },

            // auditing view page should always be served from /auditing/view, to make client history / server rendering
            // fully working; however, there's no easy way for a webwork action to check if it's being served under it's name,
            // like ViewRecords.jspa, or under rewritten url. I believe this is somehow done in KA, so later it might
            // prove valuable to see how it's done there and implement this also in auditing (TODO candidate)
            ensureCorrectPath: function(){
                var isAuditingPath = location.pathname.indexOf("auditing/view") !== -1;

                if (!isAuditingPath){
                    var params = RouterHelper.parseQueryParams();
                    location.replace(this.rootPath + RouterHelper.generateUrl(params));
                }

                return isAuditingPath;
            }
        }
    };
});

AJS.namespace('JIRA.Auditing.RecordsController', null, require('jira/auditing/records-controller'));