define('jira/auditing/time-filter-view',[
    'brace',
    'backbone',
    'jquery',
    'underscore'
], function (
    Brace,
    Backbone,
    $,
    _
) {
    var InlineDialog = AJS.InlineDialog;

    return Brace.View.extend({

        timeUnitsWords : {
            min: AJS.I18n.getText("core.dateutils.minutes"),
            hrs: AJS.I18n.getText("core.dateutils.hours"),
            day: AJS.I18n.getText("core.dateutils.days"),
            week: AJS.I18n.getText("core.dateutils.weeks")},

        template: JIRA.Templates.Auditing.Filters.dateFilter,

        initialize: function(options) {
            var fromDatePicker,
                    toDatePicker;

            var self = this;
            var allowDialogHide = false;
            var $timeFilter = $("#time-filter");

            this.consts = options.consts;

            var _hide = function() {
                $timeFilter.focus();
                _.defer(function() {
                    allowDialogHide = true;
                    popup.hide();
                    allowDialogHide = false;
                });
            };

            function _initializeDialog(content)
            {
                content.html(self.template());

                $(".time-filter-content .update").on("click", function (event)
                {
                    event.preventDefault();
                    if (self.validateAndUpdateModel())
                    {
                        _hide();
                    }
                });

                $(".time-filter-content .cancel").on("click", function (event)
                {
                    event.preventDefault();
                    _hide();
                });

                $('input[name=auditDateFilter]', '#time-filter-form').on('change', function ()
                {
                    if ($(this).val() === self.consts.WITHIN_THE_LAST)
                    {
                        $(".time-unit-value").focus();
                    }
                });

                fromDatePicker = $("#from-date-picker").datePicker({
                    overrideBrowserDefault: true
                });

                toDatePicker = $("#to-date-picker").datePicker({
                    overrideBrowserDefault: true
                });

                self._registerEventsToSelectRadioOnInputClick();
            }

            var dialogAlreadyInitialized = false;

            var popup = InlineDialog($timeFilter, "time-dialog",
                    function (content, trigger, showPopup) {
                        if(!dialogAlreadyInitialized)
                        {
                            _initializeDialog(content);
                            dialogAlreadyInitialized = true;
                        }
                        self.refreshDialogFromModel();
                        showPopup();
                        return false;
                    },
                    {
                        width: 390,
                        hideDelay: 10e6,
                        preHideCallback: function () {
                            return allowDialogHide;
                        }
                    }
            );

            $timeFilter.on('click', function() {
                if($timeFilter.hasClass("active")) {
                    _hide();
                }
            });

            $(document).on("keydown", function (event) {
                if (event.keyCode === 27) {
                    _hide();
                }
            });
        },

        refreshDialogFromModel: function(){
            var model = this.model;

            //clear all errors before showing dialog
            $(".value-error").html("");

            $("input[value=" + model.type + "]", "#time-filter-form").attr("checked", true);
            $(".time-unit").val(model.unit);
            $(".time-unit-value").val(model.unitValue);

            if (model.fromDate !== "")
            {
                $("#from-date-picker").val(this._formatDate(model.fromDate));
            }
            else
            {
                $("#from-date-picker").val("");
            }

            if (model.toDate !== "")
            {
                $("#to-date-picker").val(this._formatDate(model.toDate));
            }
            else
            {
                $("#to-date-picker").val("");
            }
        },

        render: function() {
            //we should render button here, not in template

            this.updateFilterDescription();
        },

        validateAndUpdateModel: function() {

            var timeUnit,
                    fromDateVal,
                    fromDate,
                    toDateVal,
                    toDate,
                    timeUnitValue,
                    $timeUnitField,
                    $valueError,

                    $filterType = $('input[name=auditDateFilter]:checked','#time-filter-form');
            $valueError = $(".value-error");
            $timeUnitField = $(".time-unit-value");
            $valueError.html("");

            if ($filterType.val() === this.consts.ALL_RECORDS) {
                this.trigger("auditing:update-time-filter-model");
                return true;
            }

            if ($filterType.val() === this.consts.WITHIN_THE_LAST) {
                timeUnitValue = parseInt($timeUnitField.val(), 10);

                if (isNaN(timeUnitValue)) {
                    $valueError.html(AJS.I18n.getText("jap.time.filter.number.error"));
                    return false;
                } else {
                    $timeUnitField.val(timeUnitValue);
                    this.trigger("auditing:update-time-filter-model");
                    return true;
                }
            }

            if ($filterType.val() === this.consts.BETWEEN) {

                fromDateVal = $('#from-date-picker').val();
                toDateVal = $('#to-date-picker').val();

                if (!fromDateVal || !toDateVal) {
                    $valueError.html(AJS.I18n.getText("jap.time.filter.date.empty.error"));
                    return false;
                }

                var splitFromVal = fromDateVal.split("-");
                fromDate = new Date(splitFromVal[0], splitFromVal[1]-1, splitFromVal[2]);

                var splitToVal = toDateVal.split("-");
                toDate = new Date(splitToVal[0], splitToVal[1]-1, splitToVal[2]);

                if (isNaN(fromDate.getTime()) || isNaN(toDate.getTime())) {
                    $valueError.html(AJS.I18n.getText("jap.time.filter.date.validation.error"));
                    return false;
                }

                if (fromDate.getTime() > toDate.getTime()) {
                    $valueError.html(AJS.I18n.getText("jap.time.filter.dates.range.error"));
                    return false;
                }

                this.trigger("auditing:update-time-filter-model");

                return true;
            }

            return false;
        },

        updateFilterDescription: function() {
            var fromDate,
                    toDate;

            var $timeFilter = $("#time-filter");
            var desc;

            if (this.model.type === this.consts.ALL_RECORDS) {
                desc = AJS.I18n.getText("common.words.all");

            } else if (this.model.type === this.consts.WITHIN_THE_LAST) {
                desc =  AJS.I18n.getText("jap.time.filter.within.last", this.model.unitValue, this.timeUnitsWords[this.model.unit]);

            } else if (this.model.type === this.consts.BETWEEN) {
                desc = AJS.I18n.getText("jap.time.filter.between.dates", this._formatDate(this.model.fromDate), this._formatDate(this.model.toDate));
            }

            $timeFilter.html(AJS.I18n.getText("jap.time.filter.button.message",desc));
        },

        _formatDate: function(timestamp){
            var date = new Date(+timestamp);
            var day = date.getDate();
            var month = date.getMonth() + 1;
            var year = date.getFullYear();

            month = (month < 10) ? "0" + month : month;
            day = (day < 10) ? "0" + day : day;

            return year + "-" + month + "-" + day;
        },

        _registerEventsToSelectRadioOnInputClick: function() {
            var self = this;

            $('.time-unit-value').add('.time-unit').on("click", function() {
                $("input[value=" + self.consts.WITHIN_THE_LAST + "]","#time-filter-form").attr("checked",true);
            });

            $('#from-date-picker').add('#to-date-picker').on("click", function() {
                $("input[value=" + self.consts.BETWEEN + "]","#time-filter-form").attr("checked",true);
            });
        }

    });
});

AJS.namespace('JIRA.Auditing.TimeFilterView', null, require('jira/auditing/time-filter-view'));