define("hipchat/feature/roommapping/roommapping-page",
[
    "jquery",
    "backbone",
    "underscore",
    "hipchat/feature/roommapping/mapping",
    "hipchat/feature/roommapping/mapping-view",
    "hipchat/widget/roomselector/roommapping-service",
    "hipchat/widget/roomselector/roomselector-view"
], function (
    $,
    Backbone,
    _,
    Mapping,
    MappingView,
    RoomMappingService,
    RoomSelectorView
) {

    function extractProjectNameAndKey(projectNameAndKey, projectId) {

        var key = "";
        var name = "";

        var match = /^(.*?)(?:\s\((\w+)\))?$/.exec(projectNameAndKey);

        if (match !== null) {
            key = match[2];
            name = match[1];
        }

        return {
            projectKey: key,
            projectId: projectId,
            projectName: name
        };
    }

    function getProjectKey() {
        return $("[name='projectKey']").attr("content");
    }

    var RoomMappingPageView = Backbone.View.extend({

        events: {
            "click #hipchat-project-to-room-add": "add",
            "submit .hipchat-room-mapping-form": "submitForm"
        },

        initialize: function() {
            this.roomServicePromise = RoomMappingService.roomServicePromise(this.$el.data("hipchat-connected"), getProjectKey());

            this.loggedIn = this.$el.data("hipchat-username") !== undefined;
            this.roomSelector = new RoomSelectorView({el: this.$('#hipchat-project-to-room-add-select'), loggedIn: this.loggedIn, maxWidth: "200px"});

            this.roomSelector.on("change", this.enableAddButton, this);
            this.roomSelector.on("clear", this.disableAddButton, this);

            var self = this;
            this.roomServicePromise.done(function(roomService) {
                roomService.onChange(function() {
                    self.roomSelector.clearSelectedRoom();
                    self.disableAddButton();
                });
            });

            this.collection.on("add", this.addMapping, this);
            this.collection.on("remove", this.removeMapping, this);
            this.collection.on("change:edit", this.editMapping, this);
        },

        getProjectNameFromField: function() {
            return this.$("#project-field").val();
        },

        getProjectIdFromField: function() {
            var values = this.$("#project").val();
            if (_.isArray(values)) {
                return values[0];
            }

            if (_.isEmpty(values)) {
                return this.$el.data("project-id");
            }

            return values;
        },

        enableAddButton: function() {
            var selectedRoom = this.roomSelector.getSelectedRoom();
            var selectedProject = this.getProjectNameFromField();
            if (selectedRoom.id !== "" && selectedProject !== "") {
                this.$("#hipchat-project-to-room-add").removeAttr('aria-disabled').removeAttr('disabled');
            }
        },

        disableAddButton: function() {
            this.$('#hipchat-project-to-room-add').attr({
                'aria-disabled': 'true',
                'disabled': ''
            });
        },

        submitForm: function(e) {
            e.preventDefault();
            if (this.$('#hipchat-project-to-room-add').attr("disabled") === undefined) {
                this.add(e);
            }
        },

        add: function(e) {
            e.preventDefault();
            var selectedRoom = this.roomSelector.getSelectedRoom();
            var self = this;

            var spinner = this.$(".hipchat-room-mapping-form .loading").spin();
            this.roomServicePromise.done(function(roomService) {
                if (!selectedRoom.existing) {

                    roomService.createRoom(selectedRoom.roomName).done(function(room) {
                        spinner.spinStop();
                        self.addRoom(room);
                    });

                } else {
                    var room = roomService.getRoomById(selectedRoom.id);
                    if(room) {
                        self.$(".hipchat-room-mapping-list").next().remove();

                        self.addRoom(room).fail(function() {
                            self.$(".hipchat-room-mapping-list").after($(JIRA.Templates.HipChat.Project.RoomMapping.errors({})));
                        }).always(function() {
                            spinner.spinStop();
                        });
                    }
                }

            });
        },

        addRoom: function(room) {
            var deferred = $.Deferred();
            if (room) {
                var project = extractProjectNameAndKey(this.getProjectNameFromField(), this.getProjectIdFromField());

                this.collection.create({
                    roomId: room.roomId,
                    roomName: room.roomName,
                    projectName: project.projectName,
                    projectKey: project.projectKey,
                    projectId: project.projectId,
                    edit: true,
                    config: {
                        "MATCHER:ISSUE_CREATED": true
                    }
                }, {
                    wait: true,
                    success: function(resp) {
                        deferred.resolve(resp);
                    },
                    error: function(err) {
                        deferred.reject(err);
                    }
                });
            } else {
                deferred.reject();
            }

            return deferred;
        },

        addMapping: function(mapping) {
            var mappingView = new MappingView({
                model: mapping
            });

            var $mapping = mappingView.render().$el;

            var mappingIndex = this.collection.indexOf(mapping);
            if (mappingIndex > 0) {
                var previousMapping = this.collection.at(mappingIndex - 1);
                var previousMappingView = this.$("[data-configuration-group-id='" + previousMapping.get("configurationGroupId") + "']");
                previousMappingView.after($mapping);
            } else if (mappingIndex === 0) {
                this.$(".hipchat-room-mapping-list").prepend($mapping);
            } else {
                this.$(".hipchat-room-mapping-list").append($mapping);
            }

            this.$(".hipchat-integration-steps").trigger('mapping-added.integration-steps');

            this.roomSelector.clearSelectedRoom();
            this.disableAddButton();

            this.editMapping(mapping);
        },

        removeMapping: function(mapping) {
        },

        editMapping: function(mapping) {
            if (mapping.get("edit")) {
                this.collection.each(function(model) {
                    if (model !== mapping) {
                        model.set("edit", false);
                    }
                });
            }
        }
    });


    return RoomMappingPageView;
});