define("hipchat/feature/issuepanel/dedicatedroom-view",
[
    "ajs", // only to make karma happy
    "jquery",
    "backbone",
    "hipchat/base",
    "hipchat/feature/issuepanel/dedicatedroom",
    "hipchat/widget/roomselector/roommapping-service",
    "hipchat/widget/roomselector/roomselector-view",
    "hipchat/feature/issuepanel/invitecontributors",
    "hipchat/feature/issuepanel/invitecontributors-view"
], function (
        AJS,
        $,
        Backbone,
        HipChat,
        DedicatedRoom,
        RoomMappingService,
        RoomSelectorView,
        InviteContributors,
        InviteContributorsView
) {
    var DedicatedRoomView = HipChat.View.extend({

        events: {
            "click .hipchat-create-dedicated-room": "createDedicatedRoom",
            "click a.trigger-dialog-select-room": "openSelectRoomDialog",
            "click .trash-dedicated-room": "unassignDedicatedRoom"
        },

        initialize: function (options) {
            this.loggedIn = options.loggedIn;
            this.canCreateRoom = options.canCreateRoom;
            this.canAssignRoom = options.canAssignRoom;
            this.isAdmin = options.isAdmin;
            this.hipChatUserName = options.hipChatUserName;
            this.issueKey = options.issueKey;
        },

        initInviteContributorsView: function(issueKey) {
            var inviteContributorsOptions = {
                issueKey: issueKey
            };
            var inviteModel = new InviteContributors(inviteContributorsOptions);

            var options = _.extend(this.options, {
                model: inviteModel
            });
            this.inviteContributorsView = new InviteContributorsView(options);
        },

        close: function () {
            if (this.inviteContributorsView) {
                this.inviteContributorsView.close();
            }
            if (this.dialogContentReadycallback) {
                $(document).unbind("dialogContentReady", this.dialogContentReadycallback);
            }
            this.unbind();
        },

        createDedicatedRoom: function (e) {
            var $createButton = this.$(".hipchat-create-dedicated-room");

            var self = this;
            if (self.loggedIn && self.canCreateRoom) {
                $createButton.attr("disabled", "disabled");
                self.createOrAssignDedicatedRoom().done(function (dedicatedRoom) {
                    JIRA.trace("hipchat.dedicated.room.created");
                    if (!dedicatedRoom.privateRoom)
                        self.inviteContributorsView.openInviteDialog(dedicatedRoom.roomId);
                }).always(function () {
                    $createButton.removeAttr("disabled");
                });
            } else {
                if (!self.loggedIn || !self.canCreateRoom) {
                    var dialog = AJS.InlineDialog($createButton, "dedicated-room-not-logged-in-dialog", function (content, trigger, showPopup) {

                        var html;
                        if (!self.canCreateRoom && self.isAdmin) {
                            html = JIRA.Templates.HipChat.Project.IssuePanel.cantCreateRoomMessage({});
                        } else {
                            html = JIRA.Templates.HipChat.Project.IssuePanel.notLoggedInMessage({action: "create"});
                        }

                        content.html(html);

                        showPopup();
                    }, self.dialogConfig());

                    dialog.show(e, $createButton);
                }
            }
        },

        createOrAssignDedicatedRoom: function (roomId) {
            var self = this;
            var issueKey = self.issueKey;
            var dfd = $.Deferred();

            var $errorsContainer = self.$("#hipchat-dedicated-room-errors");
            self.clearError($errorsContainer);
            var $spinner = self.$('.hc-button-spinner');
            $spinner.css({display: "inline-block"});
            $spinner.spin();

            self.model.save({issue_key: issueKey, room_id: roomId})
                    .done(function (dedicatedRoom) {
                var html = JIRA.Templates.HipChat.Project.IssuePanel.dedicatedRoomDetails({
                    loggedIn: self.loggedIn,
                    dedicatedRoom: dedicatedRoom,
                    created: true
                });

                self.$(".hipchat-dedicated-room-content")
                        .hide()
                        .html(html)
                        .fadeIn({
                            complete: function () {
                                setTimeout(function () {
                                    self.$(".hipchat-dedicated-room-icon").removeClass("success");
                                }, 1500);
                            }
                        });

                var $panelContainer = $("#hipchat-issue-panel-rooms-container");
                $panelContainer.data("dedicated-room-id", dedicatedRoom.roomId);

                JIRA.trace("hipchat.dedicated.room.created");
                self.initInviteContributorsView(self.issueKey);
                dfd.resolve(dedicatedRoom);
            }).fail(function (err) {
                if (self.loggedIn && (err.status == 403 || err.status == 401)) {
                    var dialog = new AJS.Dialog({
                        width: 500,
                        height: 300,
                        id: "oauth-failure",
                        closeOnOutsideClick: true
                    });
                    dialog.addHeader("Authentication Failure");
                    dialog.addPanel("panel 0", JIRA.Templates.HipChat.Project.IssuePanel.oauthNoLongerValidMessageDedicatedRoom({}));
                    dialog.show();
                } else {
                    self.displayError($errorsContainer,
                            roomId === undefined ?
                                    AJS.I18n.getText("jira.plugins.hipchat.viewissue.panel.dedicated.room.create.error") :
                                    AJS.I18n.getText("jira.plugins.hipchat.viewissue.panel.dedicated.room.assign.error"), err);
                }
                dfd.reject(err);
            }).always(function () {
                $spinner.spinStop();
                $spinner.css({display: "none"});
                var $createButton = $(".hipchat-create-dedicated-room");
                $createButton.removeAttr("disabled");
            });

            return dfd;
        },

        openSelectRoomDialog: function (e) {
            e.preventDefault();
            var target = e.currentTarget;
            if (this.dialog === undefined) {
                this.dialog = new JIRA.FormDialog({
                    id: target.id + "-dialog",
                    ajaxOptions: {
                        url: target.href,
                        data: {
                            decorator: "dialog",
                            inline: "false"
                        }
                    },
                    width: 400
                });
                this.initSelectRoomDialog(this.dialog);
            }

            this.dialog.show();
        },

        initSelectRoomDialog: function (dialog) {
            var self = this;
            self.dialogContentReadycallback = function (e, data) {
                if (data.options.id === "hipchat-select-dedicated-room-dialog") {
                    var roomServicePromise = RoomMappingService.roomServicePromise(true);
                    //var loggedIn = $("#hipchat-issue-panel-rooms-container").data("hipchat-username") !== undefined;
                    if (!self.loggedIn) {
                        return;
                    }
                    var roomSelector = new RoomSelectorView({
                        el: $('#hipchat-dedicated-room-select'),
                        loggedIn: self.loggedIn,
                        maxWidth: "480px"
                    });
                    var roomSelectorWatcher = {
                        roomSelector: roomSelector,
                        clearSelectedRoom: function () {
                            this.roomSelector.clearSelectedRoom();
                            this.disableSubmitButton();
                        },
                        enableSubmitButton: function () {
                            var selectedRoom = this.roomSelector.getSelectedRoom();
                            if (selectedRoom && selectedRoom.id !== "") {
                                $("#hipchat-select-dedicated-room-dialog-submit").removeAttr('aria-disabled').removeAttr('disabled');
                            }
                        },
                        disableSubmitButton: function () {
                            $('#hipchat-select-dedicated-room-dialog-submit').attr({
                                'aria-disabled': 'true',
                                'disabled': ''
                            });
                        },
                        roomSelected: function () {
                            return $('#hipchat-select-dedicated-room-dialog-submit').attr("disabled") === undefined;
                        },
                        init: function () {
                            this.roomSelector.on("change", this.enableSubmitButton, this);
                            this.roomSelector.on("clear", this.disableSubmitButton, this);
                        }
                    };
                    roomSelectorWatcher.init();

                    var submitSelectRoomDialog = function () {
                        dialog.showFooterLoadingIndicator();

                        var dfd = $.Deferred();
                        var selectedRoom = roomSelector.getSelectedRoom();
                        if (!selectedRoom.existing) {
                            // create room first
                            var roomServicePromise = RoomMappingService.roomServicePromise(true);
                            roomServicePromise.done(function (roomService) {
                                dfd = roomService.createRoom(selectedRoom.roomName);
                                dfd.fail(function(createRoomError) {
                                   dfd.reject(createRoomError);
                                });
                            }).fail(function (err) {
                                dfd.reject(err);
                            });
                        } else {
                            dfd.resolve(selectedRoom);
                        }

                        dfd.done(function (room) {
                            $('#hipchat-select-dedicated-room-dialog-close').click();

                            // The object returned by roomSelector has id, while the one returned by createRoom has roomId.
                            var roomId = room.id || room.roomId;
                            self.createOrAssignDedicatedRoom(roomId).done(function (dedicatedRoom) {
                                JIRA.trace("hipchat.dedicated.room.selected");
                                if (!dedicatedRoom.privateRoom)
                                    self.inviteContributorsView.openInviteDialog(dedicatedRoom.roomId);
                            });
                        }).fail(function (err) {

                            var $errorContainer = $("#hipchat-select-dedicated-room-dialog").find(".dialog-errors");
                            var text = AJS.I18n.getText("jira.plugins.hipchat.viewissue.panel.dedicated.room.assign.error");
                            self.displayError($errorContainer, text, err);

                        }).always(function () {
                            dialog.hideFooterLoadingIndicator();
                        });
                    };

                    var $form = $("#hipchat-select-dedicated-room-dialog").find("form");
                    $form.on("before-submit", function (e) {
                        e.preventDefault();

                        if (roomSelectorWatcher.roomSelected()) {
                            submitSelectRoomDialog();
                        }
                    });

                    roomServicePromise.done(function (roomService) {
                        roomSelectorWatcher.disableSubmitButton();
                    }).fail(function () {
                        roomSelectorWatcher.disableSubmitButton();
                    });

                    $('#hipchat-select-dedicated-room-dialog-submit').on("click", submitSelectRoomDialog);
                }
            };
            $(document).bind("dialogContentReady", {}, self.dialogContentReadycallback);
        },

        clearError: function ($errorsContainer) {
            $errorsContainer.hide();
            $errorsContainer.empty();
        },

        displayError: function ($errorsContainer, title, err) {
            $errorsContainer.show();
            $errorsContainer.empty();

            var msg;
            if (err.status === 503) {
                msg = AJS.I18n.getText("jira.plugins.hipchat.viewissue.panel.unable.to.connect");
            } else {
                msg = title;
            }

            $errorsContainer.append(msg);
        },

        unassignDedicatedRoom: function (e) {
            e.preventDefault();

            var self = this;

            if (!self.loggedIn) {

                var $button = self.$(".trash-dedicated-room");
                var dialog = AJS.InlineDialog($button, "dedicated-room-not-logged-in-dialog", function (content, trigger, showPopup) {

                    var html = JIRA.Templates.HipChat.Project.IssuePanel.notLoggedInMessage({ action: "delete"});
                    content.html(html);
                    showPopup();
                }, self.dialogConfig());

                dialog.show(e, $button);

            } else {

                var dfd = $.Deferred();

                var $errorsContainer = self.$("#hipchat-dedicated-room-errors");
                self.clearError($errorsContainer);
                self.showSpinner();

                self.$(".trash-dedicated-room").hide();
                self.model.destroy().always(function () {
                    self.hideSpinner();
                    var $unlinkButton = self.$(".trash-dedicated-room");
                    $unlinkButton.data("hide-for-progress", false);
                }).done(function () {
                    self.model = new DedicatedRoom({});

                    var html = JIRA.Templates.HipChat.Project.IssuePanel.dedicatedRoomForm({
                        canAssignRoom: self.canAssignRoom,
                        canCreateRoom: self.canCreateRoom,
                        isAdmin: self.isAdmin,
                        issueKey: self.issueKey
                    });

                    self.$(".hipchat-dedicated-room-content").html(html);
                    AJS.messages.success('#dedicated-room-message-bar', {
                        title: AJS.I18n.getText('admin.common.words.success'),
                        body: AJS.I18n.getText("jira.plugins.hipchat.viewissue.panel.dedicated.room.delete.success.message"),
                        fadeout: true
                    });

                    JIRA.trace("hipchat.dedicated.room.deleted");
                    dfd.resolve();
                }).fail(function (err) {
                    self.displayError($errorsContainer, AJS.I18n.getText("jira.plugins.hipchat.viewissue.panel.dedicated.room.unassign.error"), err);
                    dfd.reject(err);
                });

                return dfd;
            }

        },

        dialogConfig: function() {
            return {
                noBind: true,
                hideDelay: null
            };
        },

        showSpinner: function() {
            var $spinner = this.$('.hc-button-spinner');
            if ($spinner) {
                $spinner.css({display: "inline-block"});
                $spinner.spin();
            }
        },

        hideSpinner: function() {
            var $spinner = this.$('.hc-button-spinner');
            if ($spinner) {
                $spinner.spinStop();
                $spinner.css({display: "none"});
            }
        }

        });

    return DedicatedRoomView;
});
