UPM.define('ManageAddonModel',
    [
        'jquery',
        'underscore',
        'AddonActions',
        'EnableDisableAddonMixin',
        'InstalledAddonModel',
        'UpmEnvironment',
        'UpmRequireRestart',
        'UpmStrings',
        'UpmXsrfTokenState'
    ],
    function($,
             _,
             AddonActions,
             EnableDisableAddonMixin,
             InstalledAddonModel,
             UpmEnvironment,
             UpmRequireRestart,
             UpmStrings,
             UpmXsrfTokenState) {

    "use strict";

    /**
     * Subclass of AddonModel that adds properties and behaviors that are only relevant on the
     * Manage page.
     */
    return InstalledAddonModel.extend({

        mixins: [ EnableDisableAddonMixin ],
        
        sendVendorFeedback: function(action, reason, textReason, userEmail, userFullName) {
            var url = this.getLinks()['vendor-feedback'],
                data = {
                    reasonCode: reason,
                    message: textReason,
                    type: action.key.toLowerCase(),
                    pluginVersion: this.getVersion(),
                    email: userEmail,
                    fullName: userFullName
                };
            if (url) {
                return $.ajax({
                    type: 'POST',
                    url: url,
                    dataType: 'json',
                    contentType: UpmEnvironment.getContentType(),
                    data: JSON.stringify(data)
                });
            }
        },

        startUpdate: function() {
            var binaryUri = this.getLinks().binary,
                descriptorUri = this.getLinks()['update-descriptor'],
                resourceUrl,
                contentType,
                data;

            if (descriptorUri) {
                contentType = UpmEnvironment.getContentType('remote-install');
                data = { "pluginUri": descriptorUri, "pluginName": this.getName(), "pluginKey": this.getKey() };
                resourceUrl = UpmEnvironment.getResourceUrl('remote-install');
            } else {
                contentType = UpmEnvironment.getContentType('install');
                data = { "pluginUri": binaryUri, "pluginName": this.getName(), "pluginVersion":
                    this.getPacResponse().update && this.getPacResponse().update.version };
                resourceUrl = UpmEnvironment.getResourceUrl('install-uri');
            }

            if (binaryUri || descriptorUri) {
                return UpmXsrfTokenState.tryWithToken(function(token) {
                    return $.ajax({
                        type: 'POST',
                        url: resourceUrl + '?token=' + token,
                        dataType: 'json',
                        contentType: contentType,
                        data: JSON.stringify(data)
                    }).promise();
                });
            } else {
                this.triggerMessage({
                    type: 'error',
                    message: UpmStrings['upm.messages.update.error'],
                    className: 'update'
                });
            }
        },

        uninstall: function() {
            var me = this;
            return $.ajax({
                type: 'DELETE',
                url: this.getLinks()['delete'],
                dataType: 'json',
                contentType: UpmEnvironment.getContentType('json'),
            }).done(function(response) {
                var restartState = response && response.restartState;
                if (restartState) {
                    UpmRequireRestart.addChangeRequiringRestart({
                        action: restartState,
                        name: me.getName(),
                        key: me.getKey(),
                        links: { self: response.links['change-requiring-restart'] }
                    });
                    me.setRestartState(restartState);  // causes the view to refresh
                } else {
                    me.trigger('uninstalled');
                }
            }).promise();
        },

        updateAccessToken: function(token, state) {
            var me = this,
                oldToken = this.getAccessToken(),
                rqParams;

            if (token && !state) {
                state = this.isPaidViaAtlassian() ? 'ACTIVE_TRIAL' : 'NONE';
            }

            if (token) {
                rqParams = {
                    type: 'POST',
                    url: UpmEnvironment.getResourceUrl('license-tokens'),
                    contentType: UpmEnvironment.getContentType(),
                    data: JSON.stringify({ pluginKey: this.getKey(), token: token, state: state })
                };
            } else {
                rqParams = {
                    type: 'DELETE',
                    url: oldToken && oldToken.links.self
                };
            }
            return $.ajax(rqParams).done(function(tokenResponse) {
                me.setAccessToken(token ? tokenResponse : null);
                me.trigger('accessTokenChanged', oldToken);
                if (token) {
                    UPM.trace(oldToken ? 'license-token-edited' : 'license-token-generated');
                } else {
                    UPM.trace('license-token-removed');
                }
            }).promise();
        },

        _getActionsOrder: function() {
            // Ultimately it's ManageAddonView and ManageAddonDetailsView that are responsible for the ordering of
            // the action buttons, but they call this helper method to get the defaults.

            // If eval plugin is nearly expired, show Buy Now before doing update check.
            var showBuyNowBeforeUpdateCheck =
                (this.getLicenseDetails() && this.getLicenseDetails().nearlyExpiring);

            return [
                AddonActions.UPGRADE,
                AddonActions.TRY,
                AddonActions.TRIAL_SUBSCRIBE,
                AddonActions.TRIAL_RESUME,
                AddonActions.SUBSCRIBE,
                AddonActions.RENEW,
                AddonActions.RENEW_CONTACT,
                AddonActions.TRIAL_UNSUBSCRIBE,
                AddonActions.UNSUBSCRIBE,
                showBuyNowBeforeUpdateCheck ? AddonActions.BUY : AddonActions.UPDATE,
                showBuyNowBeforeUpdateCheck ? AddonActions.UPDATE : AddonActions.BUY,
                AddonActions.CHECK_LICENSE,
                AddonActions.GET_STARTED,
                AddonActions.REQUEST_UPDATE,
                AddonActions.DOWNLOAD,
                AddonActions.CONFIGURE,
                AddonActions.UNINSTALL,
                AddonActions.ENABLE,
                AddonActions.DISABLE
            ];
        },

        getActionState: function(action) {
            // subscription-related actions are all disabled if a license token is present
            if (action.isSubscriptionRelatedAction() && this.hasLink(action.legacyKey) && this.canHaveAccessToken() && this.getAccessToken()) {
                return {
                    enabled: false,
                    disabledReason: 'license-token'
                };
            }

            switch (action) {
                case AddonActions.BUY:
                    return this.hasLink('new');
                case AddonActions.CHECK_LICENSE:
                    return this.hasLink('check-license');
                case AddonActions.CONFIGURE:
                    if (this.hasLink('configure') && this.getEnabled() && !this.isApplicationPlugin()) {
                        if (this.hasLicense() && !this.getLicenseDetails().valid) {
                            return {
                                enabled: this.hasLicense() && this.getLicenseDetails().valid,
                                disabledReason: 'bad-license-cannot-configure'
                            };
                        } else {
                            return true;
                        }
                    }
                    return false;
                case AddonActions.DISABLE:
                case AddonActions.ENABLE:
                    return this.isEnableOrDisableActionAllowed(action) && !this.isApplicationPlugin();
                case AddonActions.DOWNLOAD:
                    return !this.isAtlassianConnect() && this.hasLink('binary');
                case AddonActions.GET_STARTED:
                    return this.hasLink('post-install') && this.getEnabled();
                case AddonActions.RENEW:
                    return this.hasLink('renew');
                case AddonActions.RENEW_CONTACT:
                    return this.hasLink('renew-requires-contact');
                case AddonActions.REQUEST_UPDATE:
                    return this.hasLink('request-update');
                case AddonActions.SUBSCRIBE:
                    return this.hasLink('subscribe');
                case AddonActions.TRIAL_RESUME:
                    return this.hasLink('trial-resume');
                case AddonActions.TRIAL_SUBSCRIBE:
                    return this.hasLink('trial-subscribe');
                case AddonActions.TRIAL_UNSUBSCRIBE:
                    return this.hasLink('trial-unsubscribe');
                case AddonActions.TRY:
                    return this.hasLink('try');
                case AddonActions.UNINSTALL:
                    if (this.hasLink('delete') && !this.isApplicationPlugin()) {
                        return true;
                    } else if (this.isAtlassianConnect() && this.getLicenseDetails() && this.getLicenseDetails().autoRenewal) {
                        return { enabled: false, disabledReason: 'subscribed-cannot-uninstall' };
                    } else {
                        return false;
                    }
                    break;
                case AddonActions.UNSUBSCRIBE:
                    return this.hasLink('unsubscribe');
                case AddonActions.UPDATE:
                    if (this.isUpdatable()) {
                        if (UpmEnvironment.isSafeMode()) {
                            return {
                                enabled: false,
                                disabledReason: 'safe-mode'
                            };
                        } else if (this.getPacResponse() && this.getPacResponse().update &&
                                   !this.getPacResponse().update.licenseCompatible) {
                            return {
                                enabled: false,
                                disabledReason: 'license-incompatible'
                            };
                        } else {
                            return true;
                        }
                    } else {
                        return false;
                    }
                    break;
                case AddonActions.UPGRADE:
                    return this.hasLink('upgrade');
            }
            return false;
        },

        // Ensure that pricing (if any) is always loaded when we expand the details.
        loadDetails: function() {
            var me = this;
            return InstalledAddonModel.prototype.loadDetails.apply(this).then(function() {
                return me.loadPricingModel();
            })
        }
    });
});
