UPM.define('CommonInstallAndLicensingFlows',
    [
        'jquery',
        'underscore',
        'AddonActions',
        'DownloadDialog',
        'EvalEulaDialog',
        'EvalRedirectConfirmDialogTemplate',
        'InstallOrLicenseResultDialog',
        'LicenseContactRequiredDialog',
        'NonDataCenterInstallConfirmDialog',
        'OnDemandInstallOrLicenseConfirmDialog',
        'RefreshAfterInstallDialog',
        'RoleBasedPricingDialog',
        'UpdateRequestDialog',
        'UpmAjax',
        'UpmAtlassianId',
        'UpmDialog',
        'UpmEnvironment',
        'UpmFormats',
        'UpmHamlet',
        'UpmLongRunningTasks',
        'UpmStrings'
    ],
    function($,
             _,
             AddonActions,
             DownloadDialog,
             EvalEulaDialog,
             EvalRedirectConfirmDialogTemplate,
             InstallOrLicenseResultDialog,
             LicenseContactRequiredDialog,
             NonDataCenterInstallConfirmDialog,
             OnDemandInstallOrLicenseConfirmDialog,
             RefreshAfterInstallDialog,
             RoleBasedPricingDialog,
             UpdateRequestDialog,
             UpmAjax,
             UpmAtlassianId,
             UpmDialog,
             UpmEnvironment,
             UpmFormats,
             UpmHamlet,
             UpmLongRunningTasks,
             UpmStrings) {

    /**
     * Front-end logic that is shared between the Manage and Find New pages, for actions that involve
     * installation and/or licensing of add-ons.
     */
    var CommonInstallAndLicensingFlows = {

        /**
         * Displays any confirmation dialogs that are necessary prior to executing the specified
         * install/licensing action for the add-on.  This may include an EvalEulaDialog if the user needs
         * to accept a EULA and/or the Atlassian Customer Agreement, and an OnDemandInstallOrLicenseConfirmDialog
         * for various conditions in OnDemand.
         *
         * @method confirmInstallOrLicenseAction
         * @param {AddonModel} addonModel
         * @param {AddonActions} action  optional, if omitted is assumed to be TRY
         * @return {Promise}  a Promise which will be resolved if the user accepts the dialog(s), or if no
         *   action is necessary (in which case no dialog will appear)
         */
        confirmInstallOrLicenseAction: function(addonModel, action) {
            // Note that we need to make sure we've loaded the detail properties, because some things like
            // Connect scopes aren't included in the summary representation
            return addonModel.loadDetails().then(function() {
                if (!addonModel.isInstalled()) {
                    return _doAllNecessaryConfirmationsBeforeInstalling(addonModel, action);
                } else {
                    return _doAllNecessaryConfirmationsBeforeLicensing(addonModel, action);
                }
            });
        },

        /**
         * Opens the dialog for downloading a non-deployable add-on artifact.
         *
         * @method openDownloadDialog
         * @param {AddonModel} addonModel
         */
        openDownloadDialog: function(addonModel) {
            new DownloadDialog({ model: addonModel });
        },

        /**
         * Opens the dialog that tells the user how to contact the vendor to renew a license.
         *
         * @method openRenewContactDialog
         * @param {AddonModel} addonModel
         */
        openRenewContactDialog: function(addonModel) {
            addonModel.loadLicenseDetails().done(function() {
                new LicenseContactRequiredDialog({ model: addonModel });
            });
        },

        /**
         * Opens the role-based pricing dialog, allowing the user to select a pricing tier and
         * then proceed with an action.
         *
         * @method openRoleBasedPricingDialog
         * @param {AddonModel} addonModel
         * @param {String} action  one of the constants from AddonActions
         * @param {JQuery} $loadingEl  container whose "loading" element should be shown while preparing the dialog
         * @return {Promise}  a Promise which will be resolved if the user confirms the action; see
         *   RoleBasedPricingDialog
         */
        openRoleBasedPricingDialog: function(addonModel, action, $loadingEl) {
            var $spinner = $loadingEl && $loadingEl.find('.loading');
            if ($spinner) {
                $spinner.removeClass('invisible');
            }
            return addonModel.loadDetails().
                // need to make sure we also have the license details, if any, so upgrades will work correctly
                then(function() { return addonModel.loadLicenseDetails(); }).
                then(function() {
                    return new RoleBasedPricingDialog({ model: addonModel, action: action })
                        .getResult();
                }).
                always(function() {
                    if ($spinner) {
                        $spinner.addClass('invisible');
                    }
                });
        },

        /**
         * Opens the "request update from vendor" dialog.  If the user submits it, we send the request to
         * MPAC via UPM, and then refresh the add-on state to show that an update was already requested.
         *
         * @method openUpdateRequestDialog
         * @param {AddonModel} addonModel
         */
        openUpdateRequestDialog: function(addonModel) {
            var dataCenterIncompatible = UpmEnvironment.isDataCenter() && !addonModel.getDataCenterCompatible();
            new UpdateRequestDialog({ model: addonModel, dataCenterIncompatible: dataCenterIncompatible })
                .getResult().done(function(result) {
                    _submitUpdateRequest(addonModel, result.message, result.shareDetails);
                });
        },

        /**
         * Shows the appropriate status dialog or other UI elements after an add-on has been installed/updated.
         *
         * @method showInstallOrUpdateCompletion
         * @param {AddonModel} addonModel  the model that the user has interacted with
         * @param {InstalledAddonModel} installedAddonModel  the model representing the *installed* add-on properties;
         *   this may not be the same as addonModel if we are not on the Manage page
         * @param {Object} installResponse  the final task status response from the install/update task
         * @param {boolean} isUpdate  True if this is an update, false for a new install
         * @param {Object} previousAddonProperties  optional - properties of the add-on prior to the update
         */
        showInstallOrUpdateCompletion: function(addonModel, installedAddonModel, installResponse, isUpdate, previousAddonProperties) {
            var updateForDisabledPlugin = isUpdate && previousAddonProperties && !previousAddonProperties.enabled &&
                    !installedAddonModel.getEnabled();

            if (installResponse && installResponse.status && installResponse.status.requiresRefresh) {
                new RefreshAfterInstallDialog();
                UPM.trace('requires-refresh-after-install');
                return;
            }

            // Note, the first two if blocks below are for conditions that we display as errors even though
            // the API has returned a success result.  Currently we're still using inline banners for these.
            if (installedAddonModel.getEnabledByDefault() && !installedAddonModel.getEnabled() && (!isUpdate || !updateForDisabledPlugin)) {
                addonModel.triggerMessage({
                    type: 'error',
                    message: UpmStrings['upm.messages.install.cannot.enable'],
                    className: isUpdate ? 'update' : 'install'
                });
            } else if (installedAddonModel.getUnloadable()) {
                addonModel.triggerMessage({
                    type: 'error',
                    message: UpmStrings['upm.messages.install.unloadable'],
                    className: isUpdate ? 'update' : 'install'
                });
            } else {
                addonModel.focus().done(function() {
                    _showInstallSuccessDialog(addonModel, installedAddonModel, null, isUpdate, previousAddonProperties);
                });
            }
        },

        /**
         * Attempts to start an add-on evaluation via a direct Hamlet request; redirects to MAC if that was not possible.
         * This is only for Server instances, not Cloud licensing.
         *
         * @method startEvaluation
         * @param {AddonModel} addonModel
         */
        startEvaluation: function(addonModel) {
            var apiUnavailableFallback = function() {
                CommonInstallAndLicensingFlows.submitMarketplaceActionToMAC(addonModel, AddonActions.TRY);
                return $.Deferred().promise();  // will not be resolved
            };

            var createEvaluationWithAtlIdToken = function(token) {
                if (token) {
                    UpmLongRunningTasks.startProgress('eval', {}, 'requesting');
                    return UpmHamlet.createEvaluationLicenseDirectly(addonModel.getKey(), token)
                        .then(
                            function(data) {
                                UpmLongRunningTasks.startProgress('eval', {}, 'storing');
                                addonModel.updateLicense(data.licenseKey)
                                    .always(UpmLongRunningTasks.stopProgress)
                                    .then(function() {
                                        return addonModel.loadInstalledAddonModel();
                                    })
                                    .done(function(installedAddonModel) {
                                        _showInstallSuccessDialog(addonModel, installedAddonModel, installedAddonModel.getLicenseDetails());
                                    });
                                return true;
                            },
                            function(xhr) {
                                UpmLongRunningTasks.stopProgress();
                                if (xhr.status === 403) {
                                    return $.Deferred().resolve();  // reattempt login - see UpmAtlassianId.tryWithAtlassianIdToken
                                } else {
                                    return apiUnavailableFallback();
                                }
                            });
                } else {
                    return new UpmDialog({ template: EvalRedirectConfirmDialogTemplate })
                        .getResult().then(apiUnavailableFallback);
                }
            };

            UpmAtlassianId.tryWithAtlassianIdToken(createEvaluationWithAtlIdToken);
        },

        /**
         * Starts a MAC purchase/licensing flow via an invisible form submission.  This is only for Server
         * instances, not Cloud licensing.
         *
         * @method submitMarketplaceActionToMAC
         * @param {AddonModel} addonModel
         * @param {String} action  one of the constants from AddonActions
         * @param {Number} users  optional user count
         * @param {Boolean} openInNewWindow
         */
        submitMarketplaceActionToMAC: function(addonModel, action, users, openInNewWindow) {
            // build this hidden element using the JQuery API, rather than a template, because nothing in it
            // is presentation-related - it is all basically just programmatic construction of an HTTP request
            var $form = $('<form method="post" class="hidden"></form>'),
                licenseDetails = addonModel.getLicenseDetails(),
                hostLicense = UpmEnvironment.getHostLicense(),
                orgName = (licenseDetails && licenseDetails.organizationName) ? licenseDetails.organizationName :
                    (hostLicense && hostLicense.organizationName),
                realUsers = (users && users !== 0) ? users :
                    ((hostLicense && hostLicense.maximumNumberOfUsers) ? hostLicense.maximumNumberOfUsers : -1),
                values;

            $form.attr('action', addonModel.getLinks()[action.legacyKey]);
            if (openInNewWindow) {
                $form.attr('target', '_blank');
            }

            values = _.extend(
                {
                    callback: addonModel.getLinks()['license-callback'],
                    licensefieldname: 'license',
                    users: realUsers
                },
                (licenseDetails && licenseDetails.supportEntitlementNumber) ?
                    { addon_sen: licenseDetails.supportEntitlementNumber } : {},
                orgName ? { organisation_name: orgName } : {},
                (licenseDetails && licenseDetails.contactEmail) ? { owner: licenseDetails.contactEmail } : {},
                hostLicense.supportEntitlementNumber ? { parent_sen: hostLicense.supportEntitlementNumber } : {}
            );

            $form.append(_.map(values, function(value, key) {
                return $('<input type="hidden">').attr('name', key).val(value);
            }));
            
            // TODO: send analytics event??

            $form.appendTo(document.body);
            $form.submit();
            $form.remove();
        },

        /**
         * Attempts to activate/deactivate/renew an OnDemand add-on via a direct Hamlet request.  Redirects to
         * MAC if that was not possible.  Does *not* display any success messages, since we do those differently
         * on the Manage page vs. the Find New page.
         *
         * @method updateOnDemandSubscription
         * @param {AddonModel} addonModel
         * @param {AddonActions}  action  AddonActions.SUBSCRIBE, TRIAL_SUBSCRIBE, TRIAL_RESUME, or UNSUBSCRIBE
         * @return {Promise}  a Promise that will be resolved when the request finishes or rejected if it fails
         *   (if we end up redirecting to MAC instead, the Promise is irrelevant since we will not return)
         */
        updateOnDemandSubscription: function(addonModel, action) {
            var taskType = action.isSubscriptionActivation() ? 'ondemand.addon.activate' : 'ondemand.addon.deactivate';

            UpmLongRunningTasks.startProgress(taskType, { name: addonModel.getName() });

            return addonModel.startOnDemandLicenseUpdate(action)
                .then(function(data, status, request) {
                    return _onDemandSubscriptionResponseHandler(addonModel, action, request);
                }, function(request) {
                    // UPM-5038 - Unfortunately, we have to special case this to handle when the Billing API returns
                    // an 200 with an empty JSON body. Starting with JQuery 1.9, an empty string returned for
                    // a JSON data will be considered malformed. See
                    // http://jquery.com/upgrade-guide/1.9/#jquery-ajax-returning-a-json-result-of-an-empty-string
                    // To handle this, we need to make sure that if a 20x is returned, execute the correct
                    // success handle.
                    return _onDemandSubscriptionResponseHandler(addonModel, action, request);
                })
                .promise();
        }
    };

    // Functions below are used internally by this module

    function _confirmEvalEulaIfNecessary(addonModel, action) {
        if ((action && action !== AddonActions.TRY) || addonModel.isAtlassianConnect()) {
            return $.Deferred().resolve();
        } else {
            return UpmEnvironment.checkMarketplaceEulaAccepted().then(function(acceptedMpacEula) {
                if (acceptedMpacEula && !addonModel.getEulaUrl() && !addonModel.isByAtlassian()) {
                    return $.Deferred().resolve().promise();
                } else {
                    var dialog = new EvalEulaDialog({
                        model: addonModel,
                        needMarketplaceEula: !acceptedMpacEula,
                        needAtlassianCustomerAgreement: addonModel.isByAtlassian()
                    });
                    UPM.trace("eval-eula-dialog");
                    return dialog.getResult();
                }
            });
        }
    }

    function _confirmOnDemandInstallOrLicense(addonModel) {
        return addonModel.loadPricingModel().then(function() {
                return (new OnDemandInstallOrLicenseConfirmDialog({ model: addonModel })).getResult();
            });
    }

    function _doAllNecessaryConfirmationsBeforeInstalling(addonModel, action) {
        function confirmNonDataCenter() {
            if (!addonModel.getDataCenterCompatible() && UpmEnvironment.isDataCenter()) {
                return (new NonDataCenterInstallConfirmDialog({ model: addonModel })).getResult();
            } else {
                return $.Deferred().resolve();
            }
        }

        function confirmRemoteInstall() {
            if (addonModel.isAtlassianConnect()) {
                return _confirmOnDemandInstallOrLicense(addonModel);
            } else {
                return $.Deferred().resolve();
            }
        }

        // chain together all these Promises, returning one which will pass only if they all pass
        return confirmNonDataCenter().then(confirmRemoteInstall)
            .then(function() {
                return _confirmEvalEulaIfNecessary(addonModel, action);
            });
    }

    function _doAllNecessaryConfirmationsBeforeLicensing(addonModel, action) {
        function confirmOnDemandInstall() {
            // In OnDemand, if we're licensing an already-installed add-on, we normally don't need to
            // show a warning - except if we're in a free application user tier and the add-on isn't free (UPM-4999).
            if (UpmEnvironment.isOnDemand() &&
                UpmEnvironment.isPlatformFreeTier() &&
                addonModel.isPaidViaAtlassian() &&
                !addonModel.getFreeTier()) {
                return _confirmOnDemandInstallOrLicense(addonModel);
            } else {
                return $.Deferred().resolve();
            }
        }

        return confirmOnDemandInstall()
            .then(function() {
                return _confirmEvalEulaIfNecessary(addonModel, action);
            });
    }

    function _onDemandSubscriptionResponseHandler(addonModel, action, request) {
        return $.Deferred().resolve(request.status).then(
            function(status) {
                if (status >= 200 && status < 300) {
                    return _waitForEmbeddedLicenseChange(addonModel.getLinks()['embedded-license-task'])
                        .then(function() {
                            return addonModel.refresh();
                        });
                } else {
                    return _onDemandSubscriptionUpdateFailed(addonModel, action, request);
                }
            })
            .always(UpmLongRunningTasks.stopProgress);
    }

    function _onDemandSubscriptionUpdateFailed(addonModel, action, request) {
        var status = request.status;
        if (status === 404 || status === 501) {
            CommonInstallAndLicensingFlows.submitMarketplaceActionToMAC(addonModel, action);
            return $.Deferred(); // leave the request hanging instead of returning an error - we'll be going to another page
        } else if (status === 400 && action.isSubscriptionActivation()) {
            // 400 is returned when add-on does not exist in production (private/beta testing)
            addonModel.triggerMessage({
                type: 'error',
                message: UpmFormats.format(UpmStrings['upm.messages.error.addon.activate.failed'], addonModel.getName())
            });
        } else if (status === 500) {
            addonModel.triggerMessage({
                type: 'error',
                message: UpmFormats.format(UpmStrings[action.isSubscriptionActivation() ?
                    'upm.messages.error.addon.activate.unexpected.failure' :
                    'upm.messages.error.addon.deactivate.unexpected.failure'], addonModel.getName())
            });
        } else {
            addonModel.trigger('ajaxError', _parseHamletErrorInfo(request));
        }
        return request;
    }

    function _parseHamletErrorInfo(request) {
        try {
            response = JSON.parse(request.responseText);
            if (response) {
                var m = (response.errorKey && UpmStrings[response.errorKey]) || response.error;
                if (m) {
                    return { errorMessage: m };
                }
            }
        } catch (e) {
            AJS.log('Failed to parse hamlet response text: ' + e);
        }
        return { subCode: 'ajaxServerError' };
    }

    function _showInstallSuccessDialog(displayedModel, installedAddonModel, newLicense, isUpdate, previousAddonProperties) {
        var dialog,
            nextAction,
            wasEnabled = previousAddonProperties && previousAddonProperties.enabled,
            updateForDisabledPlugin = isUpdate && !wasEnabled && !installedAddonModel.getEnabled(),
            postInstallUrl = installedAddonModel.getLinks()[isUpdate ? 'post-update' : 'post-install'];

        if (!installedAddonModel.getIncompatible() && installedAddonModel.getEnabledByDefault() && !updateForDisabledPlugin) {
            if (installedAddonModel.isLicenseUpdatable() && !installedAddonModel.getLicenseDetails && !newLicense) {
                nextAction = (previousAddonProperties && previousAddonProperties.updatableToPaid) ?
                    'new-license' : 'manage-license';
            } else if (postInstallUrl) {
                nextAction = 'get-started';
            }
        }

        dialog = new InstallOrLicenseResultDialog({
            model: installedAddonModel,
            newLicense: newLicense,
            isInstall: !isUpdate && !newLicense,
            isUpdate: isUpdate,
            nextAction: nextAction,
            updateForDisabledPlugin: updateForDisabledPlugin
        });

        UPM.trace('install-result-dialog');

        dialog.getResult().done(function() {
            switch (nextAction) {
                // "get started" action is handled by InstallOrLicenseResultDialog itself, since it just opens a link

                case 'new-license':
                    if (UpmEnvironment.isOnDemand()) {
                        CommonInstallAndLicensingFlows.updateOnDemandSubscription(displayedModel, AddonActions.TRIAL_SUBSCRIBE);
                    } else {
                        CommonInstallAndLicensingFlows.submitMarketplaceActionToMAC(displayedModel, AddonActions.TRY);
                    }
                    break;

                case 'manage-license':
                    if (installedAddonModel.getLinks()['plugin-details']) {
                        window.location.href = installedAddonModel.getLinks()['plugin-details'];
                    } else {
                        displayedModel.focus();
                    }
                    break;
            }
        });
    }

    function _submitUpdateRequest(addonModel, message, shareDetails) {
        addonModel.submitUpdateRequest(message, shareDetails)
            .done(function() {
                UpmEnvironment.refreshNotifications();
                addonModel.refresh().done(function() {
                    addonModel.triggerMessage({
                        type: 'info',
                        message: UpmStrings['upm.messages.plugin.update.request.success']
                    });
                    UPM.trace('update-request-complete');
                });
            })
            .fail(UpmAjax.signalAjaxError);
    }

    function _waitForEmbeddedLicenseChange(url) {
        return $.ajax({
            type: 'POST',
            url: url,
            dataType: 'json',
            contentType: UpmEnvironment.getContentType('embedded-license-task')
        }).then(
            function(data, status, request) {
                return UpmLongRunningTasks.pollForCompletion(request.getResponseHeader('Location'), data.pingAfter)
                    .always(UpmLongRunningTasks.stopProgress);
            },
            UpmAjax.signalAjaxError
        );
    }

    return CommonInstallAndLicensingFlows;
});
